/**
 * Action that should handle loop road building to remove/prevent vehicles jam.
 */
class BuildLoopRoadAction extends TriggerAction
{
/* public */
	/**
	 * Creates BuildLoopRoadAction instance.
	 * @param jam_center_tile Center(or so) tile of vehicles jam.
	 * @param r Radius of danger road section -
	 *  - distance from center to the end of jam.
	 */
	constructor(jam_center_tile, r)
	{
		::TriggerAction.constructor();

		this.pairs = this.PlanLoopRoad(jam_center_tile, r);
		this.safe_run_date = AIDate.GetCurrentDate();
		/* When pairs.len() == 0 we should stop fast, else need to wait */
		if (this.pairs.len() != 0) this.safe_run_date += GameTime.MONTH;
	}

	function GetName()
	{
		return "Building loop road, action id is #" + this.GetID();
	}

	function Try()
	{
		/* Nothing to connect => stop */
		if (this.pairs.len() == 0) {
			this.SelfRemove();
			return 0;
		}

		/*
		 * Since vehicles is jamming(i.e. there are a lot of them)
		 * we shold obtain money fast enough
		 */
		AICompany.SetLoanAmount(AICompany.GetMaxLoanAmount());
		if (AICompany.GetBankBalance(AICompany.COMPANY_SELF) < 20000) return -2;

		/* Try to build the loop road */
		local old_type = RoadUtils.SetRoadType(AIRoad.ROADTYPE_ROAD);

		local result = -1;
		foreach (pair_id, pair in this.pairs) {
			if (pair_id >= 2) break;

			local x0 = AIMap.GetTileX(pair[0]);
			local y0 = AIMap.GetTileY(pair[0]);
			local x1 = AIMap.GetTileX(pair[1]);
			local y1 = AIMap.GetTileY(pair[1]);
			CodeUtils.Log("Building loop road beetwen [ " + x0 + ", " + y0 + "] and [" + x1 + ", " + y1 + "]", 2);

			local pf = AdmiralAIv22PathfinderAdapter();
			pf.ConnectTilesWithNewRoad(pair[0], pair[1]);
		}

		RoadUtils.SetRoadType(old_type);

		/* Stop self if succeed (or have failed too much attepts) */
		this.build_attempts_rest = (result == 0) ? 0 : build_attempts_rest - 1;
		if (this.build_attempts_rest <= 0) this.SelfRemove();

		return result;
	}

/* protected */
	function IsReady()
	{
		return AIDate.GetCurrentDate() > this.safe_run_date;
	}

/* private */
	/** Defines how many times we should try to build a road */
	build_attempts_rest = 1;

	/**
	 * Date when we can try to build road
	 * Used in hope that vehicles leave jam area and we'll be free to build.
	 */
	safe_run_date = null;

	/**
	 * Array of pairs of tiles(array too, with just len 2) we should connect,
	 *  to successfully 'build loop road'.
	 * By 'connection' here it is assumed new road,
	 *  that must not include already existing road between pair tiles.
	 */
	pairs = null;

	/**
	 * Finds one of the given road's corners.
	 * @param t The road tile.
	 * @param initial_offset The road direction.
	 * @param corner_number One mean that we need the first meet corner, two - second, e.t.c.
	 * @param steps Max search distance.
	 * @return Corner tile ID, or -1 if it is not found.
	 * @note Will also return not corner tiles which are located in 4x6 and 6x4
	 *  grids cells corners.
	 */
	function FindRoadCorner(t, initial_offset, corner_number, steps)
	{
		local ti = AIMap.GetTileIndex;
		local offsets = [ti(-1, 0), ti(0, 1), ti(1, 0), ti(0, -1)];
		local x_result = -1;
		for (local i = 0; i < steps; i++) {
			local x = t - initial_offset;
			if (!AIRoad.AreRoadTilesConnected(t, x)) {
				foreach (dummy_id, offset in offsets) {
					if (offset == initial_offset || offset == -initial_offset) continue;
					if (!AIRoad.AreRoadTilesConnected(t, t - offset)) continue;
					if (corner_number <= 1) return t;
					return this.FindRoadCorner(t - offset, offset, corner_number - 1, steps - i);
				}
				return -1;
			}

			t = x;
			if (AIRoad.IsRoadStationTile(x)) {
				if (!AIRoad.IsDriveThroughRoadStationTile(x)) return -1;
			}
			if (AIBridge.IsBridgeTile(x)) t = AIBridge.GetOtherBridgeEnd(x);
			if (AITunnel.IsTunnelTile(x)) t = AITunnel.GetOtherTunnelEnd(x);
			if (AIRoad.GetNeighbourRoadCount(t) == 2) continue;

			foreach (dummy_id, offset in offsets) {
				if (offset == initial_offset || offset == -initial_offset) continue;
				if (!AIRoad.AreRoadTilesConnected(t, t - offset)) continue;
				if (corner_number <= 1) return t;
				return this.FindRoadCorner(t - offset, offset, corner_number - 1, steps - i);
			}
			if (AIMap.GetTileX(t) % 6 == 0 && AIMap.GetTileY(t) % 4 == 0) x_result = t;
			if (AIMap.GetTileX(t) % 4 == 0 && AIMap.GetTileY(t) % 6 == 0) x_result = t;
		}
		return AIRoad.IsRoadTile(x_result) ? x_result : -1;
	}

	/**
	 * Determine the pairs of road tiles, that must be connected
	 *  with dublicate road.
	 * @param jam_center_tile Road tile, center of the jam.
	 * @param r Radius of the jam.
	 */
	function PlanLoopRoad(jam_center_tile, r)
	{
		local old_type = RoadUtils.SetRoadType(AIRoad.ROADTYPE_ROAD);
		if (AIRoad.GetNeighbourRoadCount(jam_center_tile) != 2) {
			RoadUtils.SetRoadType(old_type);
			return [];
		}

		local tiles = [];
		local result = [];
		local ti = AIMap.GetTileIndex;
		local offsets = [ti(-1, 0), ti(1, 0), ti(0, 1), ti(0, -1)];
		for (local i = 0; i < 4; i++) {
			local offset = offsets[i];
			local t = jam_center_tile - offset;
			if (!AIRoad.AreRoadTilesConnected(t, jam_center_tile)) 	continue;
			if (AIRoad.IsRoadStationTile(t)) {
				if (!AIRoad.IsDriveThroughRoadStationTile(t)) continue;
			}
			if (AIBridge.IsBridgeTile(t) || AITunnel.IsTunnelTile(t)) continue;
			if (i % 2 == 0) i++;
			tiles.append([t, offset]);
		}

		local t1 = -1, t2 = -1;
		if (tiles.len() == 1) {
			t1 = this.FindRoadCorner(tiles[0][0], tiles[0][1], 1, 16);
			t2 = this.FindRoadCorner(tiles[0][0], -tiles[0][1], 2, 16);
			if (t1 != -1 && t2 != -1) {
				local dx = abs(AIMap.GetTileX(t1) - AIMap.GetTileX(t2));
				local dy = abs(AIMap.GetTileY(t1) - AIMap.GetTileY(t2));
				if (dx > 1 && dy > 1) result.append([t1, t2]);
			}

			t1 = this.FindRoadCorner(tiles[0][0], tiles[0][1], 2, 16);
			t2 = this.FindRoadCorner(tiles[0][0], -tiles[0][1], 1, 16);
		}
		if (tiles.len() == 2) {
			t1 = this.FindRoadCorner(tiles[0][0], tiles[0][1], 1, 16);
			t2 = this.FindRoadCorner(tiles[1][0], tiles[1][1], 1, 16);
		}

		if (t1 != -1 && t2 != -1) {
			local dx = abs(AIMap.GetTileX(t1) - AIMap.GetTileX(t2));
			local dy = abs(AIMap.GetTileY(t1) - AIMap.GetTileY(t2));
			if (dx > 1 && dy > 1) result.append([t1, t2]);
		}
		RoadUtils.SetRoadType(old_type);
		return result;
	}
}
