/**
 * Class that scans map for road jams.
 */
class SearchRoadPlugsTask extends Terron_Task
{
/* public */
	/**
	 * Creates SearchRoadPlugsTask object.
	 */
	constructor()
	{
		local t = 7 * GameTime.MONTH / 2;
		::Terron_Task.constructor(AIDate.GetCurrentDate() + t, t);
	}

/* protected */
	function Execute()
	{
		local v_list = AIVehicleList();

		v_list.Valuate(AIVehicle.GetVehicleType);
		v_list.KeepValue(AIVehicle.VT_ROAD);

		/* We'll cut off 100% not jamming road vehicles, keep only slow */
		v_list.Valuate(AIVehicle.GetCurrentSpeed);
		v_list.KeepBelowValue(12);

		v_list.Valuate(AIVehicle.GetState);
		v_list.RemoveValue(AIVehicle.VS_AT_STATION);

		/*
		 * We'll scan for 1D vehicles "clouds"(vehicles coordinate aggregate),
		 *  dense enough to be considered jams.
		 * Since we've filtered and kept only slow vehicles, it will work good.
		 * We'll split 2D one game map into 1D maps by fixing one coordinate.
		 * x_bias will keep x-fixed y-free clouds, and vice versa.
		 */
		local x_bias = {};
		local y_bias = {};

		/*
		 * Good, "cloud" represents image of set of points on 1D axis.
		 * Instead of saving each point coordinates, cloud keeps only 3 numbers:
		 *  base.s - 1D coordinate - center of the vehicles cloud
		 *  base.r - "cloud radius"
		 *  and number of points inside(v_list.len()).
		 *  v_list - list of vehicles(array with id's actually) inside cloud
		 * All(vehicles) from [s - r] till [s + r] consider to be in cloud.
		 * At start we'll create one small cloud for each slow vehicle.
		 */
		// v_list is not needed - just vehicles counter should work,
		//  but v_list is good for testing, i keep it.
		local base = {s = -1, r = 3, v_list = null};

		/*
		 * Now set up initial state by spawning a clouds
		 *  describing each vehicle's location.
		 */
		v_list.Valuate(AIVehicle.GetLocation);
		foreach (v, t in v_list) {
			local x = AIMap.GetTileX(t);
			local y = AIMap.GetTileY(t);

			/* Each vehicle will go into two maps, here into "x-fixed" */
			if (!(x in x_bias)) x_bias[x] <- {};
			if (!(y in x_bias[x])) {
				x_bias[x][y] <- clone base;
				x_bias[x][y].v_list = [];
			}
			x_bias[x][y].s = y;
			x_bias[x][y].v_list.append(v);

			/* Each vehicle will go into two maps, and here into "y-fixed" */
			if (!(y in y_bias)) y_bias[y] <- {};
			if (!(x in y_bias[y])) {
				y_bias[y][x] <- clone base;
				y_bias[y][x].v_list = [];
			}
			y_bias[y][x].s = x;
			y_bias[y][x].v_list.append(v);
		}

		/*
		 * Here we'll merge intersectioned(? intersected) clouds
		 *  and see if big enough will emerge => this is jam.
		 */
		this.MergeClouds(x_bias);
		this.MergeClouds(y_bias);

		/* And here we'll spawn actions to fix jams */
		local corporation = Corporation.Get();
		local i = 1;
		foreach (x, y_line in x_bias) {
			foreach (y, info in y_line) {
				if (info.v_list.len() < 5) continue;
				if (i > 0) {
					local t = AIMap.GetTileIndex(x, info.s);
					corporation.AddAction(BuildLoopRoadAction(t, info.r));
					i--;
				}
				RoadVehicleStuckEvent.Fire(info.v_list[0]);
			}
		}
		i = 1;
		foreach (y, x_line in y_bias) {
			foreach (x, info in x_line) {
				if (info.v_list.len() < 5) continue;
				//CodeUtils.Log("y = " + y + ", x = " + info.s +
				//", n = " + info.v_list.len() + ", d = " + info.r, 2);
				if (i > 0) {
					local t = AIMap.GetTileIndex(info.s, y); 
					corporation.AddAction(BuildLoopRoadAction(t, info.r));
					i--;
				}
				RoadVehicleStuckEvent.Fire(info.v_list[0]);
			}
		}
	}

/* private */
	/**
	 * Unite "clouds" close to each other into bigger "cloud".
	 * @param clouds_table Special table describing clouds,
	 *  clouds_table[key] - map with actual clouds.
	 *  Foreach 'key' we have a set of clouds to merge to each other(but not
	 *   with clouds from other keys).
	 */
	function MergeClouds(clouds_table)
	{
		local min_vehicles_to_jam = 5;
		foreach (dummy_fixed_coordinate, clouds in clouds_table) {
			/*
			 * Not many vehicles at all => don't bother,
			 *  even if vehicles are close to each other, it's not a big problem
			 */
			if (clouds.len() < min_vehicles_to_jam) {
				clouds.clear();
				continue;
			}

			local should_stop = true;
			do {
				should_stop = true;
				foreach (i, left_cloud in clouds) {
					foreach (j, right_cloud in clouds) {
						/*
						 * Avoid double checks,
						 * always try to merge from 'right' to 'left'
						 */
						if (left_cloud.s > right_cloud.s) continue;

						/* If clouds do not intersect(or are same) continue */
						// "intersect" mean: "center of first is inside second"
						local d = right_cloud.s - left_cloud.s;
						if (d == 0 || (d > left_cloud.r && d > right_cloud.r)) {
							continue;
						}

						/* 'left' end of the new(merged) cloud */
						local l = left_cloud.s - left_cloud.r;
						/* 'right' end of the new(merged) cloud */
						local r = right_cloud.s + right_cloud.r;

						local shift = left_cloud.r > right_cloud.r ? 1 : 0;
						left_cloud.r = (shift + r - l) / 2;
						left_cloud.s = (shift + r + l) / 2;
						left_cloud.v_list.extend(right_cloud.v_list);

						/* Remove "right" cloud - it's inside "left" now */
						right_cloud.v_list.clear();
						delete clouds[j];

						should_stop = false;
					}
				}
			} while (!should_stop);

			/*
			 * Consider "jam" as a cloud with "many" vehicles in it.
			 * Delete other.
			 */
			foreach (id, cloud in clouds) {
				if (cloud.v_list.len() < min_vehicles_to_jam) delete clouds[id];
			}
		}
	}
}
