require("pf_data_structures.nut");
require("simple_search_rectangle.nut");
require("simple_road_cost_measurer.nut");
require("simple_pf.nut");
require("simple_path_builder.nut");

/**
 * One of the "SimplePF" build strategies.
 * This one is "default", it's goal is balance between road's cost and length,
 *  though short length is a bit in favore.
 * @note Possible roads between two same tiles can be either
 *  "short" but expensive(tunnels, bridges, slopes),
 *  or "long"(move around hill e.g.) but cheap.
 */
class NewRoad extends BaseRoadCost
{
/* public */
	perfect_path_cost = 40;
	new_road_cost = 240;
	turn_cost = 80;
	slope_penalty = 320;
	perfect_slope_penalty = 60;
	base_bridge_cost = 1200;
	base_tunnel_cost = 280;
	costForCloseWater = 80;
	costForWaterSlope = 1200;
	drive_through_station_cost = 18000;
	h_base = 180;
}

/**
 * One of the "SimplePF" build strategies.
 * This one is to build straight roads.
 */
class ExcellentRoad extends BaseRoadCost
{
/* public */
	perfect_path_cost = 40;
	new_road_cost = 240;
	turn_cost = 80;
	slope_penalty = 280;
	perfect_slope_penalty = 60;
	base_bridge_cost = 500;
	base_tunnel_cost = 90;
	costForCloseWater = 40;
	costForWaterSlope = 600;
	drive_through_station_cost = 18000;
	h_base = 180;

	/*perfect_path_cost = 40;
	new_road_cost = 210;
	turn_cost = 200;
	slope_penalty = 120;
	perfect_slope_penalty = 20;
	base_bridge_cost = 480;
	base_tunnel_cost = 100;
	costForCloseWater = 20;
	costForWaterSlope = 600;
	h_base = 140;*/
}

/**
 * One of the "SimplePF" build strategies.
 * This one forces pf to search and use roads previously built in area(if any).
 */
class ReuseOldRoads extends BaseRoadCost
{
/* public */
	perfect_path_cost = 20;
	new_road_cost = 290;
	turn_cost = 10;
	slope_penalty = 60;
	perfect_slope_penalty = 20;
	base_bridge_cost = 600;
	base_tunnel_cost = 180;
	costForCloseWater = 40;
	costForWaterSlope = 600;
	drive_through_station_cost = 18000;
	h_base = 60;
}

/**
 * One of the "SimplePF" build strategies.
 * This one prohibits new roads construction =>
 *  pf can succeed <=> given source and goal tiles already connected.
 */
class FindOldRoad extends BaseRoadCost
{
	perfect_path_cost = 20;
	new_road_cost = BaseRoadCost.infinite;
	turn_cost = 0;
	slope_penalty = 0;
	base_bridge_cost = BaseRoadCost.infinite;
	base_tunnel_cost = BaseRoadCost.infinite;
	costForCloseWater = 0;
	costForWaterSlope = 0;
	drive_through_station_cost = BaseRoadCost.infinite / 3;
	h_base = 20;
}

/**
 * Adapter for the pathfinder from SimplePF.
 */
class SimplePF_Adapter
{
/* public */
	/**
	 * Creates SimplePF_Adapter object.
	 */
	constructor()
	{
		this.default_cost_table = NewRoad();
	}

	/**
	 * Connects two given tiles.
	 */
	function ConnectTiles(s, e, road_type)
	{
		local old_road_type = RoadUtils.SetRoadType(road_type);
		local path = this.GetPath([s], [e], this.default_cost_table);
		local result = (path == null) ? -1 : SimplePathBuilder.BuildPath(path);
		/*if (result == -1 && AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH) {
			result = -2;
		}*/
		AIRoad.SetCurrentRoadType(old_road_type);
		return result; 
	}

	/**
	 * Connects two given tile sets.
	 */
	function ConnectTileSets(s, e, road_type)
	{
		local old_road_type = RoadUtils.SetRoadType(road_type);
		local path = this.GetPath(s, e, this.default_cost_table);
		local t = AIController.GetTick();
		CodeUtils.Log("Building a road path...", 0);
		local result = (path == null) ? -1 : SimplePathBuilder.BuildPath(path);
		t = AIController.GetTick() - t;
		CodeUtils.Log("...ticks: " + t, 0);
		/*if (result == -1 && AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH) {
			result = -2;
		}*/
		AIRoad.SetCurrentRoadType(old_road_type);
		return result; 
	}

	/**
	 * Get the connection cost for two given tiles.
	 */
	function GetConnectionCost(s, e, road_type)
	{
		local old_road_type = RoadUtils.SetRoadType(road_type);
		local path = this.GetPath([s], [e], this.default_cost_table);

		local result = (path == null) ? -1 :
			SimplePathBuilder.GetPathCost(path);
		AIRoad.SetCurrentRoadType(old_road_type);
		return result; 
	}

	/**
	 * Checks if the two tiles are connected.
	 */
	function AreTilesConnected(s, e, road_type)
	{
		local old_road_type = RoadUtils.SetRoadType(road_type);
		local path = this.GetPath([s], [e], FindOldRoad());
		AIRoad.SetCurrentRoadType(old_road_type);
		return path != null;
	}

	/**
	 * Checks if the two sets of tiles are connected.
	 */
	function AreTileSetsConnected(s_array, e_array, road_type)
	{
		local old_road_type = RoadUtils.SetRoadType(road_type);
		local path = this.GetPath(s_array, e_array, FindOldRoad());
		AIRoad.SetCurrentRoadType(old_road_type);
		return path != null;
	}

	/**
	 * Checks if a route between the neighbourhoods of two points already exists.
	 */
	function FindRouteBetweenRects(s, e, r, road_type)
	{
		local get_road_tiles = function(t, r) {
			local list = AITileList();
			TileUtils.AddSimmetricRectangleSafe(list, t, r, r);
			list.Valuate(AIRoad.IsRoadTile);
			list.RemoveValue(0);
			return list;
		}

		local old_road_type = RoadUtils.SetRoadType(road_type);
		local s_array = [], e_array = [];
		foreach (t, dummy in get_road_tiles(s, r)) {
			s_array.append(t);
		}
		foreach (t, dummy in get_road_tiles(e, r)) {
			e_array.append(t);
		}
		local path = this.GetPath(s_array, e_array, FindOldRoad());
		AIRoad.SetCurrentRoadType(old_road_type);

		return path != null;
	}

/* private */
	/** Cost table to be used by pathfinder */
	default_cost_table = null;

	/** Minor help function */
	function GetPath(s_array, e_array, cost_table)
	{
		if (s_array.len() == 0 || e_array.len() == 0) return null;
		return SimplePF().FindPath(s_array, e_array, cost_table, []);
	}
}

/**
 * Adapter for the pathfinder from SimplePF.
 * Should reuse more previously build roads than the base class.
 */
class SimplePF_RoadsReuseAdapter extends SimplePF_Adapter
{
/* public */
	constructor()
	{
		this.default_cost_table = ReuseOldRoads();
	}
}

/**
 * Adapter for the pathfinder from SimplePF.
 * Should build straight roads.
 */
class SimplePF_ExcellentRoadsAdapter extends SimplePF_Adapter
{
/* public */
	constructor()
	{
		this.default_cost_table = ExcellentRoad();
	}
}
