enum PFNodeState
{
	NS_NOT_CHECKED
	NS_OPENED
	NS_CLOSED
};

enum BuildToken
{
	BT_INVALID
	BT_ROAD
	BT_BRIDGE
	BT_TUNNEL
	BT_NEW_BRIDGE
	BT_NEW_TUNNEL
};

enum RoadDir
{
	NE = 0
	SE = 1
	SW = 2
	NW = 3
};

/**
 * Interface for road cost tables.
 */
class BaseRoadCost
{
/* public */
	/** "Infinite" road length representation. */
	static infinite = 1 << 20;

	/** Cost for perfect road piece. */
	perfect_path_cost = 20;

	/** Cost for new road piece. */
	new_road_cost = 100;

	/** Penalty for each corner. */
	turn_cost = 100;

	/**
	 * Penalty for 'perfect' slope.<p>
	 * Perfect mean best possible slope(no terraforming needed) when road must
	 *  go up or down.
	 */
	perfect_slope_penalty = 20;

	/** Penalty for non-'perfect' slope. */
	slope_penalty = 160;

	/** New bridge tile cost. */
	base_bridge_cost = 600;

	/** New tunnel tile cost. */
	base_tunnel_cost = 140;

	/** Penalty for tiles near water(one or two corners have height 0). */
	costForCloseWater = 40;

	/** Penalty for almost full water(3 corners have height 0) tiles */
	costForWaterSlope = 600;

	/** Heuristic(for one tile)*/
	h_base = 100;
}

/**
 * Class that provides handy interface for low level pf data.
 */
class RoadPFDataStructures extends Terron_Settings
{
/* public */
	static function GetClassName()
	{
		return "RoadPFDataStructures";
	}

	static function Restore(memento)
	{
		return RoadPFDataStructures();
	}

	/**
	 * Create a new pf node from the tile's X and Y coordinate.
	 * @param x Tile X coordinate.
	 * @param y Tile Y coordinate.
	 * @return Node object, incapsulating tile [X,Y],
	 * @note No checks inside, so coordinates must be valid.
	 */
	function GetPrimitiveNodeXY(x, y)
	{
		local new_node = clone this.dummy_simple_pf_node;
		new_node.x = x;
		new_node.y = y;
		new_node.tile = AIMap.GetTileIndex(x, y);
		new_node.build_token = BuildToken.BT_INVALID;
		return new_node;
	}

	/**
	 * Create a new pf node from the tile ID.
	 * @param t Tile ID(must be valid).
	 * @return Node object, incapsulating tile t.
	 */
	function GetPrimitiveNode(t)
	{
		local new_node = clone this.dummy_simple_pf_node;
		new_node.x = AIMap.GetTileX(t);
		new_node.y = AIMap.GetTileY(t);
		new_node.tile = t;
		new_node.build_token = BuildToken.BT_INVALID;
		return new_node;
	}

	/**
	 * Array with 4 diag directions.<p>
	 * ID corresponding to RoadDir enum.
	 */
	directions = null;

	/** Pf node sceleton for a bit faster nodes creation */
	dummy_simple_pf_node = null;

/* protected */
	constructor()
	{
		::Terron_Settings.constructor();

		AISettingsSaveEvent.RemoveListener(this);

		local ti = AIMap.GetTileIndex;
		this.directions = array(4);
		this.directions[RoadDir.NE] = RoadDirection(RoadDir.NE, ti(-1, 0));
		this.directions[RoadDir.SE] = RoadDirection(RoadDir.SE, ti(0, 1));
		this.directions[RoadDir.SW] = RoadDirection(RoadDir.SW, ti(1, 0));
		this.directions[RoadDir.NW] = RoadDirection(RoadDir.NW, ti(0, -1));

		this.dummy_simple_pf_node = {
			x = null,
			y = null,
			tile = null,
			dir = null,
			parent_node = null,
			build_token = null,
			bingo = null,
			distance_to_start = 1000000,
			state = PFNodeState.NS_NOT_CHECKED,
			goal_tile = null,
		}
	}
}

/**
 * Class that represents game "diag-direction" concept.
 */
class RoadPFDataStructures.RoadDirection extends Terron_Object
{
/* public */
	/**
	 * Get ID of the bridge direction.
	 * @param t Bridge tile ID.
	 * @return Bridge direction ID.
	 */
	static function GetBridgeDirectionID(t)
	{
		local end = AIBridge.GetOtherBridgeEnd(t);
		if (!AIMap.IsValidTile(end)) return -1;
		return RoadPFDataStructures.RoadDirection.GetLineDirectionID(t, end);
	}

	/**
	 * Get ID of the tunnel direction.
	 * @param t Tuneel tile ID.
	 * @return Tunnel direction ID.
	 */
	static function GetTunnelDirectionID(t)
	{
		local end = AITunnel.GetOtherTunnelEnd(t);
		if (!AIMap.IsValidTile(end)) return -1;
		return RoadPFDataStructures.RoadDirection.GetLineDirectionID(t, end);
	}

	/** Direction ID. */
	id = null;

	/** Tile ID offset for this direction. */
	offset = null;

	/** Tile X coordinate offset for this direction. */
	x_offset = 0;

	/** Tile Y coordinate offset for this direction. */
	y_offset = 0;

	/** ID of the opposide direction. */
	opposite_dir_id = null;

	/**
	 * Array with 3 directions where road can proceed from this direction.<p>
	 * Includes "left", "right", and self obviously. Self is always last one
	 *  (has index 2 in the array).
	 */
	further_directions_ordered = null;

	/**
	 * ID of perfect tile slope for road to move up, following "this" dir.<p>
	 * "Perfect" mean:
	 *  "can build without terraforming, or 'build on slopes' game option.
	 */
	perfect_upward_slope = null;

	/**
	 * ID of perfect tile slope for road to move down, following "this" dir.<p>
	 * "Perfect" mean:
	 *  "can build without terraforming, or 'build on slopes' game option.
	 */
	perfect_downward_slope = null;

	/**
	 * RoadPFDataStructures.RoadDirection constructor.
	 * @param id New direction ID.
	 * @param offset Right offset for the given direction ID.
	 */
	constructor(id, offset)
	{
		::Terron_Object.constructor(null);
		this.id = id;
		this.offset = offset;
		this.opposite_dir_id = GetOppositeDirectionID(id);

		local l = GetToLeftDirectionID(id);
		local r = GetToRightDirectionID(id);
		this.further_directions_ordered = [l, r, id];

		this.perfect_upward_slope = GetPerfectSlope(id);
		this.perfect_downward_slope = GetPerfectSlope(this.opposite_dir_id);

		if (offset > 1) {
			this.y_offset = 1;
		} else if (offset < -1) {
			this.y_offset = -1;
		} else {
			this.x_offset = offset;
		}
	}

	/**
	 * Get next tile coordinates for following "this" direction.
	 * @param x Initial tile X coordinate.
	 * @param y Initial tile Y coordinate.
	 * @param length Step length.
	 * @return Table with coordinates of the tile received by following
	 *  from the given tile in "this" direction for the given length.
	 */
	function GetNextXY(x, y, length)
	{
		return {x = x + this.x_offset * length, y = y + this.y_offset * length};
	}

/* private */
	/** private, name obvious */
	static function GetOppositeDirectionID(dir_id)
	{
		switch (dir_id) {
			case RoadDir.NE: return RoadDir.SW;
			case RoadDir.SE: return RoadDir.NW;
			case RoadDir.SW: return RoadDir.NE;
			case RoadDir.NW: return RoadDir.SE;
		}
		assert(null);
	}

	/** private, name obvious */
	static function GetToLeftDirectionID(dir_id)
	{
		switch (dir_id) {
			case RoadDir.NE: return RoadDir.NW;
			case RoadDir.SE: return RoadDir.NE;
			case RoadDir.SW: return RoadDir.SE;
			case RoadDir.NW: return RoadDir.SW;
		}
		assert(null)
	}

	/** private, name obvious */
	static function GetToRightDirectionID(dir_id)
	{
		switch (dir_id) {
			case RoadDir.NE: return RoadDir.SE;
			case RoadDir.SE: return RoadDir.SW;
			case RoadDir.SW: return RoadDir.NW;
			case RoadDir.NW: return RoadDir.NE;
		}
		assert(null);
	}

	/** private, perfect up slope meant */
	static function GetPerfectSlope(dir_id)
	{
		switch (dir_id) {
			case RoadDir.NE: return AITile.SLOPE_NE;
			case RoadDir.SE: return AITile.SLOPE_SE;
			case RoadDir.SW: return AITile.SLOPE_SW;
			case RoadDir.NW: return AITile.SLOPE_NW;
		}
		assert(null);
	}

	/**
	 * Calculate direction ID.
	 * @param t1 Some "start" tile ID.
	 * @param t2 Some "end" tile ID.
	 * @note t1 and t2 must lie in same line.
	 * @return ID of the direction that leads from start to end.
	 */
	static function GetLineDirectionID(t1, t2)
	{
		local x1 = AIMap.GetTileX(t1);
		local y1 = AIMap.GetTileY(t1);
		local x2 = AIMap.GetTileX(t2);
		local y2 = AIMap.GetTileY(t2);
		return x1 == x2 ? (y1 < y2 ? RoadDir.SE : RoadDir.NW) :
			(x1 < x2 ? RoadDir.SW : RoadDir.NE);
	}
}

Terron_ClassTable.RegisterClass(RoadPFDataStructures);
