/**
 * "Simple pathfinder" road building section implementation.
 */
class SimplePathBuilder
{
/* public */
	/**
	 * Build a road found with "simple pathfinder".
	 * @param path Result of SimplePF().FindPath(...);
	 * @return true When succeed, false otherwise.
	 */
	static function BuildPath(path)
	{
		if (path == null) {
			AILog.Warning("path == null...");
			return -1;
		}
		if (path.len() <= 1) {
			AILog.Warning("path.len() <= 1...");
			return 0;
		}

		/*for (local i = 0; i < path.len() - 1; i++) {
			local mode = AIExecMode();
			AISign.BuildSign(path[i].tile, path[i].build_token);
		}*/
		local result = SimplePathBuilder.BuildPathTunnels(path);
		if (result != 0) {
			AILog.Warning("Failed to build road path tunnels...");
			/*for (local i = 0; i < path.len() - 1; i++) {
				local mode = AIExecMode();
				AISign.BuildSign(path[i].tile, path[i].build_token);
			}*/
			return result;
		}

		result = SimplePathBuilder.BuildPathBridges(path);
		if (result != 0) {
			AILog.Warning("Failed to build road path brdges...");
			/*for (local i = 0; i < path.len() - 1; i++) {
				local mode = AIExecMode();
				AISign.BuildSign(path[i].tile, path[i].build_token);
			}*/
			return result;
		}

		local road_sections = [];
		local j = 0;
		for (local i = 0; i < path.len() - 1; i++) {
			if (path[i].build_token != BuildToken.BT_ROAD) {
				if (i == j) continue;
				if (i != j + 1 || j == 0) {
					road_sections.append([j, i]);
				}
				j = i + 1;
			}
		}
		road_sections.append([j, path.len() - 1]);

		foreach (dummy_id, section in road_sections) {
			j = section[0];
			for (local i = j; i < section[1]; i++) {
				if (path[i + 1].dir.id != path[i].dir.id) {
					local s = path[j].tile;
					local e = path[i].tile;
					if (!SimplePathBuilder.doBuildRoadPiece(s, e)) {
						return AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH ? -2 : -1;
					}
					j = i;
				}
			}
			if (!SimplePathBuilder.doBuildRoadPiece(path[j].tile, path[section[1]].tile)) {
				result = AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH ?
					-2 : -1;
				return result;
			}
		}

		return 0;
	}

	/**
	 * Get cost of a road found with "simple pathfinder".
	 * @param path Result of SimplePF().FindPath(...);
	 * @return Some cost. Accuracy seems adequate.
	 */
	static function GetPathCost(path)
	{
		local test = AITestMode();
		local accounting = AIAccounting();
		if (SimplePathBuilder.BuildPath(path) != 0) return -1;
		return accounting.GetCosts() + 2000;
	}
}

/**
 * Helper to build bridges.
 */
/* private */ function SimplePathBuilder::BuildPathBridges(path)
{
	/* Here we'll store tiles where we succeed in bridge building */
	// to avoid test mode double build - there is two build tokens for bridge
	//  and in test mode algorithm would try to build bridge twice without
	//  special approach.
	local succeed_build_tiles = {};
	for (local i = 0; i < path.len() - 1; i++) {
		local pf_node = path[i];
		if (pf_node.build_token != BuildToken.BT_NEW_BRIDGE) continue;

		local s = pf_node.tile;
		if (AIBridge.IsBridgeTile(s) || (s in succeed_build_tiles)) continue;

		local e = path[i + 1].tile;
		if (!AITile.IsBuildable(s)) AITile.DemolishTile(s);
		if (!AITile.IsBuildable(e)) AITile.DemolishTile(e);

		local b_list = AIBridgeList_Length(AIMap.DistanceManhattan(s, e) + 1);
		if (b_list.IsEmpty()) {
			/*{
				local mode = AIExecMode();
				AISign.BuildSign(path[i].tile, "no bridge");
				AISign.BuildSign(path[i + 1].tile, "no bridge");
			}*/
			return -1;
		}

		/* build one of 3 fastest bridges */
		local k = 0;
		local best_bridge = null;
		local best_speed = -1;
		foreach (b, value in b_list) {
			local b_speed = AIBridge.GetMaxSpeed(b)
			if (b_speed > best_speed) {
				if (k > 1 || best_bridge == null) {
					best_bridge = b;
					best_speed = b_speed;
					k = AIBase.RandRange(3);
				}
				k = (k + 1) % 3;
			}		
		}
		if (!AIBridge.BuildBridge(AIVehicle.VT_ROAD, best_bridge, s, e)) {
			return AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH ? -2 : -1;
		}
		succeed_build_tiles[s] <- 0;
		succeed_build_tiles[e] <- 0;
	}

	local pf_data = RoadPFDataStructures.Get();
	foreach (dummy_id, pf_node in path) {
		if (
			pf_node.build_token != BuildToken.BT_NEW_BRIDGE &&
			pf_node.build_token != BuildToken.BT_BRIDGE
		) {
				continue;
		}

		local t = pf_node.tile;
		if (AIBridge.IsBridgeTile(t)) {
			local dir = RoadPFDataStructures.RoadDirection.GetBridgeDirectionID(t);
			local exit = t - pf_data.directions[dir].offset;
			if (!AIRoad.AreRoadTilesConnected(t, exit)) {
				SimplePathBuilder.doBuildRoadPiece(t, exit);
			}
		}
	}

	return 0;
}

/**
 * Helper to build tunnels.
 */
/* private */ function SimplePathBuilder::BuildPathTunnels(path)
{
	/* Here we'll store tiles where we succeed in bridge building */
	// to avoid test mode problems
	local succeed_build_tiles = {};
	local pf_data = RoadPFDataStructures.Get();
	foreach (dummy_id, pf_node in path) {
		if (
			pf_node.build_token != BuildToken.BT_NEW_TUNNEL &&
			pf_node.build_token != BuildToken.BT_TUNNEL
		) {
				continue;
		}

		local t = pf_node.tile;
		if (!AITunnel.IsTunnelTile(t) || (t in succeed_build_tiles)) {
			if (!AITile.IsBuildable(t)) AITile.DemolishTile(t);
			if (!AITunnel.BuildTunnel(AIVehicle.VT_ROAD, t)) {
				return AIError.GetLastError() == AIError.ERR_NOT_ENOUGH_CASH ? -2 : -1;
			}
			local e = AITunnel.GetOtherTunnelEnd(t);
			succeed_build_tiles[t] <- 0;
			succeed_build_tiles[e] <- 0;
		}

		if (AITunnel.IsTunnelTile(t)) {
			local dir = RoadPFDataStructures.RoadDirection.GetTunnelDirectionID(t);
			local exit = t - pf_data.directions[dir].offset;
			if (!AIRoad.AreRoadTilesConnected(t, exit)) {
				SimplePathBuilder.doBuildRoadPiece(t, exit);
			}
		}
	}
	return 0;
}

/**
 * Helper to build primitive road parts.
 */
/* private */ function SimplePathBuilder::doBuildRoadPiece(s, e)
{
	if (AIRoad.BuildRoad(s, e)) return true;
	switch (AIError.GetLastError()) {
		case AIError.ERR_ALREADY_BUILT:
		case AIRoad.ERR_ROAD_CANNOT_BUILD_ON_TOWN_ROAD:
			return true;

		case AIError.ERR_VEHICLE_IN_THE_WAY:
		case AIRoad.ERR_ROAD_WORKS_IN_PROGRESS:
			for (local i = 0; i < 25; i++) {
				if (AIRoad.BuildRoad(s, e)) return true;
				AIController.Sleep(25);
			}
			return false;

		case AIError.ERR_LOCAL_AUTHORITY_REFUSES:
		case AIError.ERR_AREA_NOT_CLEAR:
		case AIError.ERR_OWNED_BY_ANOTHER_COMPANY:
		case AIError.ERR_FLAT_LAND_REQUIRED:
		case AIError.ERR_LAND_SLOPED_WRONG:
		case AIError.ERR_SITE_UNSUITABLE:
		case AIError.ERR_TOO_CLOSE_TO_EDGE:
		case AIRoad.ERR_ROAD_ONE_WAY_ROADS_CANNOT_HAVE_JUNCTIONS:
		case AIError.ERR_NOT_ENOUGH_CASH:
		case AIError.ERR_PRECONDITION_FAILED:		
			return false;
		default:
			return false;
	}
	return false;
}
