/**
 * Bidirectional A* pathfinder.
 * "Simple" because it considers 'tile' as a 'pf node'
 *  (correct one should be 'tile' + 'in/out direction').
 * Some implemetation notes maybe should be here.
 */
class SimplePF
{
/* public */
	/**
	 * Find route between given tiles.
	 * @param s TileList with possible path start/begin tiles.
	 * @param e TileList with possible path end tiles.
	 * @param cost_table Tile types costs - instance of BaseRoadCost class.
	 * @param tiles_to_ignore Not used.
	 */
	function FindPath(s, e, cost_table, tiles_to_ignore)
	{
		if (s.len() == 0 || e.len() == 0) return null;
		if (s[0] == e[0]) return [s];

		this.cost_table = cost_table;
		this.pf_data = RoadPFDataStructures.Get();
		this.pf_helper = SimpleRoadCostMeasurer(cost_table);
		this.rectangle = SimpleSearchRectangle(s, e);
		local statistics = {opened = 0,closed = 0, poped = 0};

		local tt = AIController.GetTick();
		CodeUtils.Log("Starting pathfining...", 0);
		local s_open_nodes = this.SetUp(s, e[0]);
		local e_open_nodes = this.SetUp(e, s[0]);
		local s_node = rectangle.GetNode(AIMap.GetTileX(s[0]), AIMap.GetTileY(s[0]));
		local e_node = rectangle.GetNode(AIMap.GetTileX(e[0]), AIMap.GetTileY(e[0]));

		local open_nodes, goal_node;
		while (true) {
			if (s_open_nodes.Count() < e_open_nodes.Count()) {
				open_nodes = s_open_nodes;
				goal_node = e_node;
			} else {
				open_nodes = e_open_nodes;
				goal_node = s_node;
			}
			if (open_nodes.IsEmpty()) break;

			local best_node = open_nodes.Pop();
			statistics.poped++;
			if (best_node.bingo != null) {
				CodeUtils.Log("PF SUCCEED...", 0);
				CodeUtils.Log("     opened " + statistics.opened, 0);
				CodeUtils.Log("     closed " + statistics.closed, 0);
				CodeUtils.Log("     poped " + statistics.poped, 0);
				CodeUtils.Log("     ticks " + (AIController.GetTick() - tt), 0);
				return this.CreateFormatedPath(best_node);
			}
			if (best_node.state == PFNodeState.NS_CLOSED) continue;
			best_node.state = PFNodeState.NS_CLOSED;
			statistics.closed++;

			foreach (dummy_id, info in pf_helper.GetNeighbours(best_node, rectangle)) {
				local next = info.node;
				local d = best_node.distance_to_start + info.cost;
				if (d >= BaseRoadCost.infinite) continue

				if (next.goal_tile != null && next.goal_tile != best_node.goal_tile) {
					if (
						(next.build_token == BuildToken.BT_ROAD && info.build_token == BuildToken.BT_ROAD)
						|| (info.dir.id == next.dir.opposite_dir_id && best_node.build_token == BuildToken.BT_ROAD)
					)
					if (AIRoad.CanBuildConnectedRoadPartsHere(next.tile, next.parent_node.tile, best_node.tile)) {
						next.bingo = [best_node, info.cost, info.build_token, info.dir];
						if (next.dir.id != best_node.dir.opposite_dir_id) {
							d += this.cost_table.turn_cost;
						}
						local tmp = next.dir;
						next.dir = info.dir;
						open_nodes.Insert(next, d + H(next, goal_node));
						next.dir = tmp;
						continue;
					}
					continue;
				}
				if (d < next.distance_to_start) {
					next.distance_to_start = d;
					next.parent_node = best_node;
					next.goal_tile = best_node.goal_tile;
					next.state = PFNodeState.NS_OPENED;
					next.dir = info.dir;
					statistics.opened++;
					next.build_token = info.build_token;
					open_nodes.Insert(next, d + H(next, goal_node));
				}
			}
		}
		CodeUtils.Log("PF FAILED", 0);
		CodeUtils.Log("       ticks " + (AIController.GetTick() - tt), 0);
		return null;
	}

/* private */
	/** Cost defining table. */
	cost_table = null;

	/** Helper to easily access low level data */
	pf_data = null;

	/** Distance measurer. */
	pf_helper = null;

	/** Rectangle area where the path search occur. */
	rectangle = null;

	/**
	 * Heuristic function.
	 * @param a Some "start" Tile ID.
	 * @param b Some "end" Tile ID.
	 * @reutrn Expected cost for the path from start to end.
	 */
	function H(a, b)
	{
		local dx = a.x - b.x;
		local dy = a.y - b.y;

		local m = a.dir.x_offset * dx + a.dir.y_offset * dy;
		local turn_cost = (m < 0) ? this.cost_table.turn_cost :
			(m > 0) ? 2 * this.cost_table.turn_cost : 0;

		return turn_cost + this.cost_table.h_base * (abs(dx) + abs(dy));
	}

	/**
	 * Create pathfinder's "open nodes" container.
	 * @param tiles Array with pathfinding "start" tile IDS.
	 * @param expected_end_tile Expected path endpoint.
	 * @return Newly created "open nodes" container.
	 */
	function SetUp(tiles, expected_end_tile)
	{
		local is_item_in_array = function (item, array) {
			foreach (dummy_id, t in array) {
				if (t == item) return true;
			}
			return false;
		}

		local open_nodes = Terron_FibonacciHeap();
		foreach (dummy_id, t in tiles) {
			if (AIBridge.IsBridgeTile(t) || AITunnel.IsTunnelTile(t)) continue;

			local node = pf_data.GetPrimitiveNode(t);
			node.state = PFNodeState.NS_CLOSED;
			node.distance_to_start = 0;
			node.parent_node = node;
			node.goal_tile = expected_end_tile;
			node.build_token == BuildToken.BT_ROAD;

			rectangle.SetNode(node.x, node.y, node);
			foreach (dir_id, dir in pf_data.directions) {
				local next = pf_data.GetPrimitiveNode(t + dir.offset);
				if (is_item_in_array(next.tile, tiles)) continue;
				next.dir = dir;
				next.distance_to_start = pf_helper.GetCost(node, next, dir);
				if (next.distance_to_start >= BaseRoadCost.infinite) continue;
				next.parent_node = node;
				next.goal_tile = expected_end_tile;
				next.build_token = BuildToken.BT_ROAD;
				rectangle.SetNode(next.x, next.y, next);

				local h = this.cost_table.h_base * AIMap.DistanceManhattan(next.tile, expected_end_tile);
				open_nodes.Insert(next, next.distance_to_start + h);
			}
		}
		return open_nodes;
	}

	/**
	 * Create actual "path" object when pathfinder successfuly stopped.
	 * @param pf_stop_node Node object - intersection of the two search clouds.
	 * @return Array of nodes representing found path.
	 */
	function CreateFormatedPath(pf_stop_node)
	{
		if (pf_stop_node.bingo == null) assert(null);

		/*
		 * Basicaly what is going on.
		 * At start we have only reference to parent node,
		 *  and because of bidirectional search we have actually 2 parents,
		 *  and because of bidirectional search we'll receive 2 paths(initialy).
		 * a_path - nodes from path start to pf_stop_node.
		 * b_path - from pf_stop_node till end.
		 * Order of nodes in a_path and b_path will be:
		 *  from pf_stop_node to respected path start, or end.
		 * Thus we initialy receive something like this:
		 *  [S]<-...<-[STOP] * [STOP]->..->[E].
		 * Further we revert one path(b one) and receive nice result:
		 *  [S]->-...->[STOP]->..->[E]
		 */
		// there is also proper build tokens assignment:
		// initialy we have only one token per bridge or tunnel(it is more
		// convinient in pf process), and must make proper two tokens - one for
		// each of bridge's or tunnel's ends(two tokens is more convinient
		// during build process).

		local a_path = [];
		for (local i = pf_stop_node; i.parent_node.tile != i.tile;) {
			i = i.parent_node;
			a_path.insert(0, i);
		}
		for (local j = 1; j < a_path.len(); j++) {
			local x = a_path[j];
			if (x.build_token != BuildToken.BT_ROAD) {
				if (x.parent_node.build_token == 0) continue;
				a_path[j - 1].build_token = x.build_token;
			}
		}
		if (pf_stop_node.build_token != BuildToken.BT_ROAD) {
			local n = a_path.len();
			if (n > 0 && a_path[n - 1].build_token != 0) {
				a_path[n - 1].build_token = pf_stop_node.build_token;
			}
		}

		local bingo_node = pf_stop_node.bingo[0];
		local b_path = [pf_stop_node, pf_stop_node.bingo[0]];
		for (local i = pf_stop_node.bingo[0]; i.parent_node.tile != i.tile;) {
			i = i.parent_node;
			b_path.append(i);
		}
		for (local j = b_path.len() - 1; j > 1; j--) {
			b_path[j].parent_node = b_path[j - 1];
			b_path[j].dir = pf_data.directions[b_path[j - 1].dir.opposite_dir_id];
		}
		if (pf_stop_node.bingo[2] == BuildToken.BT_ROAD) {
			foreach (dummy_id, direction in pf_data.directions) {
				if (bingo_node.tile == pf_stop_node.tile + direction.offset) {
					bingo_node.dir = direction;
					break;
				}
			}
		} else {
			bingo_node.dir = pf_data.directions[pf_stop_node.bingo[3].opposite_dir_id];
		}

		for (local j = b_path.len() - 1; j >= 0; j--) {
			local token = b_path[j].parent_node.build_token;
			if (token != BuildToken.BT_ROAD && token != BuildToken.BT_INVALID) {
				b_path[j].build_token = b_path[j].parent_node.build_token;
			}
		}

		local result = a_path;
		pf_stop_node.bingo[0].build_token = pf_stop_node.bingo[2];
		//CodeUtils.Log(pf_stop_node.bingo[0].build_token + " <- " + pf_stop_node.bingo[2], 0);
		if (pf_stop_node.bingo[2] != BuildToken.BT_ROAD) {
			pf_stop_node.build_token = pf_stop_node.bingo[2];
		}
		result.extend(b_path);

		result[0].dir = result[1].dir;
		return result;
	}
}
