/**
 * Neighbour info table.<p>
 * Small table, which clones will hold information about pathfinder steps -
 *  one clone for one possible step.
 */
NeighbourInfo <- {
	/** Pathfinding node, representing step end/target */
	node = null

	/** Step cost */
	cost = 0

	/** Step direction */
	dir = -1

	/** What must be placed */
	build_token = BuildToken.BT_ROAD
};

/**
 * Class that translates game data into innner pathfinder data.
 */
class SimpleRoadCostMeasurer
{
/* public */
	/**
	 * SimpleRoadCostMeasurer constructor.
	 * @param cost_table Table that defines tile types cost.
	 */
	constructor(cost_table)
	{
		this.cost_table = cost_table;
		this.pf_data = RoadPFDataStructures.Get();
	}

	/**
	 * Get cost of path between the given(assumed adjacent) nodes.
	 * @param current "Start" node.
	 * @param next "Destination" node.
	 * @param dir Direction from "start" to "destination".
	 * @return Number that is path cost.
	 */
	function GetCost(current, next, dir)
	{
		local penalty = 0;
		local t = next.tile;

		if (AIRail.IsRailTile(t)) {
			penalty = 600;
		} else if (!AITile.IsBuildable(t)) {
			if (!AIRoad.IsRoadTile(t)) return this.cost_table.infinite;
			if (AIRoad.IsDriveThroughRoadStationTile(t)) {
				return AICompany.IsMine(AITile.GetOwner(t)) ?
					this.cost_table.drive_through_station_cost :
					this.cost_table.infinite;
			}
		}
		local s = AITile.GetSlope(t);
		if (!TileUtils.IsSlopeFlatOrWithThreeCornersRaised(s)) {
			/*
			 * This is very important part for pathfinder speed
			 *  in mountain maps. With low enough perfect_slope_penalty (how
			 *  exactly low depends on other cost table parameters,
			 *  so no concrete numbers here, though 0 is lowest => ok)
			 *  pf can "pierce" hills in well, perfectly sloped parts, and ignore
			 *  other hill parts, significantly lowering both amount of nodes to
			 *  open and search direction change numbers(mean lesser situations
			 *  like: "good, we were following N(orth) for 50 steps, but here is
			 *   hill now, so for EACH previous step check E and W direction
			 *   neighbours - maybe they are better").
			 *  Tbh speed also depend on heuristic multiplier, terrain in each
			 *  concrete case, and, yes, sometimes we need to check most of
			 *  previous steps, but usually no and for several percents of
			 *  accuracy(pf'll find less optimal path) we receive noticeable
			 *   speed bonus, so this part of code is here.
			 * Well explanation seems abit confusing, but this parameter meaning
			 *  is really hard to show without pf work printed and drawed with
			 *  signs and open/closed nodes counter at hands and other debug
			 *  stuff like that.
			 */
			if (s != dir.perfect_upward_slope && s != dir.perfect_downward_slope) {
				penalty += this.cost_table.slope_penalty;
			} else {
				penalty += this.cost_table.perfect_slope_penalty;
			}
		}
		if (AIRoad.AreRoadTilesConnected(current.tile, t)) {
			return this.cost_table.perfect_path_cost + penalty;
		}

		if (AITile.GetMinHeight(t) == 0) {
			penalty += TileUtils.IsSlopeWithOneCornerRaised(s) ?
				cost_table.costForWaterSlope : cost_table.costForCloseWater;
		}
		return this.cost_table.new_road_cost + penalty;
	}

	/**
	 * Get all neighbour nodes of the given node.
	 * @param pf_node Pathfinding node.
	 * @param rectangle Pathfinding area rectangle.
	 * @return Array of NeighbourInfo tables.
	 */
	function GetNeighbours(pf_node, rectangle)
	{
		local result = [];
		local t = pf_node.tile;
		if (
			(pf_node.build_token == BuildToken.BT_BRIDGE && AIBridge.GetOtherBridgeEnd(t) != pf_node.parent_node.tile) ||
			(pf_node.build_token == BuildToken.BT_TUNNEL && AITunnel.GetOtherTunnelEnd(t) != pf_node.parent_node.tile)
		) {
			local info = GetExistingTunnelBridgeNeighbours(pf_node, rectangle);
			pf_node.build_token = BuildToken.BT_ROAD;
			if (info != null) result.append(info);
			return result;
		}

		local s = pf_node.parent_node.tile;
		local directions = pf_data.directions;
		foreach (dummy_id, dir_id in pf_node.dir.further_directions_ordered) {
			if (dummy_id < 2 && pf_node.build_token != BuildToken.BT_ROAD) {
				continue;
			}

			local dir = directions[dir_id];
			local next_x = pf_node.x + dir.x_offset;
			local next_y = pf_node.y + dir.y_offset;
			if (!rectangle.IsInRect(next_x, next_y)) continue;

			local next = rectangle.GetNode(next_x, next_y);
			if (next.state == PFNodeState.NS_CLOSED) continue;
			
			local e = next.tile;
			if (!AIRoad.CanBuildConnectedRoadPartsHere(t, s, e)) continue;

			local info = clone ::NeighbourInfo;
			info.node = next;
			info.dir = dir;
			if (AIBridge.IsBridgeTile(e)) {
				/* Must enter bridge correct - from "front", not left/right */
				local bridge_dir_id = RoadPFDataStructures.RoadDirection.GetBridgeDirectionID(e);
				if (dir_id != bridge_dir_id || bridge_dir_id < 0) continue;
				info.cost = this.cost_table.perfect_path_cost;
				info.build_token = BuildToken.BT_BRIDGE;
			} else if (AITunnel.IsTunnelTile(e)) {
				/* Must enter tunnel correct - from "front", not left/right */
				local tunnel_dir_id = RoadPFDataStructures.RoadDirection.GetTunnelDirectionID(e);
				if (dir_id != tunnel_dir_id || tunnel_dir_id < 0) continue;
				info.cost = this.cost_table.perfect_path_cost;
				info.build_token = BuildToken.BT_TUNNEL;
			} else {
				info.cost = this.GetCost(pf_node, next, dir);
			}

			if (pf_node.dir.id != dir_id) info.cost += this.cost_table.turn_cost;

			result.append(info);

			if (dir_id == pf_node.dir.id) {
				/* Water tiles cost is infinite, so check for possible bridge */
				if (info.cost >= this.cost_table.infinite) {
					if (AITile.IsWaterTile(e)) {
						local new_bridge = this.GetNewBridgeNeighbour(pf_node, 1, 30, rectangle);
						if (new_bridge != null) result.append(new_bridge);
					}
				} else if (AIRail.IsRailTile(e)) {
					/* Cost not infinite, but we faced rails */
					/* Seek the bridge over those rails */
					local new_bridge = this.GetNewBridgeNeighbour(pf_node, 1, 6, rectangle);
					if (new_bridge != null) result.append(new_bridge);
				} else {
					local slope = AITile.GetSlope(t);
					if (slope == pf_node.dir.perfect_upward_slope) {
						local new_tunnel = this.GetNewTunnelNeighbour(pf_node, 1, 10, rectangle);
						if (new_tunnel != null) {
							/*
							 * If new tunnel length is 1 =>
							 * => always build it, not road */
							if (new_tunnel.node.tile == e) {
								foreach (dummy_id, info in result) {
									if (info.node.tile == e) info.build_token = BuildToken.BT_NEW_TUNNEL;
								}
							} else {
								result.append(new_tunnel);
							}
						}
					}
				}
			}
		}
		return result;
	}

/* protected */
	/** Saved cost table. */
	cost_table = null;

	/** Saved for fast access pf data object. */
	pf_data = null;

	/**
	 * Get special case neighbour node representing new(required to build)
	 *  bridge.
	 * @param pf_node Pathfinding node.
	 * @param min_bridge_len Minimal allowed bridge length.
	 * @param max_bridge_len Maximal allowed bridge length.
	 * @param rectangle Pathfinding area rectangle.
	 */
	function GetNewBridgeNeighbour(node, min_bridge_len, max_bridge_len, rectangle)
	{
		local test = AITestMode();
		local s = node.tile;
		if (!AIRoad.BuildRoad(s, s - node.dir.offset)) return null;

		local result = clone ::NeighbourInfo;
		local exit_x = node.x, exit_y = node.y; 
		for (local t = s, i = min_bridge_len; i < max_bridge_len; i++) {
			t += node.dir.offset;
			exit_x += node.dir.x_offset;
			exit_y += node.dir.y_offset;
			local next_x = exit_x + node.dir.x_offset;
			local next_y = exit_y + node.dir.y_offset;
			if (!rectangle.IsInRect(next_x, next_y)) return null;
			if (AITile.IsWaterTile(t) || AIRail.IsRailTile(t)) continue;
			if (!AIRoad.BuildRoad(t, t + node.dir.offset)) continue;
			if (AIRail.IsRailTile(t + node.dir.offset)) continue;

			local b_list = AIBridgeList_Length(i + 1);
			if (b_list.IsEmpty()) continue;
			if (AIBridge.BuildBridge(AIVehicle.VT_ROAD, b_list.Begin(), s, t)) {
				local exit = rectangle.GetNode(exit_x, exit_y);
				if (exit.state == PFNodeState.NS_CLOSED) continue;
				result.node = exit;
				result.dir = node.dir;
				//result.cost = i * this.cost_table.base_bridge_cost;
				result.cost = this.GetBridgeCost(i);
				result.build_token = BuildToken.BT_NEW_BRIDGE;
				if (AITile.GetMinHeight(exit.tile) == 0) {
					local sl = AITile.GetSlope(exit.tile);
					if (TileUtils.IsSlopeWithOneCornerRaised(sl)) {
						result.cost += this.cost_table.costForWaterSlope;
					}
				}
				return result;
			}
		}
		return null;
	}

	/**
	 * Get special case neighbour node representing new(required to build)
	 *  tunnel.
	 * @param pf_node Pathfinding node.
	 * @param min_tunnel_len Minimal allowed tunnel length.
	 * @param max_tunnel_len Maximal allowed tunnel length.
	 * @param rectangle Pathfinding area rectangle.
	 */
	function GetNewTunnelNeighbour(node, min_tunnel_len, max_tunnel_len, rectangle)
	{
		local test = AITestMode();
		local s = node.tile;
		if (!AIRoad.BuildRoad(s, s - node.dir.offset)) return null;

		local exit_tile = AITunnel.GetOtherTunnelEnd(s);
		if (!AIMap.IsValidTile(exit_tile)) return null;

		local tunnel_length = AIMap.DistanceManhattan(s, exit_tile);
		if (max_tunnel_len < tunnel_length || tunnel_length < min_tunnel_len) {
			return null;
		}

		local exit_slope = AITile.GetSlope(exit_tile);
		if (exit_slope != node.dir.perfect_downward_slope) return null;
		if (!AIRoad.BuildRoad(exit_tile, exit_tile + node.dir.offset)) {
			return null;
		}

		local exit_x = AIMap.GetTileX(exit_tile);
		local exit_y = AIMap.GetTileY(exit_tile);
		if (!rectangle.IsInRect(exit_x, exit_y)) return null;

		if (AITunnel.BuildTunnel(AIVehicle.VT_ROAD, s)) {
			local exit = rectangle.GetNode(exit_x, exit_y);
			if (exit.state == PFNodeState.NS_CLOSED) return null;
			local result = clone ::NeighbourInfo;
			result.node = exit;
			result.dir = node.dir;
			result.cost = tunnel_length * this.cost_table.base_tunnel_cost;
			result.build_token = BuildToken.BT_NEW_TUNNEL;
			return result;
		}
		return null;
	}

	/**
	 * Get special case neighbour node representing existing tunnel/bridge.
	 * @param pf_node Pathfinding node.
	 * @param rectangle Pathfinding area rectangle.
	 */
	function GetExistingTunnelBridgeNeighbours(pf_node, rectangle)
	{
		local t = pf_node.tile;
		local tunnel_bridge_end = null;
		if (pf_node.build_token == BuildToken.BT_BRIDGE) {
			tunnel_bridge_end = AIBridge.GetOtherBridgeEnd(t);
		} else if (pf_node.build_token == BuildToken.BT_TUNNEL) {
			tunnel_bridge_end = AITunnel.GetOtherTunnelEnd(t);
		} else {
			assert(null);
		}

		local length = AIMap.DistanceManhattan(t, tunnel_bridge_end);
		/* We are entering tunnel/bridge from side direction */
		if (tunnel_bridge_end != (t + length * pf_node.dir.offset)) return null;

		local exit_x = pf_node.x + length * pf_node.dir.x_offset;
		local exit_y = pf_node.y + length * pf_node.dir.y_offset;
		local next_x = exit_x + pf_node.dir.x_offset;
		local next_y = exit_y + pf_node.dir.y_offset;
		/* This tunnel/bridge lead too far away from our target, ignore */
		if (!rectangle.IsInRect(next_x, next_y)) return null;

		local exit = rectangle.GetNode(exit_x, exit_y);
		local result = clone ::NeighbourInfo;
		result.node = exit;
		result.dir = pf_node.dir;
		result.cost = length * this.cost_table.perfect_path_cost;
		result.build_token = pf_node.build_token;
		return result;
	}

	/**
	 * Function to artificially increase long bridges cost.
	 */
	function GetBridgeCost(bridge_len)
	{
		local m = bridge_len.tofloat() / 10;
		if (m <= 1) return bridge_len * this.cost_table.base_bridge_cost;
		return (m * this.cost_table.base_bridge_cost * bridge_len).tointeger();
	}
}
