/**
 * Class that handles simple road routes.
 */
class RoadRoute extends SimpleRoute
{
/* public */
	/**
	 * Creates a new road route.
	 * @param start_node First node.
	 * @param end_node Second node.
	 * @param cargo_id ID of the cargo to transport.
	 */
	constructor(start_node, end_node, cargo_id, is_transit)
	{
		::SimpleRoute.constructor(start_node, end_node, cargo_id, is_transit);

		this.ev = Property(this.doGetEstimatedVehicles.bindenv(this), 21);
		this.ev.SetName("Get vehicles count for route [" + this.name + "]");
		this.GetEngine();

		/* Allow additional stations for passengers transportation */
		if (AICargo.HasCargoClass(cargo_id, AICargo.CC_PASSENGERS)) {
			//this.max_level = 1;
			this.max_level = is_transit ? 0 : 1;
		}

		/* bind this route to a RoadConnection object */
		local s = start_node.location;
		local e = end_node.location;
		local map = RoadDepartment.Get().GetNodesRoadMap();
		this.connection = map.GetConnection(s, e);
		if (this.connection == null) {
			this.connection = RoadConnection(s, e);
			this.connection.cost = RoadSettings.Get().GetRoadConnectionCost(s, e);  
			map.RegisterConnection(this.connection);
		}

		CodeUtils.Log("Road route [" + this.name + "] revealed", 1);
	}

	function GetVehiclesCost()
	{
		return this.GetEstimatedVehicles() * this.best_engine_price;
	}

	function GetEstimatedIncome()
	{
		return this.GetEstimatedVehicles() * this.vehicle_profit;
	}

	/**
	 * Get number of vehicles required to saturate this route.
	 * @return Number of vehicles required to saturate this route.
	 */
	function GetEstimatedVehicles()
	{
		return this.ev.Get();
	}

	function GetMaxRouteLevel()
	{
		return this.max_level;
	}

	/**
	 * Get income for delivery of one cargo unit.
	 * @return Amount of money AI will receive for delivery of one cargo unit
	 *  from the route's start to end.
	 */
	function GetNormalCargoIncome()
	{
		return this.normal_cargo_income;
	}

/* proteted */
	/**
	 * Get estimated number of vehicles for this route.
	 * @return Estimated number of vehicles required for this route.
	 */
	function doGetEstimatedVehicles()
	{
		return this.GetEngine() == -1 ? 0 :
			max(1, (this.GetFreeProduction() / this.transport_rate).tointeger());
	}

	function doGetEngine()
	{
		/* Set correct values for the case when there is no valid engine */
		this.vehicle_profit = 0;
		this.transport_rate = 0;

		local e = RoadEngineRoster.Get().GetBestEngine(this.cargo_id, false);
		if (e == -1) {
			this.MakeNefarious(GameTime.YEAR);
			return -1;
		}

		local stats = EnginesInfo.stats[e];
		local speed = stats[EngineStat.ES_MAX_SPEED];
		local capacity = stats[EngineStat.ES_CAPACITY];
		this.best_engine_price = stats[EngineStat.ES_PRICE];

		/* time for full transport cycle */
		//local t = 8 + (668 * 2 * this.length / (24 * 0.9 * speed)).tointeger();
		// same as above
		local t = 8 + 62 * this.length / speed;

		local ci = AICargo.GetCargoIncome(this.cargo_id, this.length, t / 2); 

		this.transport_rate = (capacity * GameTime.MONTH).tofloat() / t;

		this.normal_cargo_income = ci;

		local profit = this.transport_rate * (this.one_way ? ci : 2 * ci);
		profit -= stats[EngineStat.ES_MONTH_RUNNING_COST];
		profit -= this.best_engine_price * GameTime.MONTH / (5 * stats[EngineStat.ES_MAX_AGE]);

		if (this.IsTransit()) {
			profit += capacity * this.end_node.GetAdditionalTransitIncome(this.cargo_id);
		}

		this.vehicle_profit = profit.tointeger();

		return e;
	}

	/**
	 * Get suitable station (if any) near the route start node.
	 * @param level Desired station level.
	 * @return Either suitable station for trade route on top of this(route)
	 *  or null.
	 */
	function GetStartStation(level)
	{
		foreach (dummy_id, s in this.start_node.stations.GetClassStations("RoadStation")) {
			if (s.station_level == level && s.CanSupply(this.cargo_id)) {
				if (s.CanSupplyTerminalBeExpanded(this.cargo_id)) return s;
			}
		}
		return null;
	}

	/**
	 * Get list with stations good enough to be source of trade organized on
	 *  top of this route.
	 * @param level Desired level of stations.
	 * @return Table with pairs [StationID - RoadStation object].
	 */
	function GetStartStations(level)
	{
		local result = {};
		foreach (dummy_id, s in this.start_node.stations.GetClassStations("RoadStation")) {
			if (s.station_level == level && s.CanSupply(this.cargo_id)) {
				result[s.GetStationID()] <- s;
			}
		}

		return result;
	}

	/**
	 * Get suitable station (if any) near the route end node.
	 * @param level Desired station level.
	 * @return Either suitable station or null.
	 */
	function GetEndStation(level)
	{
		local stations = this.end_node.stations.GetClassStations("RoadStation");
		local type_id = this.end_node.type_id;
		foreach (dummy_id, s in stations) {
			if (s.station_level != level && type_id != NodeTypeID.NT_TRANSIT_DROP) {
				continue;
			}
			if (this.one_way) {
				if (s.CanAccept(this.cargo_id)) {
					//if (s.GetCargoRelatedVehiclesCount(this.cargo_id) == 0) return s;
					if (s.CanAcceptTerminalBeExpanded(cargo_id)) return s;
					//return s;
				}
				continue;
			}
			if (s.CanSupply(this.cargo_id)) {
				//if (s.GetCargoRelatedVehiclesCount(this.cargo_id) == 0) return s;
				if (s.CanSupplyTerminalBeExpanded(this.cargo_id)) return s;
			}
		}
		return null;
	}

	/**
	 * Get list with stations good enough to be destination of trade organized
	 *  on top of this route.
	 * @param level Desired level of stations.
	 * @return Table with pairs [StationID - RoadStation object].
	 */
	function GetEndStations(level)
	{
		local result = {};
		local stations = this.end_node.stations.GetClassStations("RoadStation");
		local type_id = this.end_node.type_id;
		foreach (dummy_id, s in stations) {
			if (s.station_level != level && type_id != NodeTypeID.NT_TRANSIT_DROP) continue;

			if (this.one_way) {
				if (s.CanAccept(this.cargo_id)) result[s.GetStationID()] <- s;
			} else {
				if (s.CanSupply(this.cargo_id)) result[s.GetStationID()] <- s;
			}
		}

		return result;
	}

/* private */
	/** Price of the best engine. */
	best_engine_price = 0;

	/** Object that describes the state of road for this route. */
	connection = null;

	/** Estimated vehicles. */
	ev = null;

	/** Income for delivery of one cargo unit. */
	normal_cargo_income = 0;

	/** Transport capacity of one route vehicle (cargo amount per month). */
	transport_rate = null;

	/** Estimated month profit from one vehicle. */
	vehicle_profit = null;

	/** The max level of this route. */
	max_level = 0;
}
