/**
 * Class that represent road related AI settings.
 */
class RoadSettings extends Terron_Settings
{
/* public */
	static function GetClassName()
	{
		return "RoadSettings";
	}

	static function Restore(memento)
	{
		local restored = RoadSettings();
		restored.max_route_length = memento.max_route_length;
		restored.min_route_length = memento.min_route_length;
		restored.max_vehicles = memento.max_vehicles;
		return restored;
	}

	/** Avoid towns with lesser population */
	static lowest_population_limit_for_towns = 450;

	/** Defines max allowed length for cargo transfer routes. */
	static max_transit_range = 64;

	/** When "true", AI will build only roads with accurately calculated cost */
	accurate_construction = true;

	/**
	 * Indicates that one of non-road AI parts requires trucks lines to be
	 *  initialized. Needed to keep road related stuff creation unified (simple
	 *   and in one place).
	 * @note This should be set to 'true'(if needed) before Corporation object
	 *  creation.
	 */
	dummy_truck_nodes_initialization_required = false;

	/** Defines max allowed length for road routes */
	</ must_save = true />
	max_route_length = null;

	/** Defines min allowed length for road routes */
	</ must_save = true />
	min_route_length = null;

	/** The amount of road vehicles when AI must stop road expansion */
	</ must_save = true />
	max_vehicles = null;

	/** The percentage of road vehicles reserved for transit */
	reserved_for_transit = 0.0;

	/**
	 * Road "development"
	 *  - mean [current road vehicles count] / [max road vehicles allowed] -
	 *  considered enough to generate stable and reliable cash flow.
	 */
	safe_development_percentage = null;

/* protected */
	/**
	 * Set road settings up.
	 */
	constructor()
	{
		::Terron_Settings.constructor();
		this.max_route_length = AIController.GetSetting("max_road_route_length");
		if (this.max_route_length == -1) this.max_route_length = 70;

		this.min_route_length = (10 + this.max_route_length / 4).tointeger();

		if (AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_ROAD)) {
			this.max_vehicles = 0;
			return;
		}
		this.max_vehicles = AIGameSettings.IsValid("vehicle.max_roadveh") ?
			13 * AIGameSettings.GetValue("vehicle.max_roadveh") / 20 : 200;
		/* Over 1500 vehicles(per one AI instance) is REALLY not so good idea */
		// even 1000 is too much i think
		this.max_vehicles = min(1500, this.max_vehicles);
		if (AIController.GetSetting("BusModule") >= 2) {
			if (AIController.GetSetting("PlaneModule") > 0) {
				this.max_vehicles += this.max_vehicles / 10;
				this.reserved_for_transit += 0.25;
			}
			if (AIController.GetSetting("ShipModule") > 0) {
				this.max_vehicles += this.max_vehicles / 10;
				this.reserved_for_transit += 0.15;
			}
		}

		/* 100 vehicles is more than enough to consider company safe */
		this.safe_development_percentage = 100.0 / this.max_vehicles;
		if (this.safe_development_percentage < 0.1) {
			this.safe_development_percentage = 0.1;
		} else if (this.safe_development_percentage > 0.9) {
			// just if someone will decide to play with kinda 100 vehicles max
			this.safe_development_percentage = 0.9;
		}

		this.road_station_cost = Property(this.doGetStationCost.bindenv(this), 9 * GameTime.MONTH);
		this.road_station_cost.SetName("Get base road station cost");

		/* Calculate road cost multiplier - should depend on game settings... */

		this.road_cost_multiplier = 1.3;
		/*... amount of water makes huge impact because of high bridges cost */
		local water = AIGameSettings.IsValid(this.water_setting) ?
			AIGameSettings.GetValue(this.water_setting) : 2;
		switch (water) {
			case 0 : this.road_cost_multiplier += 0.2; break;
			case 1 : this.road_cost_multiplier += 0.4; break;
			case 2 : this.road_cost_multiplier += 1; break;
			case 3 : this.road_cost_multiplier += 1.8; break;
			default : this.road_cost_multiplier += 0.8;
		}

		/*
		 * ... hills mean tunnels, slope handling and not straight roads =>
		 *  more chances to screw something.
		 */
		local hills = AIGameSettings.IsValid(this.hills_setting) ?
			AIGameSettings.GetValue(this.hills_setting) : 2;
		switch (hills) {
			case 0 : this.road_cost_multiplier += 0.1; break;
			case 1 : this.road_cost_multiplier += 0.2; break;
			case 2 : this.road_cost_multiplier += 0.45; break;
			case 3 : this.road_cost_multiplier += 0.85; break;
			default : this.road_cost_multiplier += 0.4;
		}

		/* ... and roughness adds too */
		local rough = AIGameSettings.IsValid(this.smoothness) ?
			AIGameSettings.GetValue(this.smoothness) : 2;
		switch (rough) {
			case 0 : this.road_cost_multiplier += 0.05; break;
			case 1 : this.road_cost_multiplier += 0.15; break;
			case 2 : this.road_cost_multiplier += 0.3; break;
			case 3 : this.road_cost_multiplier += 0.45; break;
			default : this.road_cost_multiplier += 0.2;
		}

		/*local diff = AIGameSettings.IsValid(this.variety) ?
			AIGameSettings.GetValue(this.variety) : 0;
		switch (diff) {
			case 0 : break;
			case 1 :
			case 2 :
				if (this.road_cost_multiplier > 2.0) this.road_cost_multiplier = 2.0;
				break;
			case 3 :
				this.road_cost_multiplier = 0.7 * this.road_cost_multiplier;
				break;
			default : this.road_cost_multiplier = 0.85 * this.road_cost_multiplier;
		}
		if (this.road_cost_multiplier < 1.2) this.road_cost_multiplier = 1.2;*/
	}

	/**
	 * Get the basebrice of station building.
	 * @return Cost of new road station. 
	 */
	function GetRoadStationCost()
	{
		return road_station_cost.Get();
	}

	/**
	 * Get estimated cost of a road AB, where A and B are given tiles.
	 * @param t1 Tile ID, consider possible road start tile.
	 * @param t2 Tile ID, consider possible road finish tile.
	 * @return Estimated cost.
	 */
	function GetRoadConnectionCost(t1, t2)
	{
		local base = AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD);
		local estimate_length = 6 * AIMap.DistanceManhattan(t1, t2) / 5;
		return base * this.road_cost_multiplier * estimate_length;
	}

	/**
	 * Get estimated cost of a road with the given length.
	 * @param road_length The length of the road.
	 * @return Estimated cost.
	 */
	function GetRoadCost(road_length)
	{
		local base = AIRoad.GetBuildCost(AIRoad.ROADTYPE_ROAD, AIRoad.BT_ROAD);
		local estimate_length = 6 * road_length / 5;
		return base * this.road_cost_multiplier * estimate_length;
	}

/* private */
	/** Road station cost lies here. */
	road_station_cost = null;

	/** 'Sea level' game setting string */
	water_setting = "difficulty.quantity.sea_lakes";

	/** 'Terrain type' game setting string */
	hills_setting = "difficulty.terrain_type";

	/** 'Smoothness' game setting string */
	smoothness = "game_creation.tgen_smoothness";

	/** 'Variety distribution' game setting string */
	variety = "game_creation.variety";

	/** Helper to "guess" roads cost (without pathfinder call) */
	road_cost_multiplier = 1;

	/**
	 * Calculate the baseprice of station building.
	 * @return Cost of new road station.
	 */
	function doGetStationCost()
	{
		local r_type = AIRoad.ROADTYPE_ROAD;
		local c_bus = AIRoad.GetBuildCost(r_type, AIRoad.BT_BUS_STOP);
		local c_truck = AIRoad.GetBuildCost(r_type, AIRoad.BT_TRUCK_STOP);
		local c_road = AIRoad.GetBuildCost(r_type, AIRoad.BT_ROAD);
		return 2 * max(c_bus, c_truck) + 10 * c_road;
	}
}

Terron_ClassTable.RegisterClass(RoadSettings);
