/*
 * I see following classifiction for the road connections:
 * 1.a - Accurate road cost unknown
 * 1.b - Accurate road cost calculated
 * 2.a - Road does not exist
 * 2.b - Road exist(built)
 * And the 4 states follownig from:
 * 1.a + 2.a - cost unknown and road not built,
 * 1.a + 2.b - cost unknown and road exist - NONSENSE,dummy state(cost is zero)
 * 1.b + 2.a - cost calculated and road not built,
 * 1.b + 2.b - Road built and cost is zero.
 * Without dummy [1.a + 2.b] this states listed in the enum below. 
 */
enum ConnectionState
{
	CS_ACCURATE_COST_UNKNOWN
	CS_ACCURATE_COST_CALCULATED
	CS_ROAD_BUILT
	CS_END
}

/**
 * Class that describes road connection between two nodes.<p>
 * Provides accurate handling of the road routes cost.<p>
 * Because at game start accurate cost for roads unknown I see 2 alternatives:<p>
 * A) Forget it and use estimate value.<p>
 *    + this is fast<p>
 *    - road routes management becomes pretty random<p>
 * B) Calculate accurate cost for each route.<p>
 *    + AI knows how much it must spend for each route<p>
 *    + AI can avoid bankruptcy at game start<p>
 *    + Additional information allows "smart" money spending:
 *       AI can make more profit for every $ invested.<p>
 *    - this is horrifying slow<p>
 * Neither A nor B is not good enough for me, so I've combined A + B to method C:<p>
 * C) Use estimated cost "c1" until we need to build a road. Then obtain
 *    accurate cost "c2". If c2 <= c1, then build those road, else make c1 = c2
 *    and do not build road (select new road to build).
 *    With adequate estimation function I see following pluses and minuses:<p>
 *    + AI knows how much it must spend for each route<p>
 *    + AI can avoid bankruptcy at game start<p>
 *    + AI always make "near optimal" decision when choose road to build<p>
 *    - Still there is slowdown, but this level I can tolerate.<p>
 *    - Still "near optimal" not optimal management.<p>
 * So, this class is the lowest layer to implement C method,
 * it includes/[provides access to] the road cost, and it is here
 * the mysterious phrase "make c1 = c2" must obtain sense.
 * Note: with FreeTime event this approach seems perfect now, fails only when
 * pathfinder gives wrong "accurate" cost.
 */
class RoadConnection extends Terron_SaveableObject
{
/* public */
	static function GetClassName()
	{
		return "RoadConnection";
	}

	static function Restore(memento)
	{
		local restored = RoadConnection(memento.from, memento.to);
		restored.cost = memento.cost;
		restored.state = memento.state;
		return restored;
	}

	/** Cost of the road represented by this connection */
	</ must_save = true />
	cost = null;

	/** State of this connection, from the ConnectionState enum */
	</ must_save = true />
	state = ConnectionState.CS_ACCURATE_COST_UNKNOWN;

	/**
	 * Creates new RoadConnection object.
	 * @param from ID of some object.
	 * @param to ID of some object too. 
	 */
	constructor(from, to)
	{
		::Terron_SaveableObject.constructor(null);
		if (from > to) {
			this.from = to;
			this.to = from;
		} else {
			this.from = from;
			this.to = to;
		}
	}

	function GetName()
	{
		return "Road connection from some #" + this.from + " to some #" + this.to;
	}

/* private */
	/** ID of the entity where road begins*/
	</ must_save = true />
	from = null;

	/** ID of the entity where road ends*/
	</ must_save = true />
	to = null;
}

Terron_ClassTable.RegisterClass(RoadConnection);
