/**
 * Class that containes/[provides access to] information about game roads.
 */
class RoadMap extends Terron_SaveableObject
{
/* public */
	static function GetClassName()
	{
		return "RoadMap";
	}

	static function Restore(memento)
	{
		local restored = RoadMap();
		foreach (dummy_id, connection in memento.connections) {
			restored.RegisterConnectionFast(connection);
		}
		restored.MakeMemento();
		return restored;
	}

	constructor()
	{
		::Terron_SaveableObject.constructor(null);
		this.roadmap = TableContainer.new("Road connections");
		this.MakeMemento();
	}

	function GetMemento()
	{ 
		return this.memento;
	}

	function GetName()
	{
		return "Some Roadmap";
	}

	/**
	 * Get the connection between the given entities.
	 * @param from ID of some object.
	 * @param to ID of some other object.
	 */
	function GetConnection(from, to)
	{
		if (from > to) return this.GetConnection(to, from);
		return (!(from in this.roadmap)) ? null :
			(to in this.roadmap[from] ? this.roadmap[from][to] : null);
	}

	/**
	 * Check if there is a road between given entities.
	 * @param from ID of some object.
	 * @param to ID of some other object.
	 */
	function HasRoad(from, to)
	{
		local z = this.GetConnection(from, to);
		return z == null ? false : z.state == ConnectionState.CS_ROAD_BUILT;
	}

	/**
	 * Add the given connection to the roadmap.
	 * @param new_connection The connection to register.
	 */
	function RegisterConnection(new_connection)
	{
		this.RegisterConnectionFast(new_connection);
		if (new_connection.state == ConnectionState.CS_ROAD_BUILT) {
			this.MakeMemento();
		}
	}

	/**
	 * Registers a road inside the road map.
	 * @param from ID of some object.
	 * @param to ID of some other object.
	 */
	function RegisterRoad(from, to)
	{
		if (from > to) return this.RegisterRoad(to, from);
		local old_connection = this.GetConnection(from, to);
		if (old_connection == null) {
			local new_connection = RoadConnection(from, to);
			new_connection.state = ConnectionState.CS_ROAD_BUILT;
			new_connection.cost = 0;
			this.RegisterConnectionFast(new_connection);
		} else {
			old_connection.state = ConnectionState.CS_ROAD_BUILT;
			old_connection.cost = 0;
		}
		this.MakeMemento();
	}

	/**
	 * Removes connection from the roadmap.
	 */
	function RemoveConnection(road_connection)
	{
		if (road_connection.from in this.roadmap) {
			delete (this.roadmap[road_connection.from])[road_connection.to];
			if (road_connection.state == ConnectionState.CS_ROAD_BUILT) {
				this.MakeMemento();
			}
		}
	}

	/**
	 * Removes any road from/to the the given object.
	 * @param id ID of the object to exclude from the roadmap.
	 */
	function RemoveObject(id)
	{
		if (id in this.roadmap) delete this.roadmap[id];
		foreach (from, from_connections in this.roadmap) {
			if (id in from_connections) delete from_connections[id];
		}
		this.MakeMemento();
	}

/* private */
	/** Container with road-describing connections. */
	roadmap = null;

	/** Table that holds object's current state(for faster AI save). */
	memento = null;

	/** Updates object's memento. */
	function MakeMemento()
	{
		this.memento = {connections = []};
		foreach (from, from_table in this.roadmap) {
			foreach (to, connection in from_table) {
				if (connection.state != ConnectionState.CS_ROAD_BUILT) continue;
				this.memento.connections.append(connection);
			}
		}

		Terron.UpdateMemento();
	}

	/**
	 * Add the given connection to the roadmap.
	 * @param new_connection The connection to register.
	 * @note This function does not update object's memento.
	 */
	function RegisterConnectionFast(new_connection)
	{
		local map = this.roadmap;
		local from = new_connection.from;
		if (!(from in map)) map[from] <- TableContainer.new("From " + from);
		map[from][new_connection.to] <- new_connection;
	}
}

Terron_ClassTable.RegisterClass(RoadMap);
