/**
 * Class that coordinates efforts in "routes stealing".
 */
class RoadsStealingSection extends AbstractTransportSystem
{
/* public */
	static function GetClassName()
	{
		return "RoadsStealingSection";
	}

	/**
	 * RoadsStealingSection constructor.
	 * @param transport_class_name Only routes between transport nodes of this
	 *  given transport class would be stolen.
	 * @param node_types Stealing will be restricted to this node types. 
	 * @param route_validation_callback Function to check if newly found
	 *  routes are good enough to steal.
	 */
	constructor(transport_class_name, node_types, route_validation_callback)
	{
		::AbstractTransportSystem.constructor();

		this.candidate_to_steal_routes = TableContainer.new_custom("Candidates", ToStealRoutes());
		this.confirmed_can_steal_routes = TableContainer.new("Can steal");
		this.route_validation_callback = route_validation_callback;

		local name = "Steal road route action";
		local strategy = RoadsStealingSection_LocalBuildStrategy(this);
		local t = GameTime.MONTH / 2;
		this.steal_trade_action = UniversalAction(name, strategy, 6 * t, t);

		FreeTimeEvent.AddListener(this);

		OpponentsActivityScanTask(this, transport_class_name, node_types);
	}

	function GetName()
	{
		return "Roads Stealing Section";
	}

	/**
	 * Receive a new portion of routes to steal.
	 * "Background" task (instance of OpponentsActivityScanTask) will use it.
	 * @param routes Table with pairs [route ID, route].
	 * @return None.
	 */
	function UploadNewCandidatesToSteal(routes)
	{
		foreach (id, route in routes) {
			if (this.route_validation_callback(route)) {
				this.candidate_to_steal_routes[id] <- route;
			}
		}
	}

/* protected */
	function doGetBestAction()
	{
		local n = this.confirmed_can_steal_routes.len();
		return n > 0 ? this.steal_trade_action : null;
	}

/* private */
	/**
	 * Table with routes that can or can not be allowed to steal.
	 */
	candidate_to_steal_routes = null;

	/**
	 * Table with routes that certainly allowed be stealed.
	 */
	confirmed_can_steal_routes = null;

	/**
	 * Action tha build trade.
	 */
	steal_trade_action = null;

	/**
	 * Callback function. Defines if route is allowed/good enough to be stealed.
	 */
	route_validation_callback = null;

	/**
	 * Find one best route from candidates to steal.
	 */
	function GetBestCandidateToSteal()
	{
		return (candidate_to_steal_routes.len() == 0) ? null :
			this.candidate_to_steal_routes.GetBestItem().best_item;
	}

/* public */
	/**
	 * On free time check if candidates to steal can be stealed.
	 */
	function OnFreeTime(param)
	{
		/*
		 * We'll start pf to check if road for "candidate" route exist, and
		 *  if it's built => promote "candidate" to "target to steal".
		 */
		/*
		 * While it's free time, we still should make things fast, so
		 *  select only one route from candidates(since pf eats a lot of time).
		 */
		local best = this.GetBestCandidateToSteal();
		if (best == null) return;

		/*
		 * Candidate selected. It's either worth stealing or not.
		 *  But in any case we need to check it only once,
		 *  so remove this 'best' from candidates (and further rechecks).
		 */
		this.candidate_to_steal_routes.RemoveItem(best.GetID());

		/*
		 * First check if a record about corresponding road already exist
		 *  in our road register (nodes roadmap).
		 * If so => we don't need to waste time on pf.
		 */
		local map = RoadDepartment.Get().GetNodesRoadMap();
		local s = best.GetStart().location;
		local e = best.GetEnd().location;
		if (map.HasRoad(s, e)) {
			/* Promote best to target to steal */
			this.confirmed_can_steal_routes[best.GetID()] <- best;
			return;
		}

		/* If there is no record about road => try to find road with pf. */
		//local pf = RoadUtils.GetAccuratePF();
		local pf = SimplePF_Adapter();
		if (pf.FindRouteBetweenRects(s, e, 9, AIRoad.ROADTYPE_ROAD)) {
			/* Save informaion about founded road */
			map.RegisterRoad(s, e);

			/*
			 * ...and see if there are other nodes nearby,
			 *  so we can use them with new road
			 */
			s = best.GetStart();
			e = best.GetEnd();
			local n1 = s.GetName(), n2 = e.GetName();
			CodeUtils.Log("Road is found between " + n1 + " and " + n2 , 1);
			RoadDepartment.Get().UpdateConnectionsRoadMapWhenRoadBuilt(s, e);

			/* Promote best to "target to steal" */
			this.confirmed_can_steal_routes[best.GetID()] <- best;
			this.steal_trade_action.ForceUpdate();
		}
	}
}

/**
 * Route items context for TableContainer.
 */
class RoadsStealingSection.ToStealRoutes extends TableContainer.ItemContext
{
/* public */
	constructor()
	{
		::TableContainer.ItemContext.constructor();
		this.default_build_strategy = DefaultRoadConstructionStrategy();
	}

/* protected */
	function GetObjectRank(object)
	{
		return this.default_build_strategy.RankRoadRoute(object);
	}

/* private */
	/** private */
	default_build_strategy = null;
}
