/**
 * Class that describes universal road terminal.
 */
class RoadStation_LoadingTerminal extends Station_LoadingTerminal
{
/* public */
	static function GetClassName()
	{
		return "RoadStation_LoadingTerminal";
	}

	static function Restore(memento)
	{
		local restored = RoadStation_LoadingTerminal(
			memento.station_id, memento.cargo_id, memento.tiles
		);
		restored.depot.location = memento.depot_location;

		return restored;
	}

	/** Core information about the terminal. */
	core = null;

	depot = null;

	/** The terminal's station type. */
	station_type = null;

	constructor(station_id, cargo_id, tiles)
	{
		this.station_type = RoadUtils.GetStationType(cargo_id);

		::Station_LoadingTerminal.constructor(station_id, cargo_id);

		this.core = RoadTerminalCore(tiles);
		this.depot = RoadStationDepot(station_id, this.core);

		if (this.core.drive_through) this.base_frequency = 0.23;
	}

	function GetMemento()
	{
		local memento = {};
		memento.station_id <- this.station_id;
		memento.cargo_id <- this.cargo_id;
		memento.tiles <- this.core.tiles;
		memento.depot_location <- this.depot.location;
		return memento;
	}

	/**
	 * Checks whether the terminal can be expanded.
	 * @param cargo_id ID of the cargo to sevice with new road stops.
	 * @return True if and only if the terminal can be expanded with
	 *  additional bus/truck stops.
	 */
	function CanBeExpanded(cargo_id)
	{
		return this.core.tiles.len() > this.core.max_upgrades_allowed ? false :
			(cargo_id == this.cargo_id && this.core.failed_upgrades < 6);
	}

/* protected */
	/**
	 * Get the amount of own road vehicles heading to this terminal.
	 */
	function GetLoadingVehicles()
	{
		if (this.core.tiles.len() == 0) return 0;

		local c = this.cargo_id;
		local v_list = AIVehicleList_Station(this.station_id);

		v_list.Valuate(AIVehicle.GetCurrentSpeed);
		v_list.KeepBelowValue(20);

		/* Remove vehicles that is not going to this station */
		local f = function(v) {
			return AIOrder.GetOrderDestination(v, AIOrder.ResolveOrderPosition(v, AIOrder.ORDER_CURRENT));
		}
		v_list.Valuate(f);
		v_list.KeepValue(this.core.tiles[0]);

		/* Remove vehicles far enough from this station */
 		f = function(v, t) { return AIMap.DistanceManhattan(AIVehicle.GetLocation(v), t);}
		v_list.Valuate(f, this.core.tiles[0]);
		v_list.RemoveAboveValue(6);

		v_list.Valuate(AIVehicle.GetCapacity, c);
		v_list.RemoveValue(0);

		v_list.Valuate(AIVehicle.GetState);

		local free_station_positions = this.core.drive_through ?
			3 * this.core.tiles.len() / 2 : 2 * this.core.tiles.len();

		local stuck_vehicles = 0;
		foreach (v, state in v_list) {
			if (state == AIVehicle.VS_AT_STATION) {
				free_station_positions--;
				stuck_vehicles++
			}
			if (state == AIVehicle.VS_BROKEN) stuck_vehicles++;
			if (state == AIVehicle.VS_RUNNING) stuck_vehicles++;
		}
		return max(0, stuck_vehicles - free_station_positions);
	}

	function GetMaxLoadingVehicles()
	{
		local n = this.core.tiles.len();
		return n == 0 ? 0 : this.core.drive_through ? 2 + 2 * n / 5 : 3 + n / 2;
	}

	function GetMaxFrequency()
	{
		return this.core.tiles.len() * this.base_frequency;
	}

/* private */
	/** Max allowed vehicles arrival frequency for the terminal. */
	base_frequency = 0.2;
}

function RoadStation_LoadingTerminal::EstimateVehiclesNeeds(transport_per_vehicle)
{
	/* No vehicles => buy one */
	if (this.arrival_frequency_map.len() == 0) return 1;

	local now = this.GetCurrentState();
	local last = (now.date - this.past_state.date) > GameTime.MONTH ?
		this.past_state : this.long_past_state;

	local dt = now.date - last.date;
	local dtm = dt.tofloat() / GameTime.MONTH;
	local rating_gap = max(this.target_rating - now.rating, 0);
	local stock = now.stock; 
	if (dtm < 1) stock -= this.past_state.transport_amount_change;

	local loading = this.GetLoadingVehicles();
	if (stock <= 10 && loading <= 1) {
		/*
	 	* For recently built stations we risk and buy vehicles even when
	 	*  stock is low.
	 	*/
		if (now.date - this.build_date < 1.5 * GameTime.YEAR) return 1;
	}

	local max_vehicles = this.GetMaxLoadingVehicles();
	if (stock < transport_per_vehicle / 2 && rating_gap == 0 && loading > max_vehicles) {
		//CodeUtils.Log("Too many loading vehicles at " + this.GetName(), 2);
		if (now.date - this.past_state.date > GameTime.MONTH) {
			this.long_past_state = clone this.past_state;
			this.past_state = now;
		}
		return max_vehicles - loading;
	}

	if ((stock - last.stock / dtm) < transport_per_vehicle) {
		if (now.date - this.past_state.date > GameTime.MONTH) {
			this.long_past_state = clone this.past_state;
			this.past_state = now;
		}
		return 0;
	}

	/* Produced stock can dissapear at stations with low rating */
	stock += (stock + 1) * dtm * rating_gap / this.target_rating;
	if (last.stock == 0) stock = max(0, stock.tointeger() - 50);

	/* Simplest way to predict future step */
	local result = (stock - last.stock) / (dtm  * transport_per_vehicle);
	/* Stock reduced, but rating is low or falling => wait, do not sell */
	if (result < 0) {
		if (now.rating <= this.target_rating / 2) {
			result = (stock > transport_per_vehicle / 2) ? 1 : 0;
		} else if (now.rating <= (4 * this.target_rating / 5) || now.rating < last.rating - 2) {
			result = 0;
		} else {
			// maybe it'll prevent buy - sell - buy - sell - e.t.c... situation
			result += 1;
		}
	}

	/* Rating is low => buy, result says "almost buy" => buy */
	if (now.rating <= (4 * this.target_rating / 5)) result++;
	/* Huge stocks => more vehicles needed */
	if (stock >= 1200) result++;
	/* Tweak selling desire, sell only when stocks drop is large enough */
	if (result < 0) result = (result <= -4) ? result / 2 : -1;

	local n = result.tointeger();
	this.past_state.transport_amount_change += n * transport_per_vehicle;
	if (now.date - this.past_state.date > GameTime.MONTH) {
		this.long_past_state = clone this.past_state;
		this.past_state = now;
	}
	//CodeUtils.Log(this.GetName() + " needs " + n + " vehicles", 1);
	return n;
}

Terron_ClassTable.RegisterClass(RoadStation_LoadingTerminal);
