/**
 * Class that represents bus/truck stops.
 */
class RoadStation extends Station
{
/* public */
	static function GetClassName()
	{
		return "RoadStation";
	}

	static function Restore(memento)
	{
		return RoadStation(
			memento.station_id,
			memento.station_level,
			memento.node_id,
			memento.node_type,
			memento.station_plan
		); 
	}

	/**
	 * "Station level" - just number.
	 * But it's assumed that only stations with equal level can be connected.
	 */
	</ must_save = true />
	station_level = null;

	</ must_save = true />
	station_plan = null;

	/**
	 * Creates a road station.
	 * @param station_id In-game station ID.
	 * @param station_level Special road section entity - trade route level.
	 * @param node_id ID of a node where the station is built.
	 * @param node_type_id Type ID of the node where the station is built.
	 */
	constructor(station_id, station_level, node_id, node_type_id, station_plan)
	{
		::Station.constructor(station_id, node_id, node_type_id);

		this.station_level = station_level;
		this.station_plan = station_plan;

		foreach (dummy, terminal in station_plan.GetLoadingTerminals()) {
			this.RegisterTerminal(terminal);
		}
		foreach (dummy, terminal in station_plan.GetCargoDropTerminals()) {
			this.RegisterTerminal(terminal);
		}

		if (!this.is_town_station && node_type_id != NodeTypeID.NT_TRANSIT_DROP) {
			Terron_Event.IndustryClosed.AddListener(this);
		}
	}

	function Close()
	{
		local id = this.GetStationID();
		if (id != -1) RoadDepartment.Get().GetStationsRoadMap().RemoveObject(id);
		return ::Station.Close();
	}

	function GetEntrances()
	{
		local result = [];
		foreach (dummy, terminal in this.station_plan.GetLoadingTerminals()) {
			result.extend(terminal.core.GetEntrances());
		}

		return result;
	}

	function GetExits()
	{
		local result = [];
		local cargo_drop_terminals = this.station_plan.GetCargoDropTerminals();
		foreach (dummy, terminal in cargo_drop_terminals) {
			result.extend(terminal.core.GetEntrances());
		}

		if (result.len() == 0) result.extend(this.GetEntrances());

		return result;
	}

	function CheckDepots()
	{
		foreach (dummy, terminal in this.station_plan.GetLoadingTerminals()) {
			if (!terminal.depot.Check()) {
				CodeUtils.Log("Invalid depot near: " + terminal.GetName(), 2);
				return false;
			}
		}

		foreach (dummy, terminal in this.station_plan.GetCargoDropTerminals()) {
			if (!terminal.depot.Check()) {
				CodeUtils.Log("Invalid depot near: " + terminal.GetName(), 2);
				return false;
			}
		}

		return true;
	}

	/**
	 * Checks whether the station's terminal that provides cargo for vehicles
	 *  can be expanded.
	 * @param cargo_id ID of the cargo to sevice with new road stops.
	 * @return True if and only if the station's "supply" terminal can be
	 *  expanded with additional bus/truck stops.
	 */
	function CanSupplyTerminalBeExpanded(cargo_id)
	{
		foreach (dummy, terminal in this.station_plan.GetLoadingTerminals()) {
			if (terminal.CanBeExpanded(cargo_id)) return true;
		}

		return false;
	}

	/**
	 * Checks whether the station's cargo drop terminal can be expanded.
	 * @param cargo_id ID of the cargo to sevice with new road stops.
	 * @return True if and only if the station's cargo drop terminal can be
	 *  expanded with additional bus/truck stops.
	 */
	function CanAcceptTerminalBeExpanded(cargo_id)
	{
		foreach (dummy, terminal in this.station_plan.GetCargoDropTerminals()) {
			if (terminal.CanBeExpanded(cargo_id)) return true;
		}

		foreach (dummy, terminal in this.station_plan.GetLoadingTerminals()) {
			if (terminal.CanBeExpanded(cargo_id)) return true;
		}

		return false;
	}

/* public */
	/**
	 * Close the station if it was builded near the closing industry.
	 * @param i_id ID of the closing industry.
	 */
	function OnIndustryClose(i_id)
	{
		local node = this.GetStationNode();
		if (node == null) {
			if (this.node_type < NodeTypeID.NT_TRANSIT_START) {
				if (AIIndustry.IsValidIndustry(this.node_id)) return;
			}
			CorporationUtils.CloseSystem(this);
			return;
		}

		if (!("GetIndustryID" in node)) return;
		if (node.GetIndustryID() == i_id) CorporationUtils.CloseSystem(this);
	} 
}

Terron_ClassTable.RegisterClass(RoadStation);
