/**
 * Class that describes road-specific terminal features.
 */
class RoadTerminalCore extends Terron_SaveableObject
{
/* public */
	/** Road terminal tiles. */
	tiles = null;

	/** Flag showing if this is "drive through" terminal. */
	drive_through = false;

	/**
	 * The amount of failed attempts to upgrade station.
	 * When become big enough we must stop further upgrade attempts.
	 */
	failed_upgrades = 0;

	/**
	 * Amount of additional road stops allowed.
	 */
	max_upgrades_allowed = 3;

	/**
	 * RoadTerminalCore class constructor.
	 * @param tiles Array with road station tiles representing terminal;
	 */
	constructor(tiles)
	{
		::Terron_SaveableObject.constructor(null)
		this.tiles = [];
		if (typeof(tiles) == "array") {
			foreach (dummy_id, t in tiles) {
				this.tiles.append(t);
			}
		} else {
			foreach (t, dummy_value in tiles) {
				this.tiles.append(t);
			}
		}

		AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);
		foreach (dummy_id, t in this.tiles) {
			if (!AIRoad.IsDriveThroughRoadStationTile(t)) return;
		}
		if (this.tiles.len() > 0) this.drive_through = true;
	}

	/**
	 * Get entrances to the terminal.
	 * @return Array with front tiles of the terminal's station tiles;
	 */
	function GetEntrances()
	{
		local old_type = RoadUtils.SetRoadType(AIRoad.ROADTYPE_ROAD);

		local result = [];
		foreach (dummy_id, t in this.tiles) {
			local front = AIRoad.GetRoadStationFrontTile(t);

			if (AIRoad.AreRoadTilesConnected(t, front)
			&& !AIRoad.IsRoadStationTile(front)) {
				result.append(front);
			} else if (AIRoad.IsDriveThroughRoadStationTile(t)) {
				local back = AIRoad.GetDriveThroughBackTile(t);
				if (AIRoad.AreRoadTilesConnected(t, back)
				&& !AIRoad.IsRoadStationTile(back)) {
					result.append(back);
				}
			}
		}

		AIRoad.SetCurrentRoadType(old_type);

		return result;
	}

	/**
	 * Get exits from the terminal.
	 * @return Array with back tiles of the terminal station tiles;
	 */
	function GetExits()
	{
		local old_type = RoadUtils.SetRoadType(AIRoad.ROADTYPE_ROAD);

		local result = [];
		foreach (dummy_id, t in this.tiles) {
			if (AIRoad.IsDriveThroughRoadStationTile(t)) {
				local back = AIRoad.GetDriveThroughBackTile(t);
				if (AIRoad.AreRoadTilesConnected(t, back)
				&& !AIRoad.IsRoadStationTile(back)) {
					result.append(back);
					continue;
				}
			}

			local front = AIRoad.GetRoadStationFrontTile(t);
			if (AIRoad.IsRoadStationTile(front)) continue;
			if (AIRoad.AreRoadTilesConnected(t, front)) result.append(front);
		}

		AIRoad.SetCurrentRoadType(old_type);

		return result;
	}
}
