/**
 * Class that handles organization.
 */
class RoadNetworksCoordinator extends AbstractTransportSystem
{
/* public */
	static function GetClassName()
	{
		return "RoadNetworksCoordinator";
	}

	static function Restore(memento)
	{
		return RoadNetworksCoordinator(false);
	}

	/**
	 * RoadNetworksCoordinator constructor.
	 * @param accurate_start Defines whether module should maximize profit
	 *  in starting period to be able to relatively safely expand lately.
	 * This means that net construction will be disabled for a while, and AI
	 *  will build several starting routes in default "greedy" mode.
	 */
	constructor(accurate_start)
	{
		::AbstractTransportSystem.constructor();

		this.accurate = accurate_start;
		this.dumb_module = DumbTruckControlSection(false);
		this.ordered_town_netwotks = {};
		this.valued_town_effects = {
			[AICargo.TE_FOOD] = 1,
			[AICargo.TE_GOODS] = 1,
			[AICargo.TE_WATER] = 1,
		};

		RoadNetsManagementTask(this, GameTime.MONTH);

		RegionScanFinishedEvent.AddListener(this);
		Terron_Event.IndustryClosed.AddListener(this);
	}

	function GetName()
	{
		return "RoadNetworksCoordinator instance";
	}

	/**
	 * Create road networks for the map region.
	 * @param region Map region ID.
	 */
	function OnRegionScanFinished(region)
	{
		local t = AIController.GetTick();
		local pop = MagicNumbers.min_town_pop_for_net_root;

		local truck_hubs = GlobalTransportMap.Get().structured_transport_nodes[TruckHub.GetClassName()];
		/* Each town can be a root for road network */
		foreach (dummy_id, town_hub in truck_hubs[NodeTypeID.NT_RAW_TOWN]) {
			local town = town_hub.node;
			if (town.GetRegion() != region || town.GetPopulation() < pop) continue;

			this.ordered_town_netwotks[town.GetTownID()] <- [
				{hub = town_hub, net = null}
			];
		}

		local types = TransportSchema.Get().node_types;
		foreach (type_id, type_nodes in truck_hubs) {
			if (!this.IsTownServiceIndustryType(type_id, types)) continue;

			/*
			 * If town has service industry(e.g. one of "stores", petrol
			 *  station, water tower) => do not feed the town. Feed that
			 *  industry instead.
			 */
			foreach (dummy_node_id, truck_hub in type_nodes) {
				if (truck_hub.node.GetRegion() != region) continue;

				local t = truck_hub.node.location;
				local town_id = AITile.GetClosestTown(t);
				if (AITile.IsWithinTownInfluence(t, town_id) && town_id in this.ordered_town_netwotks) {
					this.ordered_town_netwotks[town_id].append({hub = truck_hub, net = null});
				}
			}
		}

		/* for test purposes - keeps only industries */
		/*foreach (dummy_id, town_hub in truck_hubs[NodeTypeID.NT_RAW_TOWN]) {
			local town = town_hub.node;
			local town_id = town.GetTownID();
			if (town_id in this.ordered_town_netwotks) {
				if (this.ordered_town_netwotks[town_id].len() == 1) {
					delete this.ordered_town_netwotks[town_id];
				} else {
					this.ordered_town_netwotks[town_id].remove(0);
				}
			}
		}*/

		CodeUtils.Log("ticks: " + (AIController.GetTick() - t), 2);
	}

	/**
	 * Handle industry collape.
	 * @param industry_id Just closed industry ID.
	 */
	function OnIndustryClose(industry_id)
	{
		foreach (town_id, info_stack in this.ordered_town_netwotks) {
			if (info_stack.len() == 1) continue;

			local bad_item = -1;
			for (local i = info_stack.len() - 1; i >= 0; i--) {
				if (info_stack[i].hub.node.IsTown()) continue;
				if (info_stack[i].hub.node.GetIndustryID() == industry_id) {
					bad_item = i;
					break;
				}
			}

			if (bad_item != -1) {
				local bad_net = info_stack[bad_item].net;
				bad_net.Clear();
				this.GetSubsystems().RemoveItem(bad_net.GetID());

				info_stack.remove(bad_item);
				return;
			}
		}
	}

	function GetBestAction()
	{
		if (this.accurate && RoadSettings.Get().accurate_construction) {
			return this.dumb_module.GetBestAction();
		}

		return this.GetSubsystems().len() > 0 ?
			::AbstractTransportSystem.GetBestAction() :
			this.dumb_module.GetBestAction();
	}

/* private */
	accurate = false;

	dumb_module = null;

	ordered_town_netwotks = null;

	valued_town_effects = null;

	function IsTownServiceIndustryType(industry_type_id, industry_types)
	{
		local type = industry_types[industry_type_id];
		if (
			type.raw_consumption == false ||
			type.type_name.slice(0,4) == "FAKE" ||
			industry_type_id == NodeTypeID.NT_RAW_TOWN ||
			industry_type_id == NodeTypeID.NT_PAX_TOWN
		) return false;

		foreach (c, dummy in type.consumption) {
			if (AICargo.GetTownEffect(c) in this.valued_town_effects) return true;
		}

		return false;
	}
}
