/**
 * Class that implements road transit routes construction logic.
 */
class TransitRoutesConstructionStrategy extends DumbTruckRoutesConstructionStrategy
{
/* public */
	/**
	 * Table with transport nodes which need to be supplied by transit routes.
	 */
	to_feed_transport_nodes = null;

	/**
	 * Constructor for TransitRoutesConstructionStrategy class.
	 * @param to_feed_transport_nodes Reference to container with
	 *  transport nodes to feed.
	 */
	constructor(to_feed_transport_nodes)
	{
		::DumbTruckRoutesConstructionStrategy.constructor();

		this.to_feed_transport_nodes = to_feed_transport_nodes;
		this.routes_to_build = {};
	}

	function SelectObjectToBuild()
	{
		this.EnlistGoodTransitRoutes();

		foreach (dummy_id, info in this.routes_to_build) {
			if (info.build_attempts > 10 || info.is_finished || info.route.IsNefarious()) continue;
			if (info.route.IsCompletelyUpgraded()) {
				info.is_finished = true;
				continue;
			}

			return {priority = APriority.HALF, context = info.route};
		}

		return {priority = APriority.BAD, context = null};
	}

	function HandleResult(result, route)
	{
		local return_value = ::DumbTruckRoutesConstructionStrategy.HandleResult(result, route);

		local route_id = route.GetID();
		if (!(route_id in this.routes_to_build)) {
			this.routes_to_build[route_id] <- {
				build_attempts = 0,
				route = route,
				is_finished = false,
			};
		}

		this.routes_to_build[route_id].build_attempts++;
		if (route.IsCompletelyUpgraded()) {
			this.routes_to_build[route_id].is_finished = true;
		}

		return return_value;
	}

/* private */
	/**
	 * Date of the lact check
	 */
	last_routes_check_date = -1000;

	/**
	 * Table with information about transit routes to construct.
	 * Or to-do list in other words.
	 */
	routes_to_build = null;

	/**
	 * This function checks all available transit routes and puts "good" ones
	 *  into to-do list.
	 */
	function EnlistGoodTransitRoutes()
	{
		local current_date = AIDate.GetCurrentDate();
		if (current_date - this.last_routes_check_date < GameTime.MONTH) return;

		local good_routes = {};
		foreach (dummy_id, node in this.to_feed_transport_nodes) {
			if (!node.IsEnabled()) continue;

			/*
			 * Enable transit only when there is working trade route out from
			 *  the stockpile we must feed.
			 * Thus remember ids of cargo transported FROM the target and
			 *  allow only supplying of the same cargo ids to it.
			 */
			local c_ids_needed = {};
			foreach (dummy_link_id, link in node.links_out) {
				foreach (dummy_route_id, route in link.routes) {
					local c = route.GetCargoID();
					if (!(c in c_ids_needed)
						&& !route.IsNefarious()
						&& route.GetCurrentRouteLevel() > 0
					) {
						c_ids_needed[c] <- c;
					}
				}
			}
			if (c_ids_needed.len() == 0) continue;

			local good_colors = {
				[LinkColor.LC_YELLOW] = 1, [LinkColor.LC_GREEN] = 1,
			};
			foreach (dummy_link_id, link in node.links_in) {
				if (!(link.color in good_colors)) continue;
				foreach (dummy_route_id, route in link.routes) {
					if (route.GetCargoID() in c_ids_needed
						&& route.IsTransit()
						&& !route.IsNefarious()
						&& !route.IsCompletelyUpgraded()
					) {
						good_routes[route.GetID()] <- route;
					}
				}
			}
		}

		foreach (route_id, route in good_routes) {
			if (!(route_id in this.routes_to_build)) {
				this.routes_to_build[route_id] <- {
					build_attempts = 0,
					route = route,
					is_finished = false,
				};
			}
		}

		this.last_routes_check_date = current_date;
	}
}
