/**
 * Generic A* for path search between water tiles.
 * No channels building.
 * Tiles are either "water" or not.
 */
class WASTAR
{
/* public */
	/**
	 * Find water path between two water tiles.
	 * @param s First tile.
	 * @param e Second tile.
	 * @param max_ticks_to_go Time, in ticks, allowed for calculations.
	 * @return Array of tiles representing path, or null if failed.
	 */
	function FindPath(s, e, max_ticks_to_go)
	{
		/*
		 * If all tiles in direct line connecting s and e are water tiles
		 *  => no need to launch A* at all, just grab the line and we done here.
		 * Big(WHPASTAR) pf takes around 45% lesser ticks with it.
		 * It works so well only because most of the time we need to find path
		 *  between two small, close to each other, almost 100% water squares,
		 *  thus direct line almost always is the best way.
		 */
		local x1 = AIMap.GetTileX(s), y1 = AIMap.GetTileY(s);
		local x2 = AIMap.GetTileX(e), y2 = AIMap.GetTileY(e);
		local line = TileUtils.GetLine(x1, y1, x2, y2);
		for (local i = 0; i < line.len(); i++) {
			line[i] = AIMap.GetTileIndex(line[i].x, line[i].y);
			if (!AITile.IsWaterTile(line[i])) {
				line = null;
				break;
			}
		}
		if (line != null) return line;

		local ti = AIMap.GetTileIndex;
		local offsets = [ti(-1, 0), ti(0, 1), ti(1, 0), ti(0, -1)];

		/*
		 * Found path will be reversed, so in order to receive "right" path
		 *  we can swap "s" and "e" here, or call result_array.reverse() later.
		 * First method is better => it is choosen.
		 * SO don't be surpised that start_node.tile is "e"
		 */
		local start_node = {tile = e, distance_to_start = 0};
		start_node.parent_node <- start_node;

		local open_nodes = Terron_FibonacciHeap();
		open_nodes.Insert(start_node, AIMap.DistanceManhattan(s, e));

		local max_tick = AIController.GetTick() + max_ticks_to_go;
		for (local k = 0, closed_nodes = {}; !open_nodes.IsEmpty(); k = (k + 1) % Ticks.DAY) {
			local best_node = open_nodes.Pop();

			/* Best node is "s"(start and end swaped, remember) => path found */
			if (best_node.tile == s) return this.MakePath(best_node);
			if (best_node.tile in closed_nodes) continue;

			closed_nodes[best_node.tile] <- best_node.tile;

			/*
			 * Distance is distance manhattan, so don't even need to be calculated,
			 *  every next tile is just one tile farther from start than previous.
			 */
			local d = best_node.distance_to_start + 1;
			foreach (id, offset in offsets) {
				local t = best_node.tile + offset;
				/*
				 * No channels building mean really simple
				 *  adjanced tiles check code.
				 */
				if (t == best_node.parent_node.tile) continue;
				if (!AIMarine.AreWaterTilesConnected(best_node.tile, t)) continue;
				if (t in closed_nodes) continue;

				local next = {tile = t, parent_node = best_node, distance_to_start = d};
				open_nodes.Insert(next, d + AIMap.DistanceManhattan(t, s));

				/* Now check diag adjacent tile */
				local diag_t = t + offsets[(id + 1) % 4];
				if (diag_t == best_node.parent_node.tile) continue;
				if (!AIMarine.AreWaterTilesConnected(t, diag_t)) continue;
				if (diag_t in closed_nodes) continue;

				local diag_next = {tile = diag_t, parent_node = best_node, distance_to_start = d + 0.41};
				open_nodes.Insert(diag_next, d + AIMap.DistanceManhattan(diag_t, s));
			}

			/* Stop if time limit reached */
			if ((k == Ticks.DAY - 1) && AIController.GetTick() > max_tick) return null;
		}

		return null;
	}

/* private */
	/**
	 * Make apropriate path from the found (end) node.
	 * @param path_end_node Last node of the some path.
	 * @return Array - path tile sequence.
	 */
	function MakePath(path_end_node)
	{
		local result = [path_end_node.tile];
		for (local x = path_end_node; x.tile != x.parent_node.tile;) {
			x = x.parent_node;
			result.append(x.tile);
		}

		return result;
	} 
}
