/**
 * HPA* for water paths search.
 */
class WHPASTAR
{
/* public */
	/**
	 * Find a buoy path between two given water areas.
	 * @param s Array with tiles of the first area to connect.
	 * @param e Array with tiles of the second area to connect.
	 * @param qt Water regions describing quadtree tham must include s area
	 *  and e area.
	 * @param buoy_distance Max allowed distance(manhattan, not L2) between
	 *  buoys in result path.
	 * @return Array of tiles where buoys should be built in order to connect
	 *  s and e, or null if pathfinding failed.
	 */
	function FindPath(s, e, qt, buoy_distance)
	{
		/* First, find the path in the global water quadtree hierarchy */
		local h_path = WQTASTAR().FindPath(this.GetQT(s, qt), this.GetQT(e, qt), qt);
		if (h_path == null) {
			//assert(null);
			return null;
		}

		/* Add docks exits at corresponding ends of found path */
		local qt_path = [s[0]];
		foreach (dummy_id, qt_square in h_path) {
			//qt_square.PrintSelf(qt.x_shift, qt.y_shift);
			/* Try to extract a water tile from each sea square */
			if (qt_square.value >= SeaLevelRecognition.SL_SEA) {
				local t = this.GetWaterTileFromSeaPatch(qt_square, qt);
				if (t != null) qt_path.append(t);
			}
		}
		qt_path.append(e[0]);

		/*
		 * Run "regular" water pf for each pair of adjanced tiles in the
		 *  upper scale path to recieve accurate sequence of water tiles
		 *  actually connecting given "s" and "e" areas.
		 */
		local tile_path = [];
		for (local i = 1; i < qt_path.len(); i++) {
			local t1 = qt_path[i - 1];
			local t2 = qt_path[i];
			local max_ticks = max(Ticks.DAY, 6 * AIMap.DistanceManhattan(t1, t2));
			local small_path = WASTAR().FindPath(t1, t2, max_ticks);
			if (small_path == null)	return null;

			// Note that we'll receive repeating tiles in result sequence.
			// This is qt centers. Obvious 1 candidates to places buoys.
			tile_path.extend(small_path);
		}

		/*
		 * Still, full water path is mostly needed to be sure that we avoid
		 *  lower scale obstacles which can be missed due to quadtree
		 *  restrictions(or the very nature of hierarchical approach itself).
		 * So, full water path is not something we really want, and we want
		 *  the buoy path - sequence of tiles to place buoys for ships.
		 * And so we need to throw away around 98% of found tiles, and
		 *  the tiles we'll keep will be our result.
		 * Question then is: which tiles to keep?
		 */

		/* It's good idea to find(and keep) tiles that already have buoys */
		local existing_buoys = [];
		for (local i = tile_path.len() - 1; i >= 1; i--) {
			//AISign.BuildSign(tile_path[i], "x");
			if (AIMarine.IsBuoyTile(tile_path[i])) {
				existing_buoys.push(i);
				if (tile_path[i - 1] == tile_path[i]) i--;
			}
		}

		/*
		 * This routine function will select tiles for buoys between
		 *  two given points inside path.
		 */
		local f = function(tile_path, start_index, end_index) : (buoy_distance) {
			local result = [];
			local next_buoy_tile = null;
			for (local i = start_index + 2, j = 0; i < end_index + 1; i++, j++) {
				local last = tile_path[i - 1];
				if (last == tile_path[i]) {
					next_buoy_tile = last;
				}
				if (AIMarine.IsBuoyTile(last)) {
					j = 0;
					result.append(last);
					next_buoy_tile = null;
					continue;
				}
				if (j >= buoy_distance) {
					j = 0;
					result.append(next_buoy_tile != null ? next_buoy_tile : last);
					next_buoy_tile = null;
				}
			}
			return result;
		}

		local buoy_path = [];
		local i = 0;
		for (local k = 0, d = buoy_distance; existing_buoys.len() > 0; i = k) {
			k = existing_buoys.pop();
			/*
			 * Distance to next buoy > max allowed => run routine function,
			 *  culculate and use insertion points between current tile and
			 *  next buoy tile as path steps;
			 * Distance to next buoy <= max allowed => just use that buoy as
			 *  next path point.
			 */
			buoy_path.extend(k - i > d ? f(tile_path, i, k) : [tile_path[k]]);
		}
		buoy_path.extend(f(tile_path, i, tile_path.len() - 1));

		/*foreach (dummy_id, t in buoy_path) {
			AISign.BuildSign(t, "b");
		}*/
		return buoy_path;
	}

/* private */
	/**
	 * Get quadtree leaves corresponding to the given tiles.
	 * @param tiles Array of tiles.
	 * @param big_qt Top level water quadtree.
	 * @return Array with smallest quadtree leaves set that cover given tiles.
	 */
	function GetQT(tiles, big_qt)
	{
		local result = {};
		foreach (dummy_id, t in tiles) {
			local x = AIMap.GetTileX(t);
			local y = AIMap.GetTileY(t);
			local center_tile = AIMap.GetTileIndex(x, y);
			if (center_tile in result) continue;

			local qt = big_qt.GetIntersection(x, y, x, y, -1).pop();
			if (qt.value >= SeaLevelRecognition.SL_SEA_SHORE) {
				local x = big_qt.x_shift + qt.x_min + qt.edge_len / 2;
				local y = big_qt.y_shift + qt.y_min + qt.edge_len / 2;
				result[center_tile] <- qt;
				continue;
			}

			/*
			 * If tile lies in land area => insead of this land try to grab
			 *  it's water neighbours.
			 */
			foreach (dummy_id, neighbour in big_qt.GetAdjacent(qt)) {
				if (neighbour.value < SeaLevelRecognition.SL_SEA_SHORE) continue;
				local x0 = big_qt.x_shift + neighbour.x_min + neighbour.edge_len / 2;
				local y0 = big_qt.y_shift + neighbour.y_min + neighbour.edge_len / 2;
				center_tile = AIMap.GetTileIndex(x0, y0);
				if (center_tile in result) continue;

				/*
				 * Check how much land mass lies in direct line from
				 *  neighbour's center to our candidate tile.
				 * If there are too much land => most likely it'll block
				 *  water path and thus such neighbour should be ignored.
				 */
				local line = TileUtils.GetLine(x, y, x0, y0);
				local water_tiles = 0;
				foreach (dummy, tile_xy in line) {
					local t = AIMap.GetTileIndex(tile_xy.x, tile_xy.y);
					if (AITile.IsWaterTile(t)) water_tiles++;
				}
				if (water_tiles * 100.0 / line.len() >= SeaLevelRecognition.SL_SEA) {
					result[center_tile] <- neighbour;
					continue;
				}

				// repeat for 4 sides' centers
				x0 = big_qt.x_shift + neighbour.x_min + neighbour.edge_len / 2;
				y0 = big_qt.y_shift + neighbour.y_min;
				line = TileUtils.GetLine(x, y, x0, y0);
				water_tiles = 0;
				foreach (dummy, tile_xy in line) {
					local t = AIMap.GetTileIndex(tile_xy.x, tile_xy.y);
					if (AITile.IsWaterTile(t)) water_tiles++;
				}
				if (water_tiles * 100.0 / line.len() >= SeaLevelRecognition.SL_SEA) {
					result[center_tile] <- neighbour;
					continue;
				}

				x0 = big_qt.x_shift + neighbour.x_min;
				y0 = big_qt.y_shift + neighbour.y_min + neighbour.edge_len / 2;
				line = TileUtils.GetLine(x, y, x0, y0);
				water_tiles = 0;
				foreach (dummy, tile_xy in line) {
					local t = AIMap.GetTileIndex(tile_xy.x, tile_xy.y);
					if (AITile.IsWaterTile(t)) water_tiles++;
				}
				if (water_tiles * 100.0 / line.len() >= SeaLevelRecognition.SL_SEA) {
					result[center_tile] <- neighbour;
					continue;
				}

				x0 = big_qt.x_shift + neighbour.x_min + neighbour.edge_len;
				y0 = big_qt.y_shift + neighbour.y_min + neighbour.edge_len / 2;
				line = TileUtils.GetLine(x, y, x0, y0);
				water_tiles = 0;
				foreach (dummy, tile_xy in line) {
					local t = AIMap.GetTileIndex(tile_xy.x, tile_xy.y);
					if (AITile.IsWaterTile(t)) water_tiles++;
				}
				if (water_tiles * 100.0 / line.len() >= SeaLevelRecognition.SL_SEA) {
					result[center_tile] <- neighbour;
					continue;
				}

				x0 = big_qt.x_shift + neighbour.x_min + neighbour.edge_len / 2;
				y0 = big_qt.y_shift + neighbour.y_min + neighbour.edge_len;
				line = TileUtils.GetLine(x, y, x0, y0);
				water_tiles = 0;
				foreach (dummy, tile_xy in line) {
					local t = AIMap.GetTileIndex(tile_xy.x, tile_xy.y);
					if (AITile.IsWaterTile(t)) water_tiles++;
				}
				if (water_tiles * 100.0 / line.len() >= SeaLevelRecognition.SL_SEA) {
					result[center_tile] <- neighbour;
					continue;
				}
				//neighbour.PrintSelf(big_qt.x_shift, big_qt.y_shift);
			}
		}

		return CodeUtils.TableToArray(result);
	}

	/**
	 * Try to receive water tile from the given quadtree square.
	 * @param qt Sea square describing quadtree.
	 * @param big_qt Super parent quadtree.
	 * @return Center tile of the given qt, if it is a water tile, null else.
	 */
	function GetWaterTileFromSeaPatch(qt, big_qt)
	{
		local x = big_qt.x_shift + qt.x_min + qt.edge_len / 2;
		local y = big_qt.y_shift + qt.y_min + qt.edge_len / 2;
		local t = AIMap.GetTileIndex(x, y);
		return (AIMarine.IsBuoyTile(t) || AITile.IsWaterTile(t)) ? t : null;
	}
}
