/**
 * A* path search in water quadtree.
 */
class WQTASTAR
{
/* public */
	/**
	 * Find path between given quadtrees.
	 * @param s Array with "source" quadtrees.
	 * @param e Array with "goal" quadtrees.
	 * @param qt Super parent quadtree.
	 * @return Array of quadtrees inside "super parent" representing path,
	 *  or null if failed.
	 */
	function FindPath(s, e, qt)
	{
		if (s.len() == 0 || e.len() == 0) {
			return null;
		}

		/*
		 * Found path will be reversed, so in order to receive "right" path
		 *  we can swap "s" and "e" here, or call result_array.reverse() later.
		 * First method is better => it is choosen.
		 * That's why "end_qt" in "s";
		 */
		local goal_tile = null, goal_tiles = {};
		foreach (dummy_id, end_qt in s) {
			local end_node = this.GetQTNode(end_qt, qt);
			goal_tiles[end_node.qt_center] <- end_node.qt_center;
			goal_tile = end_node.qt_center;
		}

		local open_nodes = Terron_FibonacciHeap();
		foreach (dummy_id, start_qt in e) {
			local start_node = this.GetQTNode(start_qt, qt);
			start_node.distance_to_start = 0;
			/*
			 * Distance will be actual distance manhattan between quadtree
			 *  centers if buoys are built in those centers, or twice as that
			 *  otherwise.
			 */
			open_nodes.Insert(start_node, TileUtils.DiagDistance(start_node.qt_center, goal_tile));
		}

		for (local closed_nodes = {}; !open_nodes.IsEmpty(); ) {
			local best_node = open_nodes.Pop();
			local t = best_node.qt_center;
			if (t in goal_tiles) return this.MakeQTPath(best_node);
			if (t in closed_nodes) continue;

			closed_nodes[t] <- t;

			/* Adjacent quadtrees checking */
			foreach (dummy_id, next in this.GetNeighbours(best_node, qt)) {
				local next_t = next.qt_center;
				local d = best_node.distance_to_start;
				d += TileUtils.DiagDistance(t, next_t);
				if (d >= next.distance_to_start || next_t in closed_nodes) continue;
				next.parent_node = best_node;
				next.distance_to_start = d;
				open_nodes.Insert(next, d + TileUtils.DiagDistance(next_t, goal_tile));
			}
		}

		return null;
	}

/* private */
	/**
	 * Make a pf node from the given quadtree object.
	 * @param qt Quadtree.
	 * @param big_qt Super parent quadtree.
	 * @return Node table.
	 */
	function GetQTNode(qt, big_qt)
	{
		local x = big_qt.x_shift + qt.x_min + qt.edge_len / 2;
		local y = big_qt.y_shift + qt.y_min + qt.edge_len / 2;
		return {
			qt = qt,
			qt_center = AIMap.GetTileIndex(x, y),
			parent_node = null,
			distance_to_start = 1000000,
		}
	}

	/**
	 * Get neighbours of the given node.
	 * @param pf_node Node, which neighbours we want to know.
	 * @param big_qt Super parent quadtree.
	 * @return Array with nodes adjacent to the given node.
	 */
	function GetNeighbours(pf_node, big_qt)
	{
		local result = [];
		foreach (dummy_id, neighbour in big_qt.GetAdjacent(pf_node.qt)) {
			if (neighbour.value >= SeaLevelRecognition.SL_SEA_SHORE) {
				result.append(this.GetQTNode(neighbour, big_qt));
			}
		}

		//local key = MarineBasin.key;
		//if (!(key in pf_node.qt.custom_data)) return result;
		//local right_id = pf_node.qt.custom_data[key];

		foreach (dummy_id, neighbour in big_qt.GetDiagAdjacent(pf_node.qt)) {
			//if (!(key in neighbour.custom_data)) continue;
			//if (neighbour.custom_data[key] != right_id) continue;
			if (neighbour.value >= SeaLevelRecognition.SL_SEA_SHORE) {
				result.append(this.GetQTNode(neighbour, big_qt));
			}
		}

		return result;
	}

	/**
	 * Make apropriate path from the found (end) node.
	 * @param path_end_node Last node of the some path.
	 * @return Array - path - quadtree sequence.
	 */
	function MakeQTPath(path_end_node)
	{
		local result = [path_end_node.qt];
		for (local x = path_end_node; x.parent_node != null;) {
			x = x.parent_node;
			result.append(x.qt);
		}

		return result;
	}
}
