/**
 * Class that handles cruise transport organization.
 */
class CruiseCentralControl extends AbstractTransportSystem
{
/* public */
	static function GetClassName()
	{
		return "CruiseCentralControl";
	}

	static function Restore(memento)
	{
		return CruiseCentralControl();
	}

	/**
	 * CruiseCentralControl constructor.
	 */
	constructor()
	{
		::AbstractTransportSystem.constructor();
		GlobalTransportMap.Get().SetUpNewTransportNodeClass(CruiseCenter.GetClassName());
		local a_name = "Creating new cruise line..."
		local strategy = CruiseBuildStrategy();
		local t = GameTime.MONTH;
		this.best_action = UniversalAction(a_name, strategy, 3 * t, t); 
		this.transport_politics = CruiseTransportPolitics();

		AIStartedEvent.AddListener(this);
		NewNodeEvent.AddListener(this);
	}

	function GetName()
	{
		return "CruiseCentralControl";
	}

	/**
	 * Restore own trade lines after game load.
	 */
	function OnAIStarted(is_game_loaded)
	{
		if (is_game_loaded) {
			Corporation.Get().AddAction(RestoreCruiseLinesAction());
		}
	}

	/**
	 * Intgrate new node into pax network.
	 * @param node New game node.
	 */
	function OnNewNode(node)
	{
		this.transport_politics.EnableTransportNode(node);
	}

/* protected */
	function doGetBestAction()
	{
		return this.best_action;
	}

/* private */
	/** Currently best action for CruiseCentralControl. */
	best_action = null;

	/** Pax routes handling strategy. */
	transport_politics = null;
}

/**
 * Class that controls water trade build processes.
 */
class CruiseBuildStrategy extends UniversalAction_AbstractExecutionStrategy
{
/* public */
	constructor()
	{
		::UniversalAction_AbstractExecutionStrategy.constructor();

		this.object_id = Terron_ObjectsCounter.PopNewID();

		this.cruise_lines = TableContainer.new("Cruise Lines");
		this.transit_allowed = (AIController.GetSetting("BusModule") >= 2);

		TradeRouteClosedEvent.AddListener(this);
	}

	function ConcretizeContext()
	{
		local n = this.cruise_lines.len();
		local saturated = 0;
		foreach (dummy_id, cruise_line in this.cruise_lines) {
			if (cruise_line.IsSaturated()) saturated++;
		}
		if (n - saturated > 1 || (n - saturated).tofloat() / (n + 1) > 0.2) {
			return {priority = APriority.BAD, context = null};
		}

		local best_route = null;
		local best_route_rank = APriority.BAD;
		local links = GlobalTransportMap.Get().link_map[CruiseCenter.GetClassName()];
		foreach (link_id, link in links[LinkColor.LC_GREEN]) {
			foreach (dummy_route_id, route in link.routes) { 
				local r = this.RankCruiseRoute(route);
				if (r > best_route_rank) {
					best_route_rank = r
					best_route = route;
				}
			}
		}

		return {priority = best_route_rank, context = best_route};
	}

	function GetActionCost(route)
	{
		return WaterSettings.GetDockCost() + 2 * AIEngine.GetPrice(route.GetEngine());
	}

	/**
	 * Returns cruise route rank.
	 * @param route Cruise route which rank we want to know.
	 * @return Number indicating the route's upgrade desireability.
	 */
	function RankCruiseRoute(route)
	{
		if (route.IsNefarious() || route.IsCompletelyUpgraded()) return APriority.BAD;

		local build_cost = WaterSettings.GetDockCost();
		local income = route.GetEstimatedIncome();
		local full_cost = build_cost + route.GetVehiclesCost();
		build_cost += 2 * AIEngine.GetPrice(route.GetEngine());
		build_cost = min(build_cost, full_cost);
		return ActionPriority.CalculatePriority(build_cost, full_cost, 0, income);
	}

	function Execute(route)
	{
		local s = route.GetStart();
		if (s == null) assert(null);
		return WaterRouteBuilder.BuildRoute(
			route.GetStart(), route.start_basin_id,
			route.GetEnd(), route.end_basin_id,
			route.IsOneWay(),
			route.cargo_id,
			route.accurate_length
		);
	}

	function HandleResult(result, route)
	{
		if ((typeof result) != "table") return result;

		if (result.err_code == -3 || result.err_code == 0) {
			route.SetAccurateLength(result.new_accurate_length);
		}

		if (result.err_code != 0) {
			/* Building failed, stop future retrying then */
			if (result.err_code == -1) route.MakeNefarious(2 * GameTime.YEAR);
			return result.err_code;
		}

		/* Route completed */
		local s_id = result.s_dock.GetStationID();
		local e_id = result.e_dock.GetStationID();
		local s = result.s_dock;
		local e = result.e_dock;
		//local s = s_id > e_id ? result.e_dock : result.s_dock;
		//local e = s_id > e_id ? result.s_dock : result.e_dock;
		local buoys = ShipDepartment.Get().water_paths_register.GetBuoySequence(s_id, e_id);

		this.cruise_lines.AddItem(CruiseLine(route, s, e, buoys));

		if (this.transit_allowed) {
			local c = route.cargo_id;
			TransitWantedEvent.Fire([s, [TransitType.TT_BUS], {[c] = 1}]);
			TransitWantedEvent.Fire([e, [TransitType.TT_BUS], {[c] = 1}]);
		}

		/* Suspend some routes building, to lazy renew cached values */
		local s = route.GetStart();
		local tmp = GlobalTransportMap.Get().structured_transport_nodes;
		s = tmp[CruiseCenter.GetClassName()][s.GetTypeID()][s.node_id];
		foreach (dummy_link_id, link in s.links_out) {
			foreach (dummy_route_id, to_suspend_route in link.routes) {
				to_suspend_route.MakeNefarious(32 * GameTime.DAY);
			}
		}
		return result.err_code;
	}

	/**
	 * Proper reaction on cruise line close.
	 */
	function OnTradeRouteClose(trade_route)
	{
		this.cruise_lines.RemoveItem(trade_route.GetID());
	}

/* private */
	/** Container with all cruise lines. */
	cruise_lines = null;

	/** Is transit routes supported? */
	transit_allowed = false;

	/** The object's runtime ID. */
	object_id = null;

/* public */
	/** Get the ID of this object. */
	function GetID() { return this.object_id;}
}

Terron_ClassTable.RegisterClass(CruiseCentralControl);
