/**
 * Class that handles ship trade.
 */
class CruiseLine extends TradeRoute
{
/* public */
	/** Special name for ships assigned to the cruise lines. */
	static special_ship_name_prefix = "Liner";

	static function GetClassName()
	{
		return "CruiseLine";
	}

	/**
	 * Creates a new cruise line - water passengers trade route.
	 * @param cruise_route Base water route.
	 * @param s_dock Dock station, will serve as line "start".
	 * @param e_dock Dock station, will serve as line "end".
	 * @param buoys_sequence Array with buoy path from s_dock to e_dock.
	 */
	constructor(cruise_route, s_dock, e_dock, buoys_sequence)
	{
		local c = cruise_route.GetCargoID();
		this.name = s_dock.GetName() + " -> " + e_dock.GetName();
		this.name += " (" + AICargo.GetCargoLabel(c) + ")";
		::TradeRoute.constructor([s_dock, e_dock]);
 		cruise_route.trade_routes.AddItem(this);

		this.buoys = buoys_sequence;
		this.s_dock = s_dock;
		this.e_dock = e_dock;
		this.len = WaterlineUtils.GetBuoyPathLength(s_dock.GetLocation(), e_dock.GetLocation(), buoys);
		this.cruise_route = cruise_route;

		this.vehicles_required = min(3 * cruise_route.GetEstimatedVehicles() / 4, 9);

		/* Must be done to set correct engine dependent values */
		local init_engine = this.GetBestEngineToBuy();

		/* Find vehicles already assigned to this route (if any) */
		local s_list = s_dock.GetVehicles();
		foreach (v, dummy in e_dock.GetVehicles()) {
			if (AIVehicle.IsValidVehicle(v) && s_list.HasItem(v)) {
				if (AIVehicle.GetState(v) == AIVehicle.VS_CRASHED) continue;
				AIVehicle.SetName(v, (CruiseLine.special_ship_name_prefix + v));

				local e = AIVehicle.GetEngineType(v);
				local s = AIEngine.GetMaxSpeed(e);
				//local t = 8 + (668 * 2 * this.len / (24 * 0.9 * s)).tointeger();
				// same as above
				local t = 8 + 62 * this.len / s;
				local f = 1.0 / t;

				this.s_dock.GetTerminal().AddVehicle(v, f);
				this.e_dock.GetTerminal().AddVehicle(v, f);
				this.vehicles.AddItem(v, v);
			}
		}

		Terron_Event.ShipLost.AddListener(this);

		this.actual_profit = this.cruise_route.vehicle_profit;
		this.vehicles_update_date = AIDate.GetCurrentDate();
	}

	function Close()
	{
		if (!(::TradeRoute.Close())) return false;
		this.cruise_route.trade_routes.RemoveItem(this.GetID());
		return true;
	}

	function GetCargoID()
	{
		return this.cruise_route.cargo_id;
	}

	function GetOneVehicleProfit()
	{
		return this.actual_profit;
	}

	function IsSaturated()
	{
		local vehicles_count = this.GetVehicles().Count();
		return (this.vehicles_required - vehicles_count <= 0);
	}

	/**
	 * Close self when ships lost.
	 * @note Usually when ships become lost, that is hard to fix.
	 */
	function OnShipLost(v)
	{
		if (this.vehicles.HasItem(v)) {
			CodeUtils.Log("Ship " + AIVehicle.GetName(v) + " is lost...", 2);
			if (this.actual_profit <= 0) this.Close();
			//assert(null);
		}
	}

/* protected */
	function doBuyVehicle()
	{
		local n = this.vehicles.Count();
		local start_dock = this.s_dock, finish_dock = this.e_dock;

		if (!start_dock.depot.Check()) return -1;
		if (!finish_dock.depot.Check()) return -1;

		local v = -1;

		if (n < 4) {
			local load_flag = (n % 2 == 0);
			if (!load_flag) {
				start_dock = this.e_dock;
				finish_dock = this.s_dock;
			}

			v = AIVehicle.BuildVehicle(start_dock.depot.location, this.GetBestEngineToBuy());
			if (!AIVehicle.IsValidVehicle(v)) return v;
			AIVehicle.SetName(v, (CruiseLine.special_ship_name_prefix + v));

			AIOrder.AppendOrder(v, start_dock.GetLocation(), AIOrder.OF_FULL_LOAD_ANY);
			AIOrder.AppendOrder(v, start_dock.depot.location, AIOrder.OF_NONE);

			if (load_flag) {
				foreach (dummy_id, t in this.buoys) {
					AIOrder.AppendOrder(v, t, AIOrder.OF_NONE);
				}
			} else {
				for (local i = this.buoys.len() - 1; i >= 0; i--) {
					AIOrder.AppendOrder(v, this.buoys[i], AIOrder.OF_NONE);
				}
			}

			AIOrder.AppendOrder(v, finish_dock.GetLocation(), AIOrder.OF_FULL_LOAD_ANY);
			AIOrder.AppendOrder(v, finish_dock.depot.location, AIOrder.OF_NONE);

			if (!load_flag) {
				foreach (dummy_id, t in this.buoys) {
					AIOrder.AppendOrder(v, t, AIOrder.OF_NONE);
				}
			} else {
				for (local i = this.buoys.len() - 1; i >= 0; i--) {
					AIOrder.AppendOrder(v, this.buoys[i], AIOrder.OF_NONE);
				}
			}
		} else {
			local example_vehicle = this.vehicles.Begin();
			// Get depot poistion from order 
			local t = AIOrder.GetOrderDestination(example_vehicle, 1);
			v = AIVehicle.BuildVehicle(t, this.GetBestEngineToBuy());
			if (!AIVehicle.IsValidVehicle(v)) return v;
			AIVehicle.SetName(v, (CruiseLine.special_ship_name_prefix + v));
			AIOrder.CopyOrders(v, example_vehicle);
		}

		local f = this.GetFrequency();
		this.s_dock.GetTerminal().AddVehicle(v, f);
		this.e_dock.GetTerminal().AddVehicle(v, f);
		return v;
	}

	function doSellVehicle(v)
	{
		::TradeRoute.doSellVehicle(v);

		this.s_dock.GetTerminal().RemoveVehicle(v);
		this.e_dock.GetTerminal().RemoveVehicle(v);
	}

	/**
	 * Get most suitable engine for this trade route.
	 * @return Currently best engine id for this trade route.
	 */
	function doGetEngine()
	{
		local c = this.GetCargoID();
		local s_is_transit = this.s_dock.IsTransit();
		local e_is_transit = this.e_dock.IsTransit();

		//if (!this.big_ships_user) {
			local s_waiting = AIStation.GetCargoWaiting(this.s_dock.GetStationID(), c);
			local e_waiting = AIStation.GetCargoWaiting(this.e_dock.GetStationID(), c);
			local alot_for_s = 2 * WaterSettings.Get().big_ship_capacity;
			local alot_for_e = e_is_transit ? alot_for_s : 2 * alot_for_s;
			local alot_for_s = s_is_transit ? alot_for_s : 2 * alot_for_s;

			this.big_ships_user = (s_waiting > alot_for_s && e_waiting > alot_for_e);
		//}

		local e = this.cruise_route.ChooseBestEngine(!this.big_ships_user);
		if (AIEngine.IsBuildable(e)) {
			local s = AIEngine.GetMaxSpeed(e);
			//local t = 8 + (668 * 2 * this.len / (24 * 0.9 * s)).tointeger();
			// same as above
			local t = 8 + 62 * this.len / s;
			local ci = AICargo.GetCargoIncome(c, this.len, t / 2);
			if (s_is_transit) {
				local id = this.s_dock.GetStationID();
				TransitIncomeCalculatedEvent.Fire({station_id = id, cargo_id = c, cargo_income = ci});
			}
			if (e_is_transit) {
				local id = this.e_dock.GetStationID();
				TransitIncomeCalculatedEvent.Fire({station_id = id, cargo_id = c, cargo_income = ci});
			}
			this.frequency = 1.0 / t;
		} else {
			this.frequency = 1.0; // value doesn't matter, just not null is good
			this.Disable(GameTime.YEAR + GameTime.DAY);
		}
		return e;
	}

	function GetFrequency()
	{
		return this.frequency;
	}

/* private */
	/** Improved value for profit per vehicle */
	actual_profit = 0;

	/** True when this line is able to use big ships, false else */
	big_ships_user = false;

	/** Array with buoys for this trade route */
	buoys = null;

	/** Base cruise route for this trade route */
	cruise_route = null;

	/** First dock */
	s_dock = null;

	/** Second dock */
	e_dock = null;

	/** The arrival frequency for this trade route best engine */
	frequency = 1;

	/** Length */
	len = null;

	/** Optimal number of vehicles required by this trade route */
	vehicles_required = 0;

	/** Date of last vehicle number check */
	vehicles_update_date = null;
}

function CruiseLine::EstimateVehiclesNeeds()
{
	local vehicles_count = this.GetVehicles().Count();
	local current_date = AIDate.GetCurrentDate();
	local dt = current_date - this.vehicles_update_date;
	if (dt < GameTime.MONTH) {
		return max(this.vehicles_required - vehicles_count, 0);
	}

	local e = this.GetBestEngineToBuy();
	if (!AIEngine.IsBuildable(e)) return 0;

	if (AIDate.GetYear(current_date) % 3 == this.GetID() % 3
		&& AICompany.GetBankBalance(AICompany.COMPANY_SELF) > AIEngine.GetPrice(e)) {
		/* Sell obsolete vehicles (if has money to replace) */
		local to_sell = [];
		this.vehicles.Valuate(AIVehicle.GetAge);
		foreach (v, age in this.vehicles) {
			if (age < 8 * GameTime.YEAR) continue;
			if (e != AIVehicle.GetEngineType(v) || age > 12 * GameTime.YEAR) {
				to_sell.append(v);
			}
		}

		if (to_sell.len() > 0) {
			this.SendSpecifiedVehiclesToSell(to_sell);
			this.vehicles_required = this.GetVehicles().Count() + 1;
			return 1;
		}
	}

	local t_rate = this.cruise_route.transport_rate;
	local n = min(6, this.s_dock.GetTerminal().EstimateVehiclesNeeds(t_rate));
	if (n >= 0 && !this.cruise_route.IsOneWay()) {
		n = min(n, this.e_dock.GetTerminal().EstimateVehiclesNeeds(t_rate));
	}

	this.vehicles_update_date = current_date;

	/* Calculate more accurate profit for vehicles attached to this route */
	this.actual_profit = 0;
	vehicles.Valuate(AIVehicle.GetProfitLastYear);
	foreach (v, profit_last_year in vehicles) {
		if (AIVehicle.GetAge(v) < 1 * GameTime.YEAR) continue;
		this.actual_profit += profit_last_year;
	}
	vehicles.Valuate(AIVehicle.GetProfitThisYear);
	foreach (v, profit_this_year in vehicles) {
		if (AIVehicle.GetAge(v) < 1 * GameTime.YEAR) continue;
		this.actual_profit += profit_this_year;
	}
	if (this.actual_profit != 0) {
		local t = 12 + AIDate.GetMonth(current_date);
		this.actual_profit = this.actual_profit / t;
		this.actual_profit = this.actual_profit / vehicles.Count();
		if (this.actual_profit < 0) this.actual_profit = 0;
	} else {
		this.actual_profit = this.cruise_route.vehicle_profit;
	}

	/* limit sells */
	if (n < 0) {
		if (vehicles_count > 1) {
			local c = this.GetCargoID();
			local s_waiting = AIStation.GetCargoWaiting(this.s_dock.GetStationID(), c);
			local e_waiting = AIStation.GetCargoWaiting(this.e_dock.GetStationID(), c);
			local s_rating = AIStation.GetCargoRating(this.s_dock.GetStationID(), c);
			local e_rating = AIStation.GetCargoRating(this.e_dock.GetStationID(), c);
			if ((s_waiting < 100 && s_rating > 60) || (e_waiting < 100 && e_rating > 60)) {
				this.vehicles_required = vehicles_count - 1;
				return -1;
			}
		}

		local current_month = AIDate.GetMonth(current_date);

		local vehicles = this.GetVehicles();
		vehicles.Valuate(AIVehicle.GetEngineType);

		foreach (v, engine in vehicles) {
			if (AIVehicle.GetAge(v) < 3 * GameTime.YEAR) continue;

			local month_profit = AIVehicle.GetProfitLastYear(v);
			month_profit += AIVehicle.GetProfitThisYear(v);
			month_profit = month_profit / (12 + current_month);

			if (AIEngine.GetPrice(engine) > 3 * month_profit * GameTime.MONTHS_PER_YEAR) {
				this.vehicles_required = vehicles_count - 1;
				this.SendSpecifiedVehiclesToSell([v]);
				return 0;
			}
		}

		return 0;
	}

	this.vehicles_required = vehicles_count + n;
	return n; 
}
