/**
 * Class that represent cruise(water) route.
 * @note Only for towns.
 */
class CruiseRoute extends SimpleRoute
{
/* public */
	static function GetClassName()
	{
		return "CruiseRoute";
	}

	/**
	 * Creates a new cruise route. 
	 * @param start_node Route start node.
	 * @param start_basin_id ID of the water basin where start node is located.
	 * @param end_node Route end node.
	 * @param end_basin_id ID of the water basin where end node is located.
	 * @param cargo_id ID of the cargo to transport.
	 */
	constructor(start_node, start_basin_id, end_node, end_basin_id, cargo_id)
	{
		local is_transit = (end_node.type_id == NodeTypeID.NT_TRANSIT_DROP);
		::SimpleRoute.constructor(start_node, end_node, cargo_id, is_transit);
		this.accurate_length = this.length;
		this.start_basin_id = start_basin_id;
		this.end_basin_id = end_basin_id;

		local s = start_node.GetLocation();
		local e = end_node.GetLocation();
		local dx = abs(AIMap.GetTileX(s) - AIMap.GetTileX(e));
		local dy = abs(AIMap.GetTileY(s) - AIMap.GetTileY(e));
		this.length = dx + dy;

		this.ev = Property(this.doGetEstimatedVehicles.bindenv(this), GameTime.MONTH);
		CodeUtils.Log("Cruise route " + this.name + " revealed", 1);
	}

	function GetVehiclesCost()
	{
		return this.GetEstimatedVehicles() * this.best_engine_price;
	}

	function GetEstimatedIncome()
	{
		return this.GetEstimatedVehicles() * this.vehicle_profit;
	}

	/**
	 * Get estimated number of vehicles for this route.
	 * @return Estimated number of vehicles required for this route.
	 */
	function GetEstimatedVehicles()
	{
		return this.ev.Get();
	}

	/**
	 * Set accurate length of the route instead of initial approximation.
	 * @param new_length New length.
	 */
	function SetAccurateLength(new_length)
	{
		this.accurate_length = new_length;
		this.engine.ForceUpdate();
		this.ev.ForceUpdate();
	}

/* protected */
	/**
	 * Aux function to select best ship type for this route.
	 * @param ignore_big_ships When true big(> 300 cargo units) ships'll be ignored.
	 * @return Engine ID of the ship type considered to be the best for this route.
	 */
	function ChooseBestEngine(ignore_big_ships)
	{
		return WaterlineUtils.ChooseBestEngine(cargo_id, ignore_big_ships, 65);

		/* Scan engine list and remove futile engines */
		local list = AIEngineList(AIVehicle.VT_WATER);
		/* Can't handle route cargo => futile */
		list.Valuate(AIEngine.CanRefitCargo, cargo_id);
		list.RemoveValue(0);
		/* Breaks often => futile */
		list.Valuate(AIEngine.GetReliability);
		list.KeepAboveValue(65);

		local cfg = WaterSettings.Get();
		if (!cfg.allow_slow_ships) {
			/* Slow ships do not work well so far, it's better to ignore them */
			list.Valuate(AIEngine.GetMaxSpeed);
			list.KeepAboveValue(cfg.slow_ship_speed);
		}

		if (ignore_big_ships) {
			list.Valuate(AIEngine.GetCapacity);
			list.RemoveAboveValue(cfg.big_ship_capacity);
		}

		/* Select engine with max (capacity * speed) */
		local f = function (e) {
			local reliability = AIEngine.GetReliability(e);
			return AIEngine.GetCapacity(e) * AIEngine.GetMaxSpeed(e) * reliability;
		}
		local e = CorporationUtils.ChooseBestEngineFromList(list, f);
		return e;
	}

	/**
	 * Get estimated number of vehicles for this route.
	 * @return Estimated number of vehicles required for this route.
	 */
	function doGetEstimatedVehicles()
	{
		// division by 2 because docks do not(usually) capture all town's houses
		return this.GetEngine() == -1 ? 0 :
			(this.GetFreeProduction() / (2 * this.transport_rate)).tointeger();
	}

	function doGetEngine()
	{
		/* Set correct values for the case when there is no valid engine */
		this.vehicle_profit = 0;
		this.transport_rate = 0;

		local e = this.ChooseBestEngine(true);
		if (e == -1) {
			this.MakeNefarious(GameTime.YEAR);
			return -1;
		}

		// +20% to cost to handle inflation
		this.best_engine_price = 6 * AIEngine.GetPrice(e) / 5;
		local s = AIEngine.GetMaxSpeed(e);
		/* time for full transport cycle */
		//local t = 8 + (668 * 2 * this.length / (24 * 0.9 * s)).tointeger();
		// same as above
		local t = 8 + 62 * this.accurate_length / s;

		local ci = AICargo.GetCargoIncome(this.cargo_id, this.length, t / 2 + 4);

		this.transport_rate = 1.0 * AIEngine.GetCapacity(e) * GameTime.MONTH / t;
		this.vehicle_profit = this.transport_rate * ci;
		if (!this.one_way) this.vehicle_profit = 1.8 * this.vehicle_profit;
		this.vehicle_profit -= AIEngine.GetRunningCost(e) / GameTime.MONTHS_PER_YEAR;
		return e;
	}

/* private */
	/** Accurately measured route length. */
	accurate_length = null;

	/** ID of the water basin that contain "start" dock. */
	start_basin_id = null;

	/** ID of the water basin that contain "end" dock. */
	end_basin_id = null;

	/** Estimated vehicles. */
	ev = null;

	/** Price of the best engine. */
	best_engine_price = 0;

	/** Estimated arrival frequency for ships assigned to this route. */
	frequency = null;

	/** Transport capacity of one route vehicle (cargo amount per month). */
	transport_rate = null;

	/** Estimated month profit from one vehicle. */
	vehicle_profit = null;
}
