/**
 * Class that handles [dock]-[water depot] link.
 */
class WaterDepot extends Terron_SaveableObject
{
/* public */
	static function GetClassName()
	{
		return "WaterDepot";
	}

	static function Restore(memento)
	{
		local depot = WaterDepot(memento.station_id);
		depot.location = memento.location;
		return depot;
	}

	/** Tile ID of the depot's location */
	</ must_save = true />
	location = -1;

	/**
	 * WaterDepot class constructor.
	 * @param station_id ID of dock associated with new depot object.
	 */
	constructor(station_id)
	{
		::Terron_Object.constructor(null);
		this.station_id = station_id;
	}

	/**
	 * Build a new water depot.
	 * @return 0 if depot was built, -1 if build failed.
	 */
	function Build()
	{
		local s = AIStation.GetName(this.station_id);
		CodeUtils.Log("Building a water depot near " + s + "...", 1);

		this.location = this.doBuildDepot(35);
		if (this.location == -1) {
			local s_name = AIStation.GetName(this.station_id);
			CodeUtils.Log("Failed to build a depot for " + s_name, 2);
		} else {
			CodeUtils.Log("...built", 1);
		}
		return this.location != -1 ? 0 : -1;
	}

	/**
	 * Check if depot is in place.
	 * @return True if check passed, false otherwise.
	 */
	function Check()
	{
		local t = this.location;
		if (t == -1 || !AIMarine.IsWaterDepotTile(t)) {
			this.Build();
			return AIMarine.IsWaterDepotTile(this.location);
		}
		return true;
	}

	function GetName()
	{
		return AIStation.GetName(this.station_id) + "'s service depot";
	}

/* private */
	/** Associated station's ID */
	</ must_save = true />
	station_id = null;

	/**
	 * Checks if depot can be built safely, so it will not block any water
	 *  passage.
	 * @param t1 Water depot consist of two tiles. This should be first.
	 * @param t2 Water depot consist of two tiles. This should be second.
	 * @return True if location is good, false otherwise.
	 */
	function IsSafeDepotLocation(t1, t2)
	{
		local x1 = AIMap.GetTileX(t1);
		local y1 = AIMap.GetTileY(t1);
		local x2 = AIMap.GetTileX(t2);
		local y2 = AIMap.GetTileY(t2);
		local x_min = min(x1, x2);
		local y_min = min(y1, y2);
		local x_max = max(x1, x2);
		local y_max = max(y1, y2);
		local corner1 = AIMap.GetTileIndex(x_min - 2, y_min - 2);
		local corner2 = AIMap.GetTileIndex(x_max + 2, y_max + 2);
		local list = AITileList();
		if (AIMap.IsValidTile(corner1) && AIMap.IsValidTile(corner1)) {
			list.AddRectangle(corner1, corner2);
			list.Valuate(AITile.IsWaterTile);
			list.RemoveValue(0); 
		}
		return list.Count() == 30;
	}
}

/**
 * Water depot building implementation.
 * @param steps Max distance allowed between the depot and the start_tile.
 * @return Tile ID of built depot, -1 if failed to build one. 
 */
/* private */ function WaterDepot::doBuildDepot(steps)
{
	local ti = AIMap.GetTileIndex;
	local offsets = [ti(-1, 0), ti(0, 1), ti(1, 0), ti(0, -1)];
	local list = AITileList();
	local expand_list = AITileList();

	foreach (t, dummy in AITileList_StationType(this.station_id, AIStation.STATION_DOCK)) {
		foreach (dummy_id, offset in offsets) {
			if (AITile.IsWaterTile(t - offset)) list.AddTile(t - offset);
		}
	}

	/* Add tiles connected to start into special list */
	for (local i = 0; i < steps; i++, expand_list.Clear()) {
		foreach (t1, dummy_value in list) {
			/* foreach unchecked item in list add its road neighbour into expand_list */
			foreach (dummy_id, offset in offsets) {
				local t2 = t1 - offset;
				if (!AIMarine.AreWaterTilesConnected(t1, t2)) continue;
				if (list.HasItem(t2)) continue;
				if (AIMarine.IsWaterDepotTile(t2)) {
					if (AICompany.IsMine(AITile.GetOwner(t2))) return t2;
				}

				expand_list.AddTile(t2);
			}
		}

		if (steps < 5) {
			foreach (t, dummy in expand_list) list.AddTile(t);
			continue;
		}

		foreach (t1, dummy in expand_list) {
			/* Try to build depot if we need one and there is suitable tile in expand_list */
			foreach (dummy_offset_id, offset in offsets) {
				local t2 = t1 - offset;
				if (!AIMarine.AreWaterTilesConnected(t1, t2)) continue;
				if (!this.IsSafeDepotLocation(t1, t2)) continue;

				{
					local test_mode = AITestMode;
					if (!AIMarine.BuildWaterDepot(t2, t1)) continue;
				}

				AIMarine.BuildWaterDepot(t2, t1);
				if (AIMarine.IsWaterDepotTile(t2)) {
					if (AICompany.IsMine(AITile.GetOwner(t2))) return t2;
				}
			}
		}

		foreach (t, dummy in expand_list) list.AddTile(t);
	}

	return -1;
}

Terron_ClassTable.RegisterClass(WaterDepot);
