/**
 * Base class that handles AI stations work.
 */
class Station extends AbstractTransportSystem
{
/* public */
	/**
	 * Base constructor for AI stations.
	 * @param station_id ID of in-game station corresponding to this object.
	 * @param node_id ID of a node where the station is built (base node).
	 * @param node_type_id Type ID of the node where this station is built.
	 * @note It's assumed that station accepts(/supplies)
	 *  cargo only for the base node.
	 */
	constructor(station_id, node_id, node_type_id)
	{
		::AbstractTransportSystem.constructor();
		this.can_accept = {};
		this.can_supply = {};
		this.node_id    = node_id;
		this.node_type  = node_type_id;
		this.station_id = station_id;
		this.terminals  = [];
		this.location   = AIStation.GetLocation(station_id);
		this.is_town_station = (node_type_id == NodeTypeID.NT_PAX_TOWN ||
			node_type_id == NodeTypeID.NT_RAW_TOWN);
	}

	function Close()
	{
		CodeUtils.Log("Closing station: " + this.GetName(), 2);
		/* Remove information about self from the AI */
		if (!SaveLoadUtils.IsLoading()) {
			local node = this.GetStationNode();
			if (node != null) {
				node.GetAllStations().RemoveStation(this);
				Terron.UpdateMemento();
			}
		}

		if (this.IsTransit()) TransitStationClosingEvent.Fire(this.station_id);

		local new_name = SellLostVehiclesTask.special_sell_name;
		/* Sell all vehicles */
		foreach (v, dummy in this.GetVehicles()) {
			if (AIVehicle.GetName(v) != (new_name + v)) {
				AIVehicle.SetName(v, new_name + v);
				//while (AIOrder.GetOrderCount(v) > 0) AIOrder.RemoveOrder(v, 0); 
				AIVehicle.SendVehicleToDepot(v);
			} else {
				if (AIVehicle.IsStoppedInDepot(v)) {
					AIVehicle.SellVehicle(v);
				} else if (AIVehicle.IsInDepot(v)) {
					if (AIVehicle.SellVehicle(v)) continue;
					if (AIVehicle.StartStopVehicle(v)) {
						AIController.Sleep(1);
						if (!AIVehicle.SellVehicle(v)) AIVehicle.StartStopVehicle(v);
					}
				}
			}
		}

		foreach (id, item in CorporationUtils.GetOwners(this)) {
			CorporationUtils.CloseSystem(item);
		}

		/* Demolish station itself */
		local tiles = AITileList_StationType(this.GetStationID(), AIStation.STATION_ANY);
		foreach (t, dummy in tiles) {
			AITile.DemolishTile(t);
			if (AITile.IsStationTile(t)) return false;
		}

		// It's better to do so, easy debug if something goes wrong
		this.station_id = -1;

		return true;
	}

	function GetName()
	{
		return AIStation.GetName(this.station_id);
	}

	/**
	 * Get the station location.
	 * @return The station location tile's ID.
	 */
	function GetLocation()
	{
		return this.location;
	}

	/**
	 * Get the station ID.
	 * @return The station's unique in-game ID.
	 */
	function GetStationID()
	{
		return this.station_id;
	}

	/**
	 * Get the base node of the station.
	 * @return Base node for this station.
	 */
	function GetStationNode()
	{
		//(!) should be avoided during loading.
		return NodesRepository.Get().GetNode(this.node_type, this.node_id);
	}

	/**
	 * Checks whether the station is built in order to accept given cargo.
	 * @param c Cargo ID.
	 * @return True if and only if the station can accept the given cargo.
	 */
	function CanAccept(c)
	{
		return c in this.can_accept;
	}

	/**
	 * Checks whether this station is built in order to supply given cargo.
	 * @param c Cargo ID.
	 * @return True if and only if the station can supply vehicles with
	 *  the given cargo.
	 */
	function CanSupply(c)
	{
		return c in this.can_supply;
	}

	/**
	 * Get the station vehicles.
	 * @return AIVehicleList with station's vehicles.
	 */
	function GetVehicles()
	{
		return AIVehicleList_Station(this.station_id);
	}

	/**
	 * Checks whether the station is transit station.
	 * @return True if and only if the station is used as cargo transfer point.
	 */
	function IsTransit()
	{
		return this.is_transit;
	}

/* protected */
	/**
	 * Register new terminal for this station.
	 * @param terminal Station_Terminal (derived)class object.
	 */
	function RegisterTerminal(terminal)
	{
		this.terminals.append(terminal);
		if (terminal.UnloadOnly()) {
			foreach (c, dummy in terminal.cargo_list) {
				this.can_accept[c] <- c;
			}
		} else {
			this.can_accept[terminal.cargo_id] <- terminal.cargo_id;
			this.can_supply[terminal.cargo_id] <- terminal.cargo_id;
		}
	}

/* private */
	/** Table with cargo IDs that this station can accept. */
	can_accept = null;

	/** Table with cargo IDs that this station can provide. */
	can_supply = null;

	/** Flag indicating whether this is a town station. */
	is_town_station = true;

	/** Id of the base node. */
	</ must_save = true />
	node_id = null;

	/** Type of the base node. */
	</ must_save = true />
	node_type = null

	/** Station location(just tile ID). */
	location = null;

	/** Station's in-game id. */
	</ must_save = true />
	station_id = null;

	/** True if this station is used as cargo transfer point. */
	is_transit = false;

	/** Array with station's terminals */
	// </ must_save = false />;
	// Derived classes must handle terminals save/load
	terminals = null;
}
