/**
 * Class to handle station special "loading" terminals - these specialize
 *  in cargo loading into vehicles(including situation when vehicle
 *  must both load/unload at station).<p>
 */
class Station_LoadingTerminal extends Station_Terminal
{
/* public */
	static function GetClassName()
	{
		return "Station_LoadingTerminal";
	}

	/**
	 * Creates loading terminal.
	 * @param station_id ID of the host station.
	 * @param cargo_id ID of the cargo to handle at this terminal.
	 */
	constructor(station_id, cargo_id)
	{
		::Station_Terminal.constructor(station_id);
		this.cargo_id = cargo_id;

		local current_date   = AIDate.GetCurrentDate();
		this.build_date      = current_date;
		this.past_state      = {date = current_date, rating = 65, stock = 50};
		this.long_past_state = {date = current_date, rating = 65, stock = 50};
		this.past_state.transport_amount_change      <- 0;
		this.long_past_state.transport_amount_change <- 0;
	}

	function GetName()
	{
		local station_name = AIStation.GetName(this.station_id);
		local cargo_label = AICargo.GetCargoLabel(this.cargo_id);
		return station_name + " " + cargo_label +  " loading terminal";
	}

	/**
	 * Get associated cargo's ID.
	 * @return ID of the cargo that must be serviced by this loading terminal. 
	 */
	function GetCargoID()
	{
		return this.cargo_id;
	}

	/**
	 * Get estimated number of additional vehicles required by this station.
	 * @param transport_per_vehicle Cargo amount that will be transported
	 *  per month by one additional vehicle.
	 * @return Number of additional vehicles (negative <==> need to sell). 
	 */
	function EstimateVehiclesNeeds(transport_per_vehicle)
	{
		/* No vehicles => buy one */
		if (this.arrival_frequency_map.len() == 0) return 1;

		local now = this.GetCurrentState();
		local last = (now.date - this.past_state.date) > GameTime.MONTH ?
			this.past_state : this.long_past_state;

		local dt = now.date - last.date;
		local dtm = dt.tofloat() / GameTime.MONTH;
		local rating_gap = max(this.target_rating - now.rating, 0);
		local stock = now.stock; 
		if (dtm < 1) stock -= this.past_state.transport_amount_change;

		local loading = this.GetLoadingVehicles();
		if (stock <= 10 && loading <= 1) {
			/*
		 	* For recently built stations we risk and buy vehicles even when
		 	*  stock is low.
		 	*/
			if (now.date - this.build_date < 1.5 * GameTime.YEAR) return 1;
		}

		local max_vehicles = this.GetMaxLoadingVehicles();
		if (stock < transport_per_vehicle / 2 && rating_gap == 0 && loading >= max_vehicles) {
			//CodeUtils.Log("Too many loading vehicles at " + this.GetName(), 2);
			return max_vehicles - loading;
		}

		/* Produced stock can dissapear at stations with low rating */
		stock += (stock + 1) * dtm * rating_gap / this.target_rating;
		if (last.stock == 0) stock = max(0, stock.tointeger() - 50);

		/* Simplest way to predict future step */
		local result = (stock - last.stock) / (dtm  * transport_per_vehicle);
		/* Stock reduced, but rating is low or falling => wait, do not sell */
		if (result < 0) {
			if (now.rating <= this.target_rating / 2) {
				result = (stock > transport_per_vehicle / 2) ? 1 : 0;
			} else if (now.rating <= (4 * this.target_rating / 5) || now.rating < last.rating - 2) {
				result = 0;
			} else {
				// maybe it'll prevent buy - sell - buy - sell - e.t.c... situation
				result += 1;
			}
		}

		/* Rating is low => buy, result says "almost buy" => buy */
		if (now.rating <= (4 * this.target_rating / 5)) result++;
		/* Huge stocks => more vehicles needed */
		if (stock >= 1200) result++;
		/* Tweak selling desire, sell only when stocks drop is large enough */
		if (result < 0) result = (result <= -4) ? result / 2 : -1;

		local n = result.tointeger();
		this.past_state.transport_amount_change += n * transport_per_vehicle;
		if (now.date - this.past_state.date > GameTime.MONTH) {
			this.long_past_state = clone this.past_state;
			this.past_state = now;
		}
		//CodeUtils.Log(this.GetName() + " needs " + n + " vehicles", 1);
		return n;
	}

/* protected */
	/**
	 * Only one(this) cargo ID must be serviced by the loading terminal.
	 */
	cargo_id = null;

	/**
	 * Get the amount of own vehicles currently heading to this terminal and
	 *  located very close to this terminal.
	 * @note This base function is not very accurate.
	 */
	function GetLoadingVehicles();

	/**
	 * Get the max amount of loading vehicles considered to be normal.
	 * More vehicles at the station, is considered as overflowing.
	 * @param c Cargo id to load.
	 */
	 function GetMaxLoadingVehicles(c)
	 {
	 	return 2;
	 }

/* protected */
	/** Desired cargo/station rating. */
	target_rating = 65;

/* private */
	/** Table with previous function call(or step) data */
	past_state = null;

	/** Table with at least month old function call data */
	long_past_state = null;

	/** Get the table with current step data */
	function GetCurrentState()
	{
		local id = this.station_id;
		local c = this.cargo_id;

		local state = clone ::Station_LoadingTerminal_StateReport;
		state.date = AIDate.GetCurrentDate();
		state.rating = min(AIStation.GetCargoRating(id, c), this.target_rating);
		state.stock = AIStation.GetCargoWaiting(id, c);
		return state;
	}
}

/**
 * Table to keep information about station/cargo state.<p>
 * State mean amount of cargo available for transport and transport rating at
 *  station.<p>
 * We'll use current state table(T[k]), and previous call state(T[k-1]) to
 *  try to predict next call state(T[k+1]) => we receive ability to predict
 *  number of vehicles that will be needed in future.
 */
Station_LoadingTerminal_StateReport <- {
	/** Report moment date */
	date = null,

	/** Cargo rating in report moment. */
	rating = null,

	/** Cargo rating in report moment. */
	stock = null,

	/** Delta from bought/sold vehicles until next report. */
	transport_amount_change = 0
}
