/**
 * Base class for trade routes.
 * Trade route is a system of stations and vehicles supposed do bring profit.
 */
class TradeRoute extends AbstractTransportSystem
{
/* public */
	/**
	 * TradeRoute constructor.
	 * @param stations Array with stations.
	 */
	constructor(stations)
	{
		::AbstractTransportSystem.constructor();
		if (this.name == null) this.name = "Trade Route #" + this.GetID();

		this.vehicles = AIList();
		foreach (dummy_id, station in stations) {
			this.AddSubsystem(station);
		}

		this.engine = Property(this.doGetEngine.bindenv(this), GameTime.ENGINE_REVIEW_INTERVAL);
		Corporation.Get().actions.AddItem(AIManageVehiclesAction(this));
	}

	function Close()
	{
		local n = this.vehicles.Count();
		if (n > 0) this.SendVehiclesToSell(n);

		this.engine.Destroy();

		TradeRouteClosedEvent.Fire(this);

		return true;
	}

	function GetName()
	{
		return this.name;
	}

	/**
	 * Buy a new vehicle for the trade route.
	 * @return ID on new vehicle, or -1 for any error.
	 */
	function BuyVehicle()
	{
		local new_vehicle_id = this.doBuyVehicle();
		if (!AIVehicle.IsValidVehicle(new_vehicle_id)) {
			CodeUtils.Log("Failed to build vehicle. Reason: " + AIError.GetLastErrorString(), 3);
			CodeUtils.Log("Trade route name " + this.GetName(), 3);
			CodeUtils.Log("Best engine id " + this.GetBestEngineToBuy(), 3);

			if (AIError.GetLastError() == AIVehicle.ERR_VEHICLE_TOO_MANY) {
				this.SendVehiclesToSell(this.vehicles.Count());
				this.Disable(3 * GameTime.YEAR);
				AIController.Sleep(10);
			} else {
				this.failed_buy_attempts++;
				if (this.failed_buy_attempts > 50) {
					local to_close = [];
					foreach (dummy_id, sybsystem in this.subsystems) {
						to_close.append(sybsystem);
					}
					foreach (dummy_id, item in to_close) {
						CorporationUtils.CloseSystem(item);
					}
				}
				CorporationUtils.CloseSystem(this);
			}
			return -1;
		}

		if (!AIVehicle.RefitVehicle(new_vehicle_id, this.GetCargoID())) {
			AILog.Error("Unable to refit vehicle. Reason: " + AIError.GetLastErrorString());
			AIVehicle.SellVehicle(new_vehicle_id);
			return -1;
		}

		this.vehicles.AddItem(new_vehicle_id, new_vehicle_id);

		local engine_name = AIEngine.GetName(AIVehicle.GetEngineType(new_vehicle_id));
		CodeUtils.Log("+ " + engine_name + " [" + this.name + "]", 2);

		AIVehicle.StartStopVehicle(new_vehicle_id);
		return new_vehicle_id;
	}

	/**
	 * Estimate trade route need in vehicles.
	 * @return +n(>0) - this trade route requires n additional vehicles;<p>
	 *         -n(<0) - this trade route has too many vehicles, need to sell n;<p>
	 *         0 current vehicles number is optimal.
	 */
	function EstimateVehiclesNeeds();

	/**
	 * Get the most suitable engine for this trade route.
	 * @return Best engine id.
	 */
	function GetBestEngineToBuy()
	{
		return this.engine.Get();
	}

	/**
	 * Get this trade route cargo type.
	 * @return The ID of the cargo this trade vehicles must transport.
	 */
	function GetCargoID();

	/**
	 * Get estimated month profit from one vehicle assigned to this trade route.
	 * @note Vehicle engine assumed to be valid result of this.GetBestEngineToBuy();
	 * @return Estimated month profit.
	 */
	function GetOneVehicleProfit();

	/**
	 * Get this trade route vehicles.
	 * @return list with id of vehicles servicing this trade route.
	 */
	function GetVehicles() { return this.vehicles;}

	/**
	 * Get time before next manage check required.
	 * @param n Additional vehicles counter - result of the previous
	 *  manage operation.
	 * @return Time in game days.
	 */
	function GetManageInterval(n)
	{
		return GameTime.MONTH * (n == 0 ? 3 : (n > 0 ? 1 : 2));
	}

	/**
	 * Checks(quickly) whether this trade route is completely suturated with
	 *  vehicles.
	 * @return True if and only if the trade route doesn't need any additional
	 *  vehicles right now.
	 */
	function IsSaturated();

	/**
	 * Send a several vehicles from this trade route to sell.
	 * @param n Number of vehicles to sell.
	 * @return 0.
	 */
	function SendVehiclesToSell(n)
	{
		if (n > 1) {
			CodeUtils.Log("Selling " + n + " vehicles [" + this.name + "]", 2);
		} else if (n == 1) {
			CodeUtils.Log("Selling a vehicle [" + this.name + "]", 2);
		}

		n = min(n, this.vehicles.Count());
		if (n <= 0) return 0;

		this.vehicles.Valuate(AIVehicle.GetAge);
		this.vehicles.Sort(AIList.SORT_BY_VALUE, false);
		local to_sell = array(n);
		foreach (v, age in this.vehicles) {
			if (n <= 0) break;
			to_sell[n - 1] = v;
			n--;
		}

		foreach (dummy_id, v in to_sell) {
			this.doSellVehicle(v);
		}
		return 0;
	}

	/**
	 * Send a vehicle from this trade route to sell.
	 * @param vehicle_ids_array Array with vehicle IDs to sell.
	 * @return 0.
	 */
	function SendSpecifiedVehiclesToSell(vehicle_ids_array)
	{
		CodeUtils.Log("Selling specified vehicles from [" + this.name + "]", 2);
		foreach (dummy_id, v in vehicle_ids_array) {
			this.doSellVehicle(v);
		}
		return 0;
	}

	/**
	 * Get the "trade frequency".
	 * This is one station one vehicle arriving frequency.
	 * @return 1 / [full trade cycle time for vehicle with best engine].
	 */
	function GetFrequency();

/* protected */
	/** Trade route name */
	name = null;

	/** Best engine for this trade route */
	engine = null;

	/** List of this trade route vehicles */
	vehicles = null;
	
	/**
	 * Buy new vehicle for this trade route.
	 * @return id of new vehicle (-1 if fails).
	 */
	function doBuyVehicle();

	/**
	 * Sell vehicle from this trade route.
	 * @param v The ID of the vehicle to sell.
	 * @return not used.
	 */
	function doSellVehicle(v)
	{
		CodeUtils.Log("-" + AIVehicle.GetName(v), 1);

		local new_name = SellLostVehiclesTask.special_sell_name;
		new_name = new_name + v;
		AIVehicle.SetName(v, new_name);

		AIVehicle.SendVehicleToDepot(v);
		this.vehicles.RemoveItem(v);
	}

	/**
	 * Get most suitable engine for this trade route.
	 * @return Currently best engine id for this trade route.
	 */
	function doGetEngine();

/* private */
	/** How many times this route has failed to buy vehcle. */
	failed_buy_attempts = 0;
}
