/**
 * Facade for easy access to the station class objects.
 */
class StationsFacade extends Terron_SaveableObject
{
/* public */
	static function GetClassName()
	{
		return "StationsFacade";
	}

	static function Restore(memento)
	{
		local restored = StationsFacade();
		foreach (station_id, station in memento) {
			local s_id = station_id.tointeger();
			if (AIStation.IsValidStation(s_id)) restored.AddStation(station);
		}
		return restored;
	}

	/**
	 * Creates the StationsFacade object.
	 */
	constructor()
	{
		this.stations = TableContainer.new("Some node's stations");
		this.MakeMemento();
	}

	function GetMemento()
	{
		return this.memento;
	}

	function GetName()
	{
		return this.stations.GetName();
	}

	/**
	 * Add new station to this collection.
	 * @param s Station to add.
	 */
	function AddStation(s)
	{
		local station_type = s.GetClassName();
		local container_for_this_station = this.stations.GetItem(station_type);

		if (container_for_this_station == null) {
			container_for_this_station = TableContainer.new(station_type);
			this.stations.AddItem(container_for_this_station);
		}

		container_for_this_station.AddItem(s);

		this.stations_count++;
		this.MakeMemento();
	}

	/**
	 * Get all stations of the given class.
	 * @param s station_class_name Class name, string.
	 */
	function GetClassStations(station_class_name)
	{
		if (!(station_class_name in this.stations)) {
			this.stations.AddItem(TableContainer.new(station_class_name));
			this.MakeMemento();
		}
		return this.stations[station_class_name];
	}

	/**
	 * Get the array with all of the node stations.
	 */
	function GetStationsAsArray()
	{
		local result = array(this.Count());
		local i = 0;
		foreach (dummy_id, item in this.stations) {
			foreach (id, station in item) result[i++] = station;
		}
		return result;
	}

	/**
	 * Get the amount of stations.
	 */
	function Count()
	{
		return this.stations_count;
	}

	/**
	 * Remove given station from this collection.
	 * @param s The station to remove.
	 */
	function RemoveStation(s)
	{
		local s_container = this.GetClassStations(s.GetClassName());
		if (s_container.RemoveItem(s.GetID())) this.stations_count--;
		if (s_container.len() == 0) {
			this.stations.RemoveItem(s_container.GetID());
		}
		this.MakeMemento();
	}

/* private */
	/** Table that holds object's current state(for faster AI save). */
	memento = null;

	/** The container with stations */
	stations = null;

	/** The amount of stations in the container */
	stations_count = 0;

	/** Updates object's memento. */
	function MakeMemento()
	{
		this.memento = {};
		foreach (dummy_id, type_specific_stations in this.stations) {
			foreach (id, station in type_specific_stations) {
				this.memento[station.GetStationID()] <- station;
			}
		}
		Terron.UpdateMemento();
	}
}

Terron_ClassTable.RegisterClass(StationsFacade);
