/**
 * @author Wormnest (WormAI) modified by Daniela Plachtova
 * @file air_manager.nut
 * @note All air managing functions were taken from WormAI, only little modifications for compatibility purposes were made
 */

/**
 * @brief AirManager manages all air related functions
 */
class AirManager 
{
	/* AirManager: define some constants for easier maintenance. */
	static MINIMUM_BALANCE_BUILD_AIRPORT = 100000;		/* Minimum bank balance to start building airports. */
	static MINIMUM_BALANCE_AIRCRAFT = 25000;			/* Minimum bank balance to allow buying a new aircraft. */
	static MINIMUM_BALANCE_TWO_AIRCRAFT = 5000000;		/* Minimum bank balance to allow buying 2 aircraft at once. */

	static AIRCRAFT_LOW_PRICE_CUT = 500000;				/* Bank balance below which we will try to buy a low price aircraft. */
	static AIRCRAFT_MEDIUM_PRICE_CUT = 2000000;			/* Bank balance below which we will try to buy a medium price aircraft. */
	static AIRCRAFT_LOW_PRICE = 50000;					/* Maximum price of a low price aircraft. */
	static AIRCRAFT_MEDIUM_PRICE = 250000;				/* Maximum price of a medium price aircraft. */
	static AIRCRAFT_HIGH_PRICE = 1500000;				/* Maximum price of a high price aircraft. */

	static STARTING_ACCEPTANCE_LIMIT = 150;				/* Starting limit in acceptance for finding suitable airport tile. */
	static AIRPORT_LIMIT_FACTOR = 3;					/* We limit airports to max aircraft / FACTOR * 2 (2 needed per route). */
	static AIRPORT_CARGO_WAITING_LOW_LIMIT = 250;		/* Limit of waiting cargo (passengers) on airport above which we add an aircraft. */
	static AIRPORT_CARGO_WAITING_HIGH_LIMIT = 1250;		/* Limit of waiting cargo (passengers) on airport above which we add 2 aircraft. */
	static AIRPORT2_WAITING_DIFF = 150;					/* Cargo waiting diff (less) value at the other station to allow extra aircraft. */

	_towns_used = null;									/* list of all used towns */
	_route_1 = null;									/* list of all aircrafts and first airport in route as value */
	_route_2 = null;									/* list of all aircraft and second airport in route as value */
	
	distance_of_route = {};								/* table of distances of routes */
	
	engine_usefullness = null;							/* engine usefullness */
	acceptance_limit = 0;								/* Starting limit for passenger acceptance for airport finding. */
	
	
	/**
	 * @brief constructor fro AirManager
	 */
	constructor()
	{
		acceptance_limit = STARTING_ACCEPTANCE_LIMIT;
		distance_of_route = {};
		this._towns_used = AIList();
		this._route_1 = AIList();
		this._route_2 = AIList();
		this.engine_usefullness = AIList();
	}
	
	/**
	 * @brief Evaluate aircraft 
	 */
	function EvaluateAircraft() 
	
	/**
	 * @brief Sells the airports from tile_1 and tile_2. Removes towns from towns_used list too
	 * @param airport_1_tile airport 1 tile
	 * @param airport_2_tile airport 2 tile
	 */
	function SellAirports(airport_1_tile, airport_2_tile);
	
	/**
	 * @brief Sell Aircraft
	 * @param vehicle aircraft to sell
	 * @param sell_reason 0 - old age or terrible reliability (replace)
	 * 					  1 - low profit (no replace)
	 *					  2 - industry closed (no replace)
	 */
	function SellAircraft(vehicle, sell_reason);
	
	/**
	 * @brief Manage Air Routes: 1 - sell vehicles with low profit
	 *							 2 - add new aircraft if there are many passangers waiting
	 *							 2 - sell airports with no aircraft 
	 */
	function ManageAirRoutes();
	
	/**
	 * @brief Build an aircraft with orders from tile_1 to tile_2. The best available aircraft of that time will be bought.
	 * @param tile_1 tile 1
	 * @param tile_2 tile 2
	 * @param start_tile is the tile where the airplane should start, or 0 to start at the first tile.
	 * @return if suceeded ERROR_NONE else error (see ErrorManager)
	 */
	function BuildAircraft(tile_1, tile_2, start_tile);

	/**
	 * @brief Build an airport route. Find 2 cities that are big enough and try to build airport in both cities. Then we can build an aircraft and make some money.
	 * @return airport id suceeded else error (see ErrorManager)
	 * @note We limit our amount of airports to max aircraft / 3 * 2 (2 airports for a route, and 3 planes per route)
	 */
	function BuildAirportRoute();
	
	/**
	 * @brief RemoveAirport. Remove airport at specified tile (if removing fails then give a warning)
	 * @param tile tile of an airport to remove 
	 * @note that using Sleep(x) here and trying again doesn't work for some reason (removing still fails)
	 */
	function RemoveAirport(tile);
	
	/**
	 * @brief auxiliary save function for the game
	 * @return air manager data in table
	 */
	function Save();
	
	/**
	 * @brief auxiliary load function for the game
	 * @param data to load
	 */
	function Load(data);
	
	/**
	 * @brief Get maximum distance of of engine
	 * @param engine engine id
	 * @return maximum distance
	 */
	function GetMaximumDistance(engine);
	
	/**
	 * @brief Find a suitable spot for an airport, walking all towns hoping to find one. When a town is used, it is marked as such and not re-used.
	 * @param airport_type airport type
	 * @param center_tile center tile of town
	 * @return suitable tile for airport else error (see ErrorManager)
	 */
	function FindSuitableAirportSpot(airport_type, center_tile);
	
	/**
	 * @brief Handle Crashed Aircraft (build new one and remove crashed one from route list)
 	 * @param v crashed aircraft
	 */
	function HandleCrashedAircraft(v) 
	
	/**
	 * @brief Print list of used towns (for debug)
	 */
	function DebugListTownsUsed();
	
	/**
	 * @brief List all routes (per route all stations and all vehicles on that route with relevant info)
	 */
	function DebugListRoutes();
	
	/**
	 * @brief Print route1 list (for debug)
	 */
	function DebugListRoute1();
	
	/**
	 * @brief Print route info (for debug)
	 */
	function DebugListRouteInfo();
	
	/**
	 * @brief Print route2 list (for debug)
	 */
	function DebugListRoute2();
	
	/**
	 * @brief Print distance of route (for debug)
	 */
	function DebugListDistanceOfRoute();
}

function AirManager::DebugListTownsUsed()
{
	AILog.Info("---------- DEBUG _towns_used and related stations ----------");
	if (!this._towns_used) {
		AILog.Warning("WARNING: _towns_used is null!");
	}
	else {
		AILog.Info("Number of towns used: " + this._towns_used.Count())
		//foreach(t in _towns_used) {
		for (local t = _towns_used.Begin(); !_towns_used.IsEnd(); t = _towns_used.Next()) {
			local tile = _towns_used.GetValue(t);
			local is_city = (AITown.IsCity(t) ? "city" : "town");
			AILog.Info("Town: " + AITown.GetName(t) + " (id: " + t + "), " + is_city +
				", population: " + AITown.GetPopulation(t) + ", houses: " + AITown.GetHouseCount(t) +
				", grows every " + AITown.GetGrowthRate(t) + " days");
			AILog.Info("    Location: " + Util.WriteTile(AITown.GetLocation(t)) +
				", station tile " + Util.WriteTile(tile) + ").")
			local sid = AIStation.GetStationID(tile);
			local st_veh = AIVehicleList_Station(sid);
			AILog.Info("Station: " + AIStation.GetName(sid) + " (id: " + sid + "), waiting cargo: " + 
				AIStation.GetCargoWaiting(sid, ::ai_instance._general_manager.passenger_cargo_id) + ", cargo rating: " + 
				AIStation.GetCargoRating(sid, ::ai_instance._general_manager.passenger_cargo_id) + ", aircraft: " +
				st_veh.Count());
		}
	}
	AILog.Info("");
}

function AirManager::DebugListRoutes()
{
	AILog.Info("---------- DEBUG route info ----------");
		AILog.Info("Number or routes: " + (this._towns_used.Count() / 2) );
		for (local t = _towns_used.Begin(); !_towns_used.IsEnd(); t = _towns_used.Next()) {
			local st_tile = _towns_used.GetValue(t);
			// Find out whether this station is the first or last order
			local route1 = AIList();
			route1.AddList(_route_1);
			// Keep only those with our station tile
			route1.KeepValue(st_tile);
			if (route1.Count() == 0) continue;
			// List from and to station names, and distance between them, and total profit of
			// all planes on route in the last year
			local st_id = AIStation.GetStationID(st_tile);
			local st_veh = AIVehicleList_Station(st_id);
			if (st_veh.Count() == 0) {
				AILog.Warning("Station " + AIStation.GetName(st_id) + " has 0 aircraft!");
				continue;
			}
			local first = true;
			local total_profit = 0;
			// Sort vehicle list on last years profit
			st_veh.Valuate(AIVehicle.GetProfitLastYear);
			for (local veh = st_veh.Begin(); !st_veh.IsEnd(); veh = st_veh.Next()) {
				if (first) {
					// Get list of stations this vehicle has in its orders
					local veh_stations = AIStationList_Vehicle(veh);
					local st_end_id = -1;
					foreach(veh_st_id, dummy_val in veh_stations) {
						// Since we have only 2 stations in our orders any id not the same
						// as st_id will be our target station id
						if (veh_st_id != st_id) {
							st_end_id = veh_st_id;
							break;
						}
					}
					local st_end_tile = _route_2.GetValue(veh);
					local sq_dist = AITile.GetDistanceSquareToTile(st_tile, st_end_tile)
					AILog.Info( "Route from " + AIStation.GetName(st_id) +" ("+st_id+ ") to " +
						AIStation.GetName(st_end_id) +" ("+st_end_id+") "+
						", distance: " + sqrt(sq_dist).tointeger());
					first = false;
				}
				// Show info about aircraft
				AILog.Info("     " + AIVehicle.GetName(veh) + " (id: " + veh + "), age: " +
					Util.GetAgeString(AIVehicle.GetAge(veh)) + ", capacity: " + 
					AIVehicle.GetCapacity(veh, ::ai_instance._general_manager.passenger_cargo_id));
				local last_profit = AIVehicle.GetProfitLastYear(veh);
				// Increment total profit for this route
				total_profit += last_profit;
				AILog.Info("        Profit last year: " + last_profit + ", this year: " + 
					AIVehicle.GetProfitThisYear(veh));
			}
			AILog.Warning("     Total " + st_veh.Count() + " aircraft. Total profit last year: " + total_profit + ", average: " + (total_profit / st_veh.Count()));
		}
	AILog.Info("");
}

function AirManager::DebugListRoute1()
{
	//this._route_1.AddItem(vehicle, tile_1);
	//this._route_2.AddItem(vehicle, tile_2);
	AILog.Info("---------- DEBUG _route_1 ----------");
	if (!this._route_1) {
		AILog.Warning("WARNING: _route_1 is null!");
	}
	else {
		AILog.Info("Number or routes used: " + this._route_1.Count());
		for (local r = _route_1.Begin(); !_route_1.IsEnd(); r = _route_1.Next()) {
			AILog.Info("Aircraft: " + AIVehicle.GetName(r) + " (id: " + r + ", tile " + Util.WriteTile(_route_1.GetValue(r)) + ").");
		}
	}
	AILog.Info("");
}

function AirManager::DebugListRouteInfo()
{
	//this._route_1.AddItem(vehicle, tile_1);
	//this._route_2.AddItem(vehicle, tile_2);
	local temp_route = AIList();
	temp_route.AddList(this._route_1); // so that we don't sort the original list
	AILog.Info("---------- DEBUG route info ----------");
	if (!temp_route) {
		AILog.Warning("WARNING: route list is null!");
	}
	else {
		temp_route.Sort(AIList.SORT_BY_ITEM, true);
		AILog.Info("Number of aircraft used: " + temp_route.Count());
		for (local r = temp_route.Begin(); !temp_route.IsEnd(); r = temp_route.Next()) {
			local tile1 = 0;
			local tile2 = 0;
			local t1 = 0;
			local t2 = 0;
			local route_start = AIList();
			local route_end = AIList();
			route_start.AddList(this._towns_used);
			route_end.AddList(this._towns_used);
			tile1 = temp_route.GetValue(r);
			tile2 = _route_2.GetValue(r);
			route_start.KeepValue(tile1);
			t1 = route_start.Begin();
			route_end.KeepValue(tile2);
			t2 = route_end.Begin();
			local dist = this.distance_of_route.rawget(r);
			AILog.Info("Aircraft: " + AIVehicle.GetName(r) + " (id: " + r + "), from: " + 
				AITown.GetName(t1) + ", to: " + AITown.GetName(t2) + ", distance: " + dist);
		}
	}
	AILog.Info("");
}

function AirManager::DebugListRoute2()
{
	//this._route_1.AddItem(vehicle, tile_1);
	//this._route_2.AddItem(vehicle, tile_2);
	AILog.Info("---------- DEBUG _route_2 ----------");
	if (!this._route_1) {
		AILog.Warning("WARNING: _route_2 is null!");
	}
	else {
		AILog.Info("Number routes used: " + this._route_2.Count());
		for (local r = _route_2.Begin(); !_route_2.IsEnd(); r = _route_2.Next()) {
			AILog.Info("Aircraft: " + AIVehicle.GetName(r) + " (id: " + r + ", tile " + Util.WriteTile(_route_2.GetValue(r)) + ").");
		}
	}
	AILog.Info("");
}

function AirManager::DebugListDistanceOfRoute()
{
	//this.distance_of_route.rawset(vehicle, AIMap.DistanceManhattan(tile_1, tile_2));
	AILog.Info("---------- DEBUG distance_of_route ----------");
	if (!this._route_1) {
		AILog.Warning("WARNING: _route_2 is null!");
	}
	else {
		AILog.Info("Number routes used: " + this._route_2.Count());
		for (local r = _route_2.Begin(); !_route_2.IsEnd(); r = _route_2.Next()) {
			AILog.Info("Aircraft: " + AIVehicle.GetName(r) + " (id: " + r + ", tile " + Util.WriteTile(_route_2.GetValue(r)) + ").");
		}
	}
	AILog.Info("");
}

function AirManager::RemoveAirport(tile)
{
	if (!AIAirport.RemoveAirport(tile)) {
		AILog.Warning(AIError.GetLastErrorString());
		AILog.Warning("Failed to remove airport " + AIStation.GetName(AIStation.GetStationID(tile)) +
			" at tile " + Util.WriteTile(tile) );
	}
}

function AirManager::BuildAirportRoute()
{

	// No sense building airports if we already have the max (or more because amount can be changed in game)
	local max_vehicles = Vehicle.GetVehicleLimit(AIVehicle.VT_AIR);
	if (max_vehicles <= this._route_1.Count()) {
		AILog.Info("We already have the maximum number of aircraft. No sense in building an airport.");
		return ErrorManager.ERROR_MAX_AIRCRAFT;
	}
	
	// Check for our maximum allowed airports (max only set by our own script, not OpenTTD)
	local airport_count = this._towns_used.Count();
	if ((max_vehicles * 2 / AIRPORT_LIMIT_FACTOR) <= airport_count) {
		AILog.Info("Not building more airports. We already have a reasonable amount for the current aircraft limit.");
		return ErrorManager.ERROR_MAX_AIRPORTS;
	}

	// See for capacity of different airport types:
	// Airport capacity test: http://www.tt-forums.net/viewtopic.php?f=2&t=47279
	local airport_type = (AIAirport.IsValidAirportType(AIAirport.AT_LARGE) ? AIAirport.AT_LARGE : AIAirport.AT_SMALL);

	/* Get enough money to work with */
	local reservation_id = ::ai_instance._money_manager.ReserveMoney(AIAirport.GetPrice(airport_type)*2 + AIRCRAFT_LOW_PRICE, 
																   (1.1*(AIAirport.GetPrice(airport_type)*2 + AIRCRAFT_LOW_PRICE)).tointeger());

	if(reservation_id == null) {
		AILog.Warning("ErrorManager.ERROR_NOT_ENOUGH_CASH");
		return ErrorManager.ERROR_NOT_ENOUGH_CASH;
	}																
	/* Show some info about what we are doing */
	AILog.Info(Helper.GetCurrentDateString() + " Trying to build an airport route");

	local tile_1 = this.FindSuitableAirportSpot(airport_type, 0);
	if (tile_1  == ErrorManager.ERROR_FIND_AIRPORT_FINAL || tile_1 == ErrorManager.ERROR_FIND_AIRPORT_ACCEPTANCE) {
		if ((this._towns_used.Count() == 0) && (tile_1 = ErrorManager.ERROR_FIND_AIRPORT_ACCEPTANCE)) {
			// We don't have any airports yet so try again at a lower acceptance limit
			while(tile_1 == ErrorManager.ERROR_FIND_AIRPORT_ACCEPTANCE) {
				tile_1 = this.FindSuitableAirportSpot(airport_type, 0);
			}
			if (tile_1  == ErrorManager.ERROR_FIND_AIRPORT_FINAL || tile_1 == ErrorManager.ERROR_FIND_AIRPORT_ACCEPTANCE) {
			::ai_instance._money_manager.ReleaseReservation(reservation_id);
			return ErrorManager.ERROR_FIND_AIRPORT1;
			}
		}
		else {
			::ai_instance._money_manager.ReleaseReservation(reservation_id);
			return ErrorManager.ERROR_FIND_AIRPORT1;
		}
	}
	local tile_2 = this.FindSuitableAirportSpot(airport_type, tile_1);
	if (tile_2  == ErrorManager.ERROR_FIND_AIRPORT_FINAL || tile_2 == ErrorManager.ERROR_FIND_AIRPORT_ACCEPTANCE) {
		// Check for 1, not 0, here since if we get here we have at least 1 airport.
		if ((this._towns_used.Count() == 1) && (tile_2 = ErrorManager.ERROR_FIND_AIRPORT_ACCEPTANCE)) {
			// We don't have any airports yet so try again at a lower acceptance limit
			while(tile_2 == ErrorManager.ERROR_FIND_AIRPORT_ACCEPTANCE) {
				tile_2 = this.FindSuitableAirportSpot(airport_type, 0);
			}
			if (tile_2  == ErrorManager.ERROR_FIND_AIRPORT_FINAL || tile_2 == ErrorManager.ERROR_FIND_AIRPORT_ACCEPTANCE) {
				this._towns_used.RemoveValue(tile_1);
				::ai_instance._money_manager.ReleaseReservation(reservation_id);
				return ErrorManager.ERROR_FIND_AIRPORT2;
			}
		}
		else {
			this._towns_used.RemoveValue(tile_1);
			::ai_instance._money_manager.ReleaseReservation(reservation_id);
			return ErrorManager.ERROR_FIND_AIRPORT2;
		}
	}

	/* In certain cases building an airport still fails for unknown reason. */
	/* Build the airports for real */
	if (!AIAirport.BuildAirport(tile_1, airport_type, AIStation.STATION_NEW)) {
		AILog.Warning(AIError.GetLastErrorString());
		AILog.Error("Although the testing told us we could build an airport, it still failed at tile " + Util.WriteTile(tile_1) + ".");
		this._towns_used.RemoveValue(tile_1);
		this._towns_used.RemoveValue(tile_2);
		::ai_instance._money_manager.ReleaseReservation(reservation_id);
		return ErrorManager.ERROR_BUILD_AIRPORT1;
	}
	if (!AIAirport.BuildAirport(tile_2, airport_type, AIStation.STATION_NEW)) {
		AILog.Warning(AIError.GetLastErrorString());
		AILog.Error("Although the testing told us we could build an airport, it still failed at tile " + Util.WriteTile(tile_2) + ".");
		this.RemoveAirport(tile_1);
		this._towns_used.RemoveValue(tile_1);
		this._towns_used.RemoveValue(tile_2);
		::ai_instance._money_manager.ReleaseReservation(reservation_id);
		return ErrorManager.ERROR_BUILD_AIRPORT1;
	}

	local ret = this.BuildAircraft(tile_1, tile_2, tile_1);
	if (ret != ErrorManager.ERROR_NONE) {
		// For some reason removing an airport in here sometimes fails, sleeping a little
		// helps for the cases we have seen.
		AIController.Sleep(1);
		this.RemoveAirport(tile_1);
		this.RemoveAirport(tile_2);
		this._towns_used.RemoveValue(tile_1);
		this._towns_used.RemoveValue(tile_2);
		AILog.Info("Cancelled route because we couldn't build an aircraft.");
	}
	else {
		local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
		if ((balance >= MINIMUM_BALANCE_TWO_AIRCRAFT) && (Vehicle.GetVehicleLimit(AIVehicle.VT_AIR) > this._route_1.Count())) {
			/* Build a second aircraft and start it at the other airport. */
			ret = this.BuildAircraft(tile_1, tile_2, tile_2);
		}
		AILog.Info("Done building air route");
	}

	AILog.Info("");
	::ai_instance._money_manager.ReleaseReservation(reservation_id);
	return ret;
}

function AirManager::GetMaximumDistance(engine) 
{
	local max_dist = AIEngine.GetMaximumOrderDistance(engine);
	if (max_dist == 0) {
		/* Unlimited distance. Since we need to be able to keep values above a squared distance
		We set it to a predefined maximum value. Maps can be maximum 2048x2048. Diagonal will
		be more than that. To be safe we compute 10000 * 10000. */
		return 10000 * 10000;
	}
	else {
		return max_dist;
	}
}

function AirManager::BuildAircraft(tile_1, tile_2, start_tile)
{
	// Don't try to build aircraft if we already have the max (or more because amount can be changed in game)
	if (Vehicle.GetVehicleLimit(AIVehicle.VT_AIR) <= this._route_1.Count()) {
		AILog.Warning("We already have the maximum number of aircraft. No sense in building an airport.");
		return ErrorManager.ERROR_MAX_AIRCRAFT;
	}

	/* order_start_tile: where our order should start */
	local order_start_tile = start_tile;
	if (start_tile == 0) {
		order_start_tile = tile_1;
	}
	/* Build an aircraft */
	local hangar = AIAirport.GetHangarOfAirport(order_start_tile);
	local engine = null;
	local eng_price = 0;

	local engine_list = AIEngineList(AIVehicle.VT_AIR);
	/* check if engines are valid and buildable */
	engine_list.Valuate(AIEngine.IsValidEngine);
	engine_list.KeepValue(1);
	engine_list.Valuate(AIEngine.IsBuildable);
	engine_list.KeepValue(1);
	/* When bank balance < AIRCRAFT_LOW_PRICE_CUT, buy cheaper planes */
	local balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
	
	/* Balance below a certain minimum? Wait until we buy more planes. */
	if (balance < MINIMUM_BALANCE_AIRCRAFT) {
		AILog.Warning("We are low on money (" + balance + "). We are not gonna buy an aircraft right now.");
		return ErrorManager.ERROR_NOT_ENOUGH_CASH;
	}
	
	/* We don't want helicopters so weed them out. */
	engine_list.Valuate(AIEngine.GetPlaneType);
	engine_list.RemoveValue(AIAirport.PT_HELICOPTER);
	
	engine_list.Valuate(AIEngine.GetPrice);
	engine_list.KeepBelowValue(balance < AIRCRAFT_LOW_PRICE_CUT ? AIRCRAFT_LOW_PRICE : (balance < AIRCRAFT_MEDIUM_PRICE_CUT ? AIRCRAFT_MEDIUM_PRICE : AIRCRAFT_HIGH_PRICE));

	engine_list.Valuate(AIEngine.GetCargoType);
	engine_list.KeepValue(::ai_instance._general_manager.passenger_cargo_id);
	
	// Newer versions of OpenTTD allow NewGRFs to set a maximum distance a plane can fly between orders
	// That means we need to make sure planes can fly the distance necessary for our intended order.
	// Since distance is returned squared we need to get the squared distance for our intended order.
	engine_list.Valuate(this.GetMaximumDistance);
	//foreach (eng,x in engine_list) {
	//	AILog.Info("Engine: " + AIEngine.GetName(eng) + ", distance: " + AIEngine.GetMaximumOrderDistance(eng));
	//}
	//local distance_between_stations = AIOrder.GetOrderDistance(null, tile_1, tile_2);
	local distance_between_stations = AIMap.DistanceSquare(tile_1, tile_2);
	engine_list.KeepAboveValue(distance_between_stations);
	// debugging:
	//AILog.Info("squared distance: " + distance_between_stations);
	//foreach (eng,x in engine_list) {
	//	AILog.Info("Engine: " + AIEngine.GetName(eng) + ", distance: " + AIEngine.GetMaximumOrderDistance(eng));
	//}
	////////////

	//engine_list.Valuate(AIEngine.GetCapacity);
	//engine_list.KeepTop(1);
	engine_list.Valuate(GetCostFactorValuator, this.engine_usefullness);
	engine_list.KeepBottom(1);

	/* Make sure that there was a suitable engine found. */
	if (engine_list.Count() == 0) {
		// Most likely no aircraft found for the range we wanted.
		AILog.Warning("Couldn't find a suitable aircraft.");
		return ErrorManager.ERROR_BUILD_AIRCRAFT_INVALID;
	}
	
	engine = engine_list.Begin();

	if (!AIEngine.IsValidEngine(engine)) {
		AILog.Warning("Couldn't find a suitable aircraft. Most likely we don't have enough available funds.");
		return ErrorManager.ERROR_BUILD_AIRCRAFT_INVALID;
	}
	/* Price of cheapest engine can be more than our bank balance, check for that. */
	eng_price = AIEngine.GetPrice(engine);
	if (eng_price > balance) {
		AILog.Warning("Can't buy aircraft. The cheapest selected aircraft (" + eng_price + ") costs more than our available funds (" + balance + ").");
		return ErrorManager.ERROR_NOT_ENOUGH_CASH;
	}
	local vehicle = AIVehicle.BuildVehicle(hangar, engine);
	if (!AIVehicle.IsValidVehicle(vehicle)) {
		AILog.Warning(AIError.GetLastErrorString());
		AILog.Error("Couldn't build the aircraft: " + AIEngine.GetName(engine));
		return ErrorManager.ERROR_BUILD_AIRCRAFT;
	}

	/* Send him on his way */
	/* If this isn't the first vehicle with this order, then make a shared order. */
	local veh_list = AIList();
	veh_list.AddList(this._route_1);
	veh_list.KeepValue(tile_1);
	if (veh_list.Count() > 0) {
		local share_veh = veh_list.Begin();
		AIOrder.ShareOrders(vehicle, share_veh);
		if(AIOrder.GetOrderCount(vehicle) == 0) {
			AIOrder.AppendOrder(vehicle, tile_1, AIOrder.OF_FULL_LOAD_ANY);
			AIOrder.AppendOrder(vehicle, tile_2, AIOrder.OF_FULL_LOAD_ANY);
		}
		AILog.Info("Not the first vehicle: share orders.");
	}
	else {
		/* First vehicle with these orders. */
		AIOrder.AppendOrder(vehicle, tile_1, AIOrder.OF_FULL_LOAD_ANY);
		AIOrder.AppendOrder(vehicle, tile_2, AIOrder.OF_FULL_LOAD_ANY);
		AILog.Info("First vehicle: set orders.");
	}
	/* If vehicle should be started at another tile than tile_1 then skip to that order. */
	/* Currently always assumes it is tile_2 and that that is the second order, thus 1. */
	if (order_start_tile != tile_1) {
		AILog.Info("Order: skipping to other tile.");
		AIOrder.SkipToOrder(vehicle, 1);
	}
	
	/* When breakdowns are on add go to depot orders on every airport.
	   Ignore this when we added aircraft to shared orders. */
	if ((veh_list.Count() == 0) && (AIGameSettings.GetValue("difficulty.vehicle_breakdowns") > 0)) {
		/* Get the hangar tiles of both airports. */
		local Depot_Airport_1 = AIAirport.GetHangarOfAirport(tile_1);
		local Depot_Airport_2 = AIAirport.GetHangarOfAirport(tile_2);
		/* Add the depot orders: only go there if service is needed. */
		if (!AIOrder.InsertOrder(vehicle, 1, Depot_Airport_2, AIOrder.OF_SERVICE_IF_NEEDED ))
			{ AILog.Warning("Failed to insert go to depot order!"); }
		if (!AIOrder.InsertOrder(vehicle, 3, Depot_Airport_1, AIOrder.OF_SERVICE_IF_NEEDED ))
			{ AILog.Warning("Failed to insert go to depot order!"); }
	}
	
	AIVehicle.StartStopVehicle(vehicle);
	this.distance_of_route.rawset(vehicle, AIMap.DistanceManhattan(tile_1, tile_2));
	this._route_1.AddItem(vehicle, tile_1);
	this._route_2.AddItem(vehicle, tile_2);

	AILog.Info("Finished building aircraft " + AIVehicle.GetName(vehicle) + ", type: " + 
		AIEngine.GetName(engine) + ", price: " + eng_price );
	AILog.Info("Yearly running costs: " + AIEngine.GetRunningCost(engine) + ",  capacity: " + 
		AIEngine.GetCapacity(engine) + ", Maximum speed: " + AIEngine.GetMaxSpeed(engine) +
		", Maximum distance: " + AIEngine.GetMaximumOrderDistance(engine));

	return ErrorManager.ERROR_NONE;
}

function AirManager::FindSuitableAirportSpot(airport_type, center_tile)
{
	local airport_x, airport_y, airport_rad;

	airport_x = AIAirport.GetAirportWidth(airport_type);
	airport_y = AIAirport.GetAirportHeight(airport_type);
	airport_rad = AIAirport.GetAirportCoverageRadius(airport_type);

	local town_list = GeneralManager.GetTownList(true);

	/* Now find 2 suitable towns */
	for (local town = town_list.Begin(); !town_list.IsEnd(); town = town_list.Next()) {
		/* Don't make this a CPU hog */
		AIController.Sleep(1);

		local tile = AITown.GetLocation(town);

		/* Create a 30x30 grid around the core of the town and see if we can find a spot for a small airport */
		local list = AITileList();
		/* Safely add a rectangle taking care of border tiles. */
		Util.SafeAddRectangle(list, tile, 15);
		//list.AddRectangle(tile - AIMap.GetTileIndex(15, 15), tile + AIMap.GetTileIndex(15, 15));
		list.Valuate(AITile.IsBuildableRectangle, airport_x, airport_y);
		list.KeepValue(1);
		if (center_tile != 0) {
			/* If we have a tile defined, check to see if it's within the minimum and maximum allowed. */
			list.Valuate(AITile.GetDistanceSquareToTile, center_tile);
			local min_distance = AIController.GetSetting("min_airport_distance");
			local max_distance = AIController.GetSetting("max_airport_distance");
			/* Keep above minimum distance. */
			list.KeepAboveValue(min_distance * min_distance);
			/* Keep below maximum distance. */
			list.KeepBelowValue(max_distance * max_distance);
			// TODO: In early games with low maximum speeds we may need to adjust maximum and
			// maybe even minimum distance to get a round trip within a year.
		}
		/* Sort on acceptance, remove places that don't have acceptance */
		list.Valuate(AITile.GetCargoAcceptance, ::ai_instance._general_manager.passenger_cargo_id, airport_x, airport_y, airport_rad);
		list.RemoveBelowValue(this.acceptance_limit);
		
		/** debug off
		for (tile = list.Begin(); !list.IsEnd(); tile = list.Next()) {
			AILog.Info("Town: " + AITown.GetName(town) + ", Tile: " + Util.WriteTile(tile) +
				", Passenger Acceptance: " + list.GetValue(tile));
		} **/

		/* Couldn't find a suitable place for this town, skip to the next */
		if (list.Count() == 0) continue;
		/* Walk all the tiles and see if we can build the airport at all */
		{
			local test = AITestMode();
			local good_tile = 0;

			for (tile = list.Begin(); !list.IsEnd(); tile = list.Next()) {
				AIController.Sleep(1);
				if (!AIAirport.BuildAirport(tile, airport_type, AIStation.STATION_NEW)) continue;
				good_tile = tile;
				break;
			}

			/* Did we find a place to build the airport on? */
			if (good_tile == 0) continue;
		}

		AILog.Info("Found a good spot for an airport in " + AITown.GetName(town) + " (id: "+ town + 
			", tile " + Util.WriteTile(tile) + ", acceptance: " + list.GetValue(tile) + ").");

		/* Mark the town as used, so we don't use it again */
		this._towns_used.AddItem(town, tile);

		return tile;
	}

	local ret = 0;
	if (this.acceptance_limit > 25) {
		this.acceptance_limit -= 25;
		ret = ErrorManager.ERROR_FIND_AIRPORT_ACCEPTANCE;
		AILog.Info("Lowering acceptance limit for suitable airports to " + this.acceptance_limit );
	}
	else {
		// Maybe remove this? Minimum of 25 seems low enough.
		//this.acceptance_limit = 10;
		ret = ErrorManager.ERROR_FIND_AIRPORT_FINAL;
	}
	AILog.Info("Couldn't find a suitable town to build an airport in");
	return ret;
}

function AirManager::ManageAirRoutes()
{

	local list = AIVehicleList();
	list.Valuate(AIVehicle.GetVehicleType);
	list.KeepValue(AIVehicle.VT_AIR);
	local low_profit_limit = 0;
	
	/* Show some info about what we are doing */
	AILog.Info(Helper.GetCurrentDateString() + " Managing air routes.");
	
	list.Valuate(AIVehicle.GetAge);
	/* Give the plane at least 2 full years to make a difference, thus check for 3 years old. */
	list.KeepAboveValue(365 * 2);
	list.Valuate(AIVehicle.GetProfitLastYear);

	/* Decide on the best low profit limit at this moment. */
	if (Vehicle.GetVehicleLimit(AIVehicle.VT_AIR) > this._route_1.Count()) {
		/* Since we can still add more planes keep all planes that make at least some profit. */
		// TODO: When maintenance costs are on we should set low profit limit too at least
		// the yearly costs.
		low_profit_limit = GeneralManager.BAD_YEARLY_PROFIT;
		list.KeepBelowValue(low_profit_limit);
	}
	else {
		//  extensive computation for low profit limit.
		local list_count = 0;
		local list_copy = AIList();
		// Set default low yearly profit
		low_profit_limit = GeneralManager.BAD_YEARLY_PROFIT;
		list_count = list.Count();
		// We need a copy of list before cutting off low_profit
		list_copy.AddList(list);
		list.KeepBelowValue(low_profit_limit);
		if (list.Count() == 0) {
			// All profits are above our current low_profit_limit
			// Get vehicle with last years highest profit
			// We need to get the vehicle list again because our other list has removed
			// vehicles younger than 3 years, we want the absolute high profit of all vehicles
			local highest = AIVehicleList();
			highest.Valuate(AIVehicle.GetProfitLastYear);
			highest.KeepTop(1);
			local v = highest.Begin();
			local high_profit = highest.GetValue(v);
			// get profits below 20% of that
			low_profit_limit = high_profit * 3 / 10; // TESTING: 30%
			// Copy the list_copy back to list which at this point is (should be) empty.
			list.AddList(list_copy);
			// Apparently need to use Valuate again on profit for it to work
			list.Valuate(AIVehicle.GetProfitLastYear);
			list.KeepBelowValue(low_profit_limit);
			// DEBUG:
			//foreach (i,v in list) {
			//	AILog.Info("Vehicle " + i + " has profit: " + v);
			//}
			AILog.Warning("Computed low_profit_limit: " + low_profit_limit + " (highest profit: " +
				high_profit + "), number below limit: " + list.Count());
		}
		else if (list_count == 0) {
			AILog.Info("All aircraft younger than 3 years: recomputing low_profit_limit not needed.");
		}
		else {
			AILog.Warning("There are " + list.Count() + " aircraft below last years bad yearly profit limit.");
		}
	}

	// TODO: Don't sell all aircraft from the same route all at once, try selling 1 per year?
	for (local i = list.Begin(); !list.IsEnd(); i = list.Next()) {
		/* Profit last year and this year bad? Let's sell the vehicle */
		::ai_instance._general_manager.SendToDepotForSelling(i, 1); // true = 1 = low profit
		/* Sell vehicle provided it's in depot. If not we will get it a next time.
		   This line can also be removed probably since we handle selling once a 
		   month anyway. */
	}

	/* Don't try to add planes when we are short on cash */
	if (::ai_instance._money_manager.GetAvailableMoney() < AIRCRAFT_LOW_PRICE) return ErrorManager.ERROR_NOT_ENOUGH_CASH;
	else if (Vehicle.GetVehicleLimit(AIVehicle.VT_AIR) <= this._route_1.Count()) {
		// No sense building plane if we already have the max (or more because amount can be changed in game)
		AILog.Info("We already have the maximum number of aircraft. No sense in checking if we need to add planes.");
		return ErrorManager.ERROR_MAX_AIRCRAFT;
	}

	list = AIStationList(AIStation.STATION_AIRPORT);
	list.Valuate(AIStation.GetCargoWaiting, ::ai_instance._general_manager.passenger_cargo_id);
	list.KeepAboveValue(AIRPORT_CARGO_WAITING_LOW_LIMIT);
	list.Sort(AIList.SORT_BY_VALUE, false);
	
	for (local i = list.Begin(); !list.IsEnd(); i = list.Next()) {
		local list2 = AIVehicleList_Station(i);
		/* No vehicles going to this station, abort and sell */
		if (list2.Count() == 0) {
			// This can happen when after building 2 airports it fails to build an aircraft
			// due to lack of money or whatever and then removing one of the airports fails
			// due to unknown reasons. A fix that seems to help so far is doing a Sleep(1)
			// before removing the airports but just to be sure we check here anyway.
			// In that case tile_1 and 2 will be 0 although there still is a station.
			local t1 = this._route_1.GetValue(i);
			local t2 = this._route_2.GetValue(i);
			if ((t1 == 0) && (t2 == 0)) {
				AILog.Warning("Airport " + AIStation.GetName(i) + " still exists. Trying to remove it now.");
				this.RemoveAirport(AIStation.GetLocation(i));
			}
			else {
				AILog.Warning("***** Encountered station without vehicles, should not happen here! *****");
				AILog.Info("Station " + i + " = " + AIStation.GetName(i) );
				AILog.Info("Stations at tiles " + Util.WriteTile(t1) + " and " + Util.WriteTile(t2) );
				this.SellAirports(t1, t2);
			}
			continue;
		};

		/* Find the first vehicle that is going to this station */
		local v = list2.Begin();
		local dist = this.distance_of_route.rawget(v) / 2;
		
		/* Find the id of the other station and then request that stations waiting cargo. */
		local st = this._route_1.GetValue(v);
		if (st == AIStation.GetLocation(i)) {
			// Need _route_2 for the station tile of the other one
			st = this._route_2.GetValue(v);
		}
		local s2_id = AIStation.GetStationID(st);
		local s2_waiting = AIStation.GetCargoWaiting(s2_id, ::ai_instance._general_manager.passenger_cargo_id);

		list2.Valuate(AIVehicle.GetAge);
		list2.KeepBelowValue(dist);
		/* Do not build a new vehicle if we bought a new one in the last DISTANCE / 2 days */
		if (list2.Count() != 0) continue;

		/* Do not build new vehicle if there isn't at least some waiting cargo at the other station too. */
		if  (s2_waiting <= AIRPORT_CARGO_WAITING_LOW_LIMIT-AIRPORT2_WAITING_DIFF) continue;

		// TODO: Maybe also check for aircraft waiting in depot because that could be a sign of
		// too many aircraft too!

		AILog.Info("Station " + AIStation.GetName(i) + "(id: " + i +
			") has a lot of waiting passengers (cargo: " + list.GetValue(i) + "), adding a new aircraft for the route.");
		AILog.Info("Other station: " + AIStation.GetName(s2_id) + " waiting passengers: " + s2_waiting);

		/* Make sure we have enough money */
		local reservation_id = ::ai_instance._money_manager.ReserveMoney(AIRCRAFT_LOW_PRICE*2, (1.1*AIRCRAFT_LOW_PRICE*2).tointeger());
		if(reservation_id == null) {
			return ErrorManager.ERROR_NOT_ENOUGH_CASH;
		}

		/* Build the aircraft. */
		local ret = this.BuildAircraft(this._route_1.GetValue(v), this._route_2.GetValue(v), 0);
		/* If we have a real high amount of waiting cargo/passengers then add 2 planes at once. */
		/* Provided buying the first plane went ok. */
		/* Do not build new vehicle if there isn't at least some waiting cargo at the other station too. */
		if ((ret == ErrorManager.ERROR_NONE) && (AIStation.GetCargoWaiting(i, ::ai_instance._general_manager.passenger_cargo_id) > AIRPORT_CARGO_WAITING_HIGH_LIMIT) &&
			(s2_waiting > AIRPORT_CARGO_WAITING_HIGH_LIMIT-AIRPORT2_WAITING_DIFF)) {
			AILog.Info(" Building a second aircraft since waiting passengers is very high.");

			/* Build the aircraft. */
			ret = this.BuildAircraft(this._route_1.GetValue(v), this._route_2.GetValue(v), 0);
		}
		::ai_instance._money_manager.ReleaseReservation(reservation_id);
		AILog.Info(Helper.GetCurrentDateString() + " Finished managing air routes.");
		return ret;
	}

}

function AirManager::SellAirports(airport_1_tile, airport_2_tile) 
{
	/* Remove the airports */
	AILog.Info("==> Removing airports " + AIStation.GetName(AIStation.GetStationID(airport_1_tile)) + " and " + 
		AIStation.GetName(AIStation.GetStationID(airport_2_tile)) + " since they are not used anymore");
	this.RemoveAirport(airport_1_tile);
	this.RemoveAirport(airport_2_tile);
	/* Free the _towns_used entries */
	this._towns_used.RemoveValue(airport_1_tile);
	this._towns_used.RemoveValue(airport_2_tile);
	// TODO: Make a list of removed airports/tiles so that we don't build a new airport
	// in the same spot soon after we have removed it!
}

/**
 * @brief Evaluate aircraft 
 */
function AirManager::EvaluateAircraft() 
{
	/* Show some info about what we are doing */
	AILog.Info(Helper.GetCurrentDateString() + " Evaluating aircraft.");
	
	local engine_list = AIEngineList(AIVehicle.VT_AIR);
	//engine_list.Valuate(AIEngine.GetPrice);
	//engine_list.KeepBelowValue(balance < AIRCRAFT_LOW_PRICE_CUT ? AIRCRAFT_LOW_PRICE : (balance < AIRCRAFT_MEDIUM_PRICE_CUT ? AIRCRAFT_MEDIUM_PRICE : AIRCRAFT_HIGH_PRICE));

	engine_list.Valuate(AIEngine.GetCargoType);
	engine_list.KeepValue(::ai_instance._general_manager.passenger_cargo_id);

	// Only use this one when debugging:
	//engine_list.Valuate(AIEngine.GetCapacity);
	
	// First fill temporary list with our usefullness factors
	local factor_list = AIList();
	// Remember best engine for logging purposes
	local best_engine = null;
	local best_factor = 10000000; // Very high factor so any engine will be below it
	
	foreach(engine, value in engine_list) {
		// From: http://thegrebs.com/irc/openttd/2012/04/20
		// <frosch123>	both AIOrder::GetOrderDistance and AIEngine::GetMaximumOrderDistance() return 
		// squared euclidian distance
		// <frosch123>	so, you can compare them without any conversion
		// <+michi_cc>	krinn: Always use AIOrder::GetOrderDistance to query the distance.You can pass 
		// tiles that either are part of a station or are not, it will automatically calculate the right thing.
		// <+michi_cc>	AIEngine::GetMaximumOrderDistance and AIOrder::GetOrderDistance complement each other, 
		// and you can always use > or < on the returned values without knowing if it is square, manhatten or 
		// whatever that is applicable for the vehicle type.
		// <krinn>	vehlist.Valuate(AIEngine.GetMaximumOrderDistance); + vehlist.KeepValue(distance*distance)
		local _ayear = 24*365;	// 24 hours * 365 days
		local _eval_distance = 50000;	// assumed distance for passengers to travel
		if (AIEngine.IsValidEngine(engine)) {
			local speed = AIEngine.GetMaxSpeed(engine);
			local cap = AIEngine.GetCapacity(engine);
			local ycost = AIEngine.GetRunningCost(engine);
			//local costfactor = ycost / (speed * cap);
			local distance_per_year = speed * _ayear;
			local pass_per_year = cap * distance_per_year / _eval_distance;
			// No real values thus to get a sensible int value multiply with 100
			local cost_per_pass = (ycost * 100) / pass_per_year;
			if (cost_per_pass < best_factor) {
				best_factor = cost_per_pass;
				best_engine = engine;
			}
			if (AIController.GetSetting("debug_show_lists") == 1) {
				// Show info about evaluated engines
				AILog.Info("Engine: " + AIEngine.GetName(engine) + ", price: " + AIEngine.GetPrice(engine) +
					", yearly running costs: " + AIEngine.GetRunningCost(engine));
				AILog.Info( "    Capacity: " + AIEngine.GetCapacity(engine) + ", Maximum speed: " + 
					AIEngine.GetMaxSpeed(engine) + ", Maximum distance: " + AIEngine.GetMaximumOrderDistance(engine));
				AILog.Warning("    Aircraft usefulness factors d: " + distance_per_year + ", p: " + pass_per_year +
					", pass cost factor: " + cost_per_pass);
			}
			// Add the cost factor to our temporary list
			factor_list.AddItem(engine,cost_per_pass);
		}
	}
	this.engine_usefullness.Clear();
	this.engine_usefullness.AddList(factor_list);
	AILog.Info("Evaluated engines count: " + this.engine_usefullness.Count());
	AILog.Info("Best overall engine: " + AIEngine.GetName(best_engine) + ", cost factor: " + best_factor);
}

function AirManager::HandleCrashedAircraft(v) 
{
	AILog.Info("We have a crashed aircraft (" + v + "), buying a new one as replacement");
	this.BuildAircraft(this._route_1.GetValue(v), this._route_2.GetValue(v), 0);
	this._route_1.RemoveItem(v);
	this._route_2.RemoveItem(v);

}

function AirManager::SellAircraft(vehicle, sell_reason)
{
	if(sell_reason) {
			local veh_name = AIVehicle.GetName(vehicle);
			// Try to sell the vehicle
			if (AIVehicle.SellVehicle(vehicle)) {
				AILog.Info("--> Sold " + veh_name + " (id: " + vehicle + ").");
				/* Check if we are the last one serving those airports; else sell the airports */
				local list2 = AIVehicleList_Station(AIStation.GetStationID(this._route_1.GetValue(vehicle)));
				if (list2.Count() == 0) {
					local t1 = this._route_1.GetValue(vehicle);
					local t2 = this._route_2.GetValue(vehicle);
					this.SellAirports(t1, t2);
				}
				/* Remove the aircraft from the routes. */
				this._route_1.RemoveItem(vehicle);
				this._route_2.RemoveItem(vehicle);
				/* Remove aircraft from our to_depot list. */
				::ai_instance._general_manager._vehicles_to_depot.RemoveItem(vehicle);
			}
		} else {		
			local veh_name = AIVehicle.GetName(vehicle);
			
			local tile_1 = this._route_1.GetValue(vehicle);
			local tile_2 = this._route_2.GetValue(vehicle);
			
			// Try to sell the vehicle
			if (AIVehicle.SellVehicle(vehicle)) {
				
				AILog.Info("--> Sold " + veh_name + " (id: " + vehicle + "). Buying new vehicle.");
				if(this.BuildAircraft(tile_1, tile_2, 0) != ErrorManager.ERROR_NONE) {
					AILog.Info("Could not build new aircraft");
				}
				/* Remove the aircraft from the routes. */
				this._route_1.RemoveItem(vehicle);
				this._route_2.RemoveItem(vehicle);
				/* Remove aircraft from our to_depot list. */
				::ai_instance._general_manager._vehicles_to_depot.RemoveItem(vehicle);
			}
		}
}

function AirManager::Save() 
{
	local tu = ExtendedList();
	local r1 = ExtendedList();
	local r2 = ExtendedList();
	
	tu.AddList(this._towns_used);	
	r1.AddList(this._route_1);	
	r2.AddList(this._route_2);
	
	local data = {
		townsused = tu.toarray(),
		route1 = r1.toarray(),
		route2 = r2.toarray()
	};
	
	return data;
}

function AirManager::Load(data) 
{
	if ("townsused" in data) {
		local t = ExtendedList();
		t.AddFromArray(data.townsused)
		this._towns_used.AddList(t);
	}
	if ("route1" in data) {
		local r = ExtendedList();
		r.AddFromArray(data.route1)
		this._route_1.AddList(r);
	}
	if ("route2" in data) {
		local r = ExtendedList();
		r.AddFromArray(data.route2)
		this._route_2.AddList(r);
	}
}