/**
 * @author Daniela Plachtova
 * @file general_manager.nut
 */

 /**
  * @brief class General Manager (manages all managers related functions)
  */
class GeneralManager
{
	
	static DEFAULT_DELAY_MANAGE_ROUTES = 1000;		/* Default delay for managing air routes. */
	static DEFAULT_DELAY_HANDLE_LOAN = 2500;		/* Default delay for handling our loan. */
	static DEFAULT_DELAY_HANDLE_EVENTS = 100;		/* Default delay for handling events. */
	
	static BAD_YEARLY_PROFIT = 20000;				/* Yearly profit limit below which profit is deemed bad. */
	static VEHICLE_AGE_LEFT_LIMIT = 150;			/* Number of days limit before maximum age for vehicle to get sent to depot for selling. */
	static VEHICLE_MIN_RELIABILITY = 15;			/* Minimal reliability for vehicles */
	
	_vehicles_to_depot = null;						/* list of all vehicles that have been send to depot for selling */
	
	passenger_cargo_id = -1;						/* id of passangers cargo */
	mail_cargo_id = -1;								/* id of mail cargo */
	
	/**
	 * @brief constructor GeneralManager
	 */
	constructor() 
	{
		_vehicles_to_depot = AIList();
		
		local list = AICargoList();
		for (local i = list.Begin(); !list.IsEnd(); i = list.Next()) {
			if (AICargo.HasCargoClass(i, AICargo.CC_PASSENGERS)) {
				this.passenger_cargo_id = i;
			}
			if (AICargo.HasCargoClass(i, AICargo.CC_MAIL)) {
				this.mail_cargo_id = i;
			}
		}
	}
	
	/**
	 * Get Town List
	 * @param air_using true if this is used by AirManager
	 * @return town list
	 */
	function GetTownList(air_using);
	/**
	 * @brief Get freight cargo list
	 * @return orded cargo by income
	 * @note taken from trAIns
	 */
	function GetOrdedCargoList();
	/**
	 * @brief Send the vehicle to depot to be sold when it arrives.
	 * @param vehicle vehicle to be sold
	 * @param sell_reason 0 - old age or terrible reliability
						  1 - low profit
						  2 - industry closed
	 */
	function SendToDepotForSelling(vehicle,sell_reason);
	
	/**
	 * @brief Sell the vehicle provided it's in depot. If it's not yet in depot it will fail silently.
	 * @param vehicle vehicle to be sold
	 * @param sell_reason 0 - old age or terrible reliability
	 					  1 - low profit
						  2 - industry closed
	 */
	function SellVehicleInDepot(vehicle, sell_reason);
	/**
	 * @brief Sell all vehicles in depot that are marked to be sold.
	 */
	function SellVehiclesInDepot();
	
	/**
     * @brief ManageVehicleRenewal will check all vehicles for being old or needing upgrading
	 *        to a newer type. It will send all vehicles that are non optimal to depot for
	 *        selling.
	 */
	function ManageVehicleRenewal();
	
	/**
	 * @brief Manage Urgent tasks
	 */
	function ManageUrgentTasks();

	/**
	 * @brief Handle Events
	 */
	function HandleEvents();
	
	/**
	 * @brief auxiliary save function for game
	 * @return general manager data in table
	 */
	function Save();
	
	/**
	 * @brief auxiliary load function for game
	 * @param data data to load
	 */
	function Load(data);
}

function GeneralManager::ManageUrgentTasks()
{
	this.ManageVehicleRenewal();
	this.SellVehiclesInDepot();
}

function GeneralManager::GetTownList(air_using) 
{
	local town_list = AITownList();
	/* Remove all the towns we already used */
	if(air_using) town_list.RemoveList(::ai_instance._air_manager._towns_used);

	town_list.Valuate(AITown.GetPopulation);
	town_list.KeepAboveValue(::ai_instance.GetSetting("min_town_size"));
	town_list.Valuate(AIBase.RandItem);
	town_list.KeepTop(50);
	/*
	AILog.Info("TOWNLIST");
	for(local town = town_list.Begin(); !town_list.IsEnd(); town = town_list.Next()) {
		AILog.Info("town: " + AITown.GetName(town) + " Population: " + AITown.GetPopulation(town));
	}
	AILog.Info("--------");
	*/
	return town_list;
}	

function GeneralManager::GetOrdedCargoList()
{
	local cargos = AICargoList();
	cargos.Valuate(AICargo.IsValidCargo);
	cargos.KeepValue(1);
	cargos.Valuate(AICargo.HasCargoClass , AICargo.CC_PASSENGERS);
	cargos.KeepValue(0);
	cargos.Valuate(AICargo.HasCargoClass , AICargo.CC_MAIL);
	cargos.KeepValue(0);
	cargos.Valuate(AICargo.GetCargoIncome , 40 , 10);
	cargos.Sort(AIList.SORT_BY_VALUE , false);
	if(cargos.Count() == 0) throw("There is no cargos I can deal with.");
	return cargos;
}
	
function GeneralManager::SendToDepotForSelling(vehicle,sell_reason)
{
	/* Send the vehicle to depot if we didn't do so yet */
	if (!_vehicles_to_depot.HasItem(vehicle)) {
		local info_text = "--> Sending " + AIVehicle.GetName(vehicle) + " (id: " + vehicle + 
			") to depot because of ";
		if (sell_reason == 1) {
			info_text += "low profits: " + AIVehicle.GetProfitLastYear(vehicle) + " / " + 
				AIVehicle.GetProfitThisYear(vehicle);
		}
		else if(sell_reason == 0) {
			info_text += "old age or terrible reliability: " +  Util.GetAgeString(AIVehicle.GetAge(vehicle)) + " / " + 
				Util.GetAgeString(AIVehicle.GetMaxAge(vehicle));
		} else if (sell_reason == 2) info_text += "industry closed";
		AILog.Info(info_text);
		/* Send it to depot. */
		if (!AIVehicle.SendVehicleToDepot(vehicle))
		{
			//AILog.Warning(AIError.GetLastErrorString());
			AILog.Warning("Failed to send vehicle " + AIVehicle.GetName(vehicle) + "to depot!");
			return;
		}
		/* Add it to our list of vehicles that were sent to depot. */
		_vehicles_to_depot.AddItem(vehicle,sell_reason)
	}
}

function GeneralManager::SellVehicleInDepot(vehicle, sell_reason)
{
	// Make sure vehicle occurs in _vehicles_to_depot
	if (_vehicles_to_depot.HasItem(vehicle)) {
		if(AIVehicle.GetVehicleType(vehicle) == AIVehicle.VT_AIR) {
			::ai_instance._air_manager.SellAircraft(vehicle, sell_reason);
		}
		if(AIVehicle.GetVehicleType(vehicle) == AIVehicle.VT_RAIL) {
			::ai_instance._rail_manager.SellTrain(vehicle, sell_reason);
		}
	} else {
		/* Since vehicle not yet being in depot is an expected error we
		   won't show a log message for that. */
		if (AIError.GetLastError() != AIVehicle.ERR_VEHICLE_NOT_IN_DEPOT) {
			AILog.Warning(AIError.GetLastErrorString());
			AILog.Warning("Failed to sell vehicle " + AIVehicle.GetName(vehicle));
		}
	}
}

function GeneralManager::SellVehiclesInDepot()
{
	AILog.Info("- Check for vehicles waiting in depot to be sold. Vehicles to depot: " + _vehicles_to_depot.Count());
	for(local vehicle = _vehicles_to_depot.Begin(); !_vehicles_to_depot.IsEnd(); vehicle = _vehicles_to_depot.Next()) {
		this.SellVehicleInDepot(vehicle, _vehicles_to_depot.GetValue(vehicle));
	}
}

function GeneralManager::ManageVehicleRenewal()
{
	AILog.Info("- Check for vehicles that are old or have terrible reliability.")
	local list = AIVehicleList();
	list.Valuate(AIVehicle.GetAgeLeft);
	/* Keep vehicles whose age is below the limit we set. */
	list.KeepBelowValue(VEHICLE_AGE_LEFT_LIMIT);
	/* Send them all to depot to be sold. */
	for (local veh = list.Begin(); !list.IsEnd(); veh = list.Next()) {
		SendToDepotForSelling(veh, 0); // 0 for old age 
	}
	list = AIVehicleList();
	list.Valuate(AIVehicle.GetAge);
	list.KeepAboveValue(365*3);
	list.Valuate(AIVehicle.GetReliability);
	/* Keep vehicles whose reliability is below the limit we set. */
	list.KeepBelowValue(VEHICLE_MIN_RELIABILITY);
	/* Send them all to depot to be sold. */
	for (local veh = list.Begin(); !list.IsEnd(); veh = list.Next()) {
		SendToDepotForSelling(veh, 0); //0 for terrible reliability
	}
}

function GeneralManager::HandleEvents()
{
	while (AIEventController.IsEventWaiting()) {
		local e = AIEventController.GetNextEvent();
		switch (e.GetEventType()) {
			case AIEvent.ET_VEHICLE_CRASHED: {
				local ec = AIEventVehicleCrashed.Convert(e);
				local v = ec.GetVehicleID();
				AILog.Info("We have crashed vehicle: " + AIVehicle.GetName(v));
				if(AIVehicle.GetVehicleType(v) == AIVehicle.VT_AIR) ::ai_instance._air_manager.HandleCrashedAircraft(v);
				if(AIVehicle.GetVehicleType(v) == AIVehicle.VT_RAIL) ::ai_instance._rail_manager.HandleCrashedTrain(v);
			} break;
			//industry closed for trains
			case AIEvent.ET_INDUSTRY_CLOSE: {
				local ic = AIEventIndustryClose.Convert(e);
				local industry = ic.GetIndustryID();
				for(local i = 0; i < ::ai_instance._rail_manager._routes.len(); i++) {
					//look if we used now closed industry
					if(!(::ai_instance._rail_manager._routes[i]._is_town_route)) {
						if(::ai_instance._rail_manager._routes[i]._station_producing._industryID == industry || 
						   ::ai_instance._rail_manager._routes[i]._station_accepting._industryID == industry) {
							AILog.Info("There is industry (" + AIIndustry.GetName(industry) + ") closed where we have route " + ::ai_instance._rail_manager._routes[i]._route_id + ". Sending vehicles to depot for selling.");
							for(local v = ::ai_instance._rail_manager._routes[i]._vehicle_list.Begin(); !::ai_instance._rail_manager._routes[i]._vehicle_list.IsEnd(); v = ::ai_instance._rail_manager._routes[i]._vehicle_list.Next()) {
								this.SendToDepotForSelling(v, 2); //2 for industry closed
							}
							break;
						}
					}
				}
			} break;
			default:
				break;
		}
	}
}


function GeneralManager::Save() 
{
	local vtp = ExtendedList();
	vtp.AddList(this._vehicles_to_depot);
	local data = {
		vehiclestodepot = vtp.toarray()
	}
	
	return data;
}

function GeneralManager::Load(data) 
{
	if ("vehiclestodepot" in data) {
		local vtp = ExtendedList();
		vtp.AddFromArray(data.vehiclestodepot)
		this._vehicles_to_depot.AddList(vtp);
	}
}
