
require("aystar.nut")
require("util.nut");
require("valuators.nut")
require("air_manager.nut")
require("money_manager.nut")
require("error_manager.nut")
require("general_manager.nut")
require("rail_manager.nut")
require("rail_builder.nut")
require("rail_pathfinder.nut")
require("rail_route.nut")
require("rail_demolisher.nut")
require("industry_pair.nut")
require("town_pair.nut")
require("station_info.nut")
require("direction.nut")



// Import SuperLib
import("util.superlib", "SuperLib", 38);

Result <- SuperLib.Result;
Log <- SuperLib.Log;
Helper <- SuperLib.Helper;
Data <- SuperLib.DataStore;
ScoreList <- SuperLib.ScoreList;
Money <- SuperLib.Money;

Tile <- SuperLib.Tile;
DirectionS <- SuperLib.Direction;

Engine <- SuperLib.Engine;
Vehicle <- SuperLib.Vehicle;

Station <- SuperLib.Station;
Airport <- SuperLib.Airport;
Industry <- SuperLib.Industry;
Town <- SuperLib.Town;

Order <- SuperLib.Order;
OrderList <- SuperLib.OrderList;

Road <- SuperLib.Road;
RoadBuilder <- SuperLib.RoadBuilder;

// Import List library
import("AILib.List", "ExtendedList", 3);

/* Default delays */
const SLEEPING_TIME = 100;						/* Default time to sleep between loops of our AI (NB: should be a multiple of 100). */

/* Warning: delays should always be a multiple of 100 since Modulo is used! */
const DEFAULT_DELAY_EVALUATE_AIRCRAFT = 25000;		 /* Default delay for evaluating aircraft usefullness. */
const DEFAULT_DELAY_BUILD_AIR_ROUTE = 500; 			 /* Default delay before building a new airport route. */
const DEFAULT_DELAY_BUILD_INDUSTRY_RAIL_ROUTE = 400; /* Default delay before building a new industry rail route. */
const DEFAULT_DELAY_BUILD_TOWN_RAIL_ROUTE = 1700; 	 /* Default delay before building a new town rail route. */
const DEFAULT_DELAY_MANAGE_ROUTES = 1000; 			 /* Default delay for managing all routes. */
const DEFAULT_DELAY_HANDLE_LOAN = 2500;			 	 /* Default delay for handling our loan. */
const DEFAULT_DELAY_HANDLE_EVENTS = 100;			 /* Default delay for handling events. */
const DEFAULT_DELAY_URGENT_TASKS = 500;				 /* Default delay for urgent tasks. */



class TracAI extends AIController {
	/* Declare the variables here. */
	name = null;
	
	delay_build_air_route = 0;
	delay_build_industry_rail_route = 0;
	delay_build_town_rail_route = 0;
	delay_evaluate_aircraft = 0;
	delay_manage_routes = 0;
	delay_handle_loan = 0;
	delay_handle_events = 0;
	delay_urgent_tasks = 0;
	
	/* TracAI: New variables added. */
	/* Variables that need to be saved into a savegame. */
	_air_manager = null;
	_money_manager = null;
	_error_manager = null;
	_general_manager = null;
	_rail_manager = null;
	
	
	/* DO NOT SAVE variables below this line. These will not be saved. */ 
	loaded_from_save = false;
	ticker = null;
	aircraft_disabled_shown = 0;		/* Has the aircraft disabled in game settings message been shown (1) or not (0). */
	aircraft_max0_shown = 0;			/* Has the max aircraft is 0 in game settings message been shown. */
	trains_disabled_shown = 0;			/* Has the trains disabled in game settings message been shown (1) or not (0). */
	trains_max0_shown = 0;				/* Has the max trains is 0 in game settings message been shown. */


	/**
	 * @brief main constructor for tracAI
	 */
	constructor() {
		::ai_instance <- this;
		/* Initialize the class variables here (or later when possible). */
		this.loaded_from_save = false;
		ticker = 0;
		this.aircraft_disabled_shown = 0;
		this.aircraft_max0_shown = 0;
		this.trains_disabled_shown = 0;
		this.trains_max0_shown = 0;

		_air_manager = AirManager();
		_money_manager = MoneyManager();
		_error_manager = ErrorManager();
		_general_manager = GeneralManager();
		_rail_manager = RailManager();
		
		}
};


/**
 * @brief InitSettings initializes a number of required variables based on the gamesettings of our AI.
 */
function TracAI::InitSettings()
{
	local ai_speed = GetSetting("ai_speed");
	if (ai_speed == 1) {
		ai_speed = 3;
	}
	else if (ai_speed == 3) {
		ai_speed = 1;
	}
	this.delay_evaluate_aircraft = DEFAULT_DELAY_EVALUATE_AIRCRAFT * ai_speed;
	this.delay_build_air_route = DEFAULT_DELAY_BUILD_AIR_ROUTE * ai_speed;
	this.delay_build_industry_rail_route = DEFAULT_DELAY_BUILD_INDUSTRY_RAIL_ROUTE * ai_speed;
	this.delay_build_town_rail_route = DEFAULT_DELAY_BUILD_TOWN_RAIL_ROUTE * ai_speed;
	this.delay_manage_routes = DEFAULT_DELAY_MANAGE_ROUTES * ai_speed;
	this.delay_handle_loan = DEFAULT_DELAY_HANDLE_LOAN * ai_speed;
	this.delay_handle_events = DEFAULT_DELAY_HANDLE_EVENTS * ai_speed;
	this.delay_urgent_tasks = DEFAULT_DELAY_URGENT_TASKS * ai_speed;
	
	// N.B.: WARNING: Since the ticker MODulo is used to start after delays, all delays
	// above are assumed to be multiples of 100.
	
	/* Since autorenew can change the vehicle id it may cause trouble to have it turned on,
	 * therefore we turn it off and will renew manually in the future. */
	AICompany.SetAutoRenewStatus(false); 
}

/**
 * @brief Welcome says hello to the user and prints out it's current AI gamesettings.
 */
function TracAI::Welcome()
{
	/* Say hello to the user */
	AILog.Info("Welcome to TracAI. I am currently in development.");
	AILog.Info("- AI speed: " + GetSetting("ai_speed"));
	AILog.Info("----------------------------------");
}
/**
 * @brief Start function
 */
function TracAI::Start()
{
	
	if (_general_manager.passenger_cargo_id == -1) {
		AILog.Error("TracAI could not find the passenger cargo. Aircrafts wont be working properly");
	}

	/* Give the boy a name */
	if (!AICompany.SetName("TracAI")) {
		local i = 2;
		while (!AICompany.SetName("TracAI #" + i)) {
			i++;
		}
	}
	this.name = AICompany.GetName(AICompany.COMPANY_SELF);
	
	InitSettings();	// Initialize some AI game settings.
	Welcome();		// Write welcome and AI settings in log.
	
	if (loaded_from_save) {

		// We need to redo distance_of_route table
		foreach( veh, tile_1 in _air_manager._route_1) {
			local tile_2 = _air_manager._route_2.GetValue(veh);
			AILog.Info("Vehicle: " + veh + " tile1: " + Util.WriteTile(tile_1) + " tile2: " + Util.WriteTile(tile_2));
			AILog.Info("Distance: " + AIMap.DistanceManhattan(tile_1, tile_2));
			_air_manager.distance_of_route.rawset(veh, AIMap.DistanceManhattan(tile_1, tile_2));
		}
		/* Debugging info */
		AILog.Info("AIR ROUTES");
		AILog.Info("----------");
		_air_manager.DebugListRouteInfo();
		AILog.Info("");
		AILog.Info("RAIL ROUTES");
		AILog.Info("----------");
		_rail_manager.PrintRoutes();
		//remove unfinished routes that were not completed because of save/load
		_rail_manager.RemoveUnfinishedRoutes();
		
		foreach(idx, sign in AISignList()) {
			AISign.RemoveSign(idx);
		}
	}
	
	/* We start with almost no loan, and we take a loan when we want to build something */
	AICompany.SetLoanAmount(AICompany.GetLoanInterval());

	/* The amount of time we may sleep between loops.
	   Warning: don't change this value unless your understand the implications for all the delays! 
	*/
	local sleepingtime = SLEEPING_TIME;
	/* Factor to multiply the build delay with. */
	 local build_rail_delay_factor = 1;
	 local build_air_delay_factor = 1;
	 


	/* Let's go on for ever */
	while (true) {
		AILog.Warning("ticker: " + ticker);
		//AILog.Warning("Failed tries to find route: " + this._rail_manager.found_paths_failed_counter);
		//AILog.Warning("Money reservation: " + this._money_manager._reservations.Count() + " (should be zero)");
		//_rail_manager.PrintRoutes();
		
		/* Need to check if we can build vehicles and how many. Since this can change we do it inside the loop. */
		this.CheckSettingsForDisabledVehicles();
		if((this.aircraft_disabled_shown || this.aircraft_max0_shown) && 
		   (this.trains_disabled_shown || trains_max0_shown)) {
		   
		   AILog.Warning("We have both trains and aircraft disabled...unless you allow us our transport type we wont be doing anything.")
		} else {
			/* Handle urgent tasks once in a while */
			if (ticker % this.delay_urgent_tasks == 0 && ticker != 0) {
				AILog.Info(Helper.GetCurrentDateString() + " --- Urgent Tasks ---");
				_general_manager.ManageUrgentTasks();
				AILog.Info(Helper.GetCurrentDateString() + " --- Urgent Tasks Done ---");
			}
			
			/* Evaluate the available aircraft once in a while. */
			if ((ticker % this.delay_evaluate_aircraft == 0 || ticker == 0)) {
				_air_manager.EvaluateAircraft();
				/* This seems like a good place to show some debuggin info in case we turned
				   that setting on. */
				if (GetSetting("debug_show_lists") == 1) {
					/* Debugging info */
					_air_manager.DebugListTownsUsed();
					_air_manager.DebugListRoutes();
					_rail_manager.PrintRoutes();
				}
			}
			/* Once in a while, with enough money, try to build airport */
			if (!(aircraft_disabled_shown || aircraft_max0_shown) && (ticker % (build_air_delay_factor * this.delay_build_air_route) == 0 || ticker == 0) && _money_manager.GetAvailableMoney() >= AirManager.MINIMUM_BALANCE_BUILD_AIRPORT) {
				AILog.Warning("Building new air route");
				local ret = _air_manager.BuildAirportRoute();
				if ((ret == ErrorManager.ERROR_FIND_AIRPORT1) || (ret == ErrorManager.ERROR_MAX_AIRPORTS) ||
					(ret == ErrorManager.ERROR_MAX_AIRCRAFT) && ticker != 0) {
					/* No more route found or we have max allowed aircraft, delay even more before trying to find an other */
					build_air_delay_factor = 3;
				}
				else {
					/* Set default delay back in case we had it increased, see above. */
					build_air_delay_factor = 1;
				}
			}
			/* Once in a while, with enough money, try to build industry railroute */
			if (!(trains_disabled_shown || trains_max0_shown) && (ticker % (build_rail_delay_factor * this.delay_build_industry_rail_route) == 0 || ticker == 0) && _money_manager.GetAvailableMoney() >= RailManager.MINIMUM_BALANCE_BUILD_RAIL_ROUTE) {
				AILog.Warning("Building new industry rail route");
				this._rail_manager.CreateNewIndustryRoute();
				switch(this._error_manager.GetLastError()) {
					case ErrorManager.ERROR_NOT_ENOUGH_CASH: build_rail_delay_factor = 3;
															 break;
					case ErrorManager.ERROR_TOO_MANY_VEHICLES: build_rail_delay_factor = 3;
															   break;										   
					default: build_rail_delay_factor = 1;
				}
			}
			
			/* Once in a while, with enough money, try to build town railroute */
			if (!(trains_disabled_shown || trains_max0_shown) && (ticker % (build_rail_delay_factor * this.delay_build_town_rail_route) == 0) && ticker != 0 && _money_manager.GetAvailableMoney() >= RailManager.MINIMUM_BALANCE_BUILD_RAIL_ROUTE) {
				AILog.Warning("Building new town rail route");
				this._rail_manager.CreateNewTownRoute();
				switch(this._error_manager.GetLastError()) {
					case ErrorManager.ERROR_NOT_ENOUGH_CASH: build_rail_delay_factor = 3;
															 break;
					case ErrorManager.ERROR_TOO_MANY_VEHICLES: build_rail_delay_factor = 3;
															   break;
					default: build_rail_delay_factor = 1;
				}
			}
			
			/* Manage the routes once in a while */
			if (ticker % this.delay_manage_routes == 0) {
				if(!(aircraft_disabled_shown || aircraft_max0_shown)) _air_manager.ManageAirRoutes();
				if(!(trains_disabled_shown || trains_max0_shown)) _rail_manager.ManageRailRoutes();
			}
			/* Try to get rid of our loan once in a while */
			if (ticker % this.delay_handle_loan == 0) {
				AICompany.SetLoanAmount(0);
			}
			/* Check for events once in a while */
			if (ticker % this.delay_handle_events == 0) {
				_general_manager.HandleEvents();
			}

			
		}
		
		/* Make sure we do not create infinite loops */
		Sleep(sleepingtime);
		ticker += sleepingtime;
		
		//remove sign made in this round (only used when debugging)
		/*
		foreach(idx, sign in AISignList()) {
			AISign.RemoveSign(idx);
		}*/
		
	} // END OF OUR MAIN LOOP
}

/**
 * @brief Check setting for disabled vehicles
 */
function TracAI::CheckSettingsForDisabledVehicles() 
{
		if (AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_AIR)) {
			if (aircraft_disabled_shown == 0) {
				aircraft_disabled_shown = 1;
			}
		}
		if (Vehicle.IsVehicleTypeDisabledByAISettings(AIVehicle.VT_AIR)) {
			if (aircraft_disabled_shown == 0) {
				aircraft_disabled_shown = 1;
			}
		}
		if (Vehicle.GetVehicleLimit(AIVehicle.VT_AIR) == 0) {
			if (aircraft_max0_shown == 0) {
				aircraft_max0_shown = 1;
			}
		}
		
		if (AIGameSettings.IsDisabledVehicleType(AIVehicle.VT_RAIL)) {
			if (trains_disabled_shown == 0) {
				trains_disabled_shown = 1;
			}
		}
		if (Vehicle.IsVehicleTypeDisabledByAISettings(AIVehicle.VT_RAIL)) {
			if (trains_disabled_shown == 0) {
				trains_disabled_shown = 1;
			}
		}
		if (Vehicle.GetVehicleLimit(AIVehicle.VT_RAIL) == 0) {
			if (trains_max0_shown == 0) {
				trains_max0_shown = 1;
			}
		}
}
/**
 * @brief main Save function
 * @return saved data table
 */
function TracAI::Save()
{
   /* Debugging info */
	local MyOps1 = this.GetOpsTillSuspend();
	local MyOps2 = 0;
/* only use for debugging:
    AILog.Warning("Saving data to savegame not implemented yet!");
    AILog.Info("Ops till suspend: " + this.GetOpsTillSuspend());
    AILog.Info("");
*/
    /* Save the data */
    local data = {};
	if(this._general_manager != null) data.rawset("generalmanager", this._general_manager.Save());
	if(this._air_manager != null) data.rawset("airmanager", this._air_manager.Save());
	if(this._rail_manager != null) data.rawset("railmanager", this._rail_manager.Save());
	
    /* Debugging info 
    DebugListTownsUsed();
    DebugListRouteInfo();
*/   
/* only use for debugging:
    AILog.Info("Tick: " + this.GetTick() );
*/
    MyOps2 = this.GetOpsTillSuspend();
	if (MyOps2 < 10000) {
		AILog.Error("SAVE: Using almost all allowed ops: " + MyOps2 );
	}
	else if (MyOps2 < 20000) {
		AILog.Warning("SAVE: Using a high amount of ops: " + MyOps2 );
	}
	else {
		AILog.Info("Saving TracAI game data. Used ops: " + (MyOps1-MyOps2) );
	}
   
    return data;
 }
 
 /**
  * @brief main Load function
  * @param version version of AI
  * @param data data to load
  */
function TracAI::Load(version, data)
{
   /* Debugging info */
	local MyOps1 = this.GetOpsTillSuspend();
	local MyOps2 = 0;
	AILog.Info("Loading savegame saved by TracAI version " + version);

	if (data.rawin("airmanager")) {
		this._air_manager.Load(data.rawget("airmanager"));
	}
	
	if (data.rawin("generalmanager")) {
		this._general_manager.Load(data.rawget("generalmanager"));
	}
	
	if (data.rawin("railmanager")) {
		this._rail_manager.Load(data.rawget("railmanager"));
	}
	loaded_from_save = true;

    /* Debugging info */
    MyOps2 = this.GetOpsTillSuspend();
	if (MyOps2 < 10000) {
		AILog.Error("LOAD: Using almost all allowed ops: " + MyOps2 );
	}
	else if (MyOps2 < 20000) {
		AILog.Warning("LOAD: Using a high amount of ops: " + MyOps2 );
	}
	else {
		AILog.Info("Loading TracAI game data. Used ops: " + (MyOps1-MyOps2) );
		//AILog.Info("Loading: ops till suspend: " + MyOps2 + ", ops used in load: " + (MyOps1-MyOps2) );
	}
}
 
