/*
 * @author Luis Henrique O. Rios (trAIns) Copyright (C) 2009, modified by Daniela Plachtova 
 * @file money_manager.nut
 * @note original licence can be found in licence.txt
 */

 /** 
  * @brief class MoneyManager 
  */
class MoneyManager 
{
	_total_money_reserved = null;		/* total money reserved */
	_reservations = null;				/* list of all current reservations */
	_reservation_id = null;				/* reservation id */
	
	/**
	 * @brief constructor for MoneyManager
	 */
	constructor()
	{
		this._reservation_id = this._total_money_reserved = 0;
		this._reservations = AIList();
	}
	
	/**
	 * @brief Get Money Available
	 * @return company available money
	 */
	function GetAvailableMoney();
	
	/**
	 * @brief Get Amount Reserved 
	 * @param reservation_id reservation id
	 * @return amount of reserved money for reservation_id
	 */
	function GetAmountReserved(reservation_id);
	/**
	 * @brief Release Reservation
	 * @param reservation_id reservation id
	 */
	function ReleaseReservation(reservation_id);
	/**
	 * @brief Reserve Money
	 * @param min_amount_to_reserve minimal amount of money to reserve
	 * @param max_amount_to_reserve maximal amount of money to reserve
	 */
	function ReserveMoney(min_amount_to_reserve = 0 , max_amount_to_reserve = 0);

	/**
	 * @brief Tries to repay loan
	 */
	function RepayLoan();
}

function MoneyManager::ReserveMoney(min_amount_to_reserve = 0 , max_amount_to_reserve = 0)
{
	local amount_to_reserve;
	local bank_balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
	local loan_amount = AICompany.GetLoanAmount();
	local max_loan_amount = AICompany.GetMaxLoanAmount();
	local total_money_available = bank_balance + max_loan_amount - loan_amount - _total_money_reserved;
	assert(min_amount_to_reserve <= max_amount_to_reserve && min_amount_to_reserve >= 0);

	if(min_amount_to_reserve == 0){
		if(total_money_available <= 0){
			RepayLoan();
			return null;
		}
		AICompany.SetMinimumLoanAmount(max_loan_amount);
		amount_to_reserve = total_money_available;
	}else{
		if(total_money_available < min_amount_to_reserve){
			RepayLoan();
			return null;
		}
		amount_to_reserve = min(max_amount_to_reserve , total_money_available);
		/* Check if it is necessary to loan some money. */
		if(bank_balance - this._total_money_reserved < amount_to_reserve){
			AICompany.SetMinimumLoanAmount(loan_amount + (this._total_money_reserved + amount_to_reserve - bank_balance));
		}
	}
	this._reservations.AddItem(this._reservation_id , amount_to_reserve);
	this._total_money_reserved += amount_to_reserve;
	return this._reservation_id++;
}

function MoneyManager::ReleaseReservation(reservation_id)
{
	local amount_reserved;
	assert(reservation_id != null && this._reservations.HasItem(reservation_id));
	amount_reserved = this._reservations.GetValue(reservation_id);
	this._reservations.RemoveItem(reservation_id);
	this._total_money_reserved -= amount_reserved;
	if(this._reservations.Count() == 0) this._reservation_id = 0;
	RepayLoan();
}

function MoneyManager::RepayLoan()
{
	local bank_balance = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
	local available_money = bank_balance - (1.5 * this._total_money_reserved).tointeger();
	local loan_amount = AICompany.GetLoanAmount();
	local loan_interval = AICompany.GetLoanInterval();

	if(loan_amount > 0 && available_money > 0){
		local amount_to_pay = min(available_money , loan_amount);
		amount_to_pay -= (amount_to_pay % loan_interval);
		if(amount_to_pay > 0){
			amount_to_pay = loan_amount - amount_to_pay;
			AICompany.SetLoanAmount(amount_to_pay);
		}
	}else if(available_money <= 0){
		local amount_to_loan = (-available_money) + (loan_interval - ((-available_money) % loan_interval));
		assert(amount_to_loan % loan_interval == 0);
		if(loan_amount + amount_to_loan <= AICompany.GetMaxLoanAmount())
			AICompany.SetLoanAmount(loan_amount + amount_to_loan);
	}
}

function MoneyManager::GetAmountReserved(reservation_id)
{
	assert(reservation_id != null && this._reservations.HasItem(reservation_id));
	return this._reservations.GetValue(reservation_id);
}

function MoneyManager::GetAvailableMoney()
{/*
	AILog.Info("BankBalance: " + AICompany.GetBankBalance(AICompany.COMPANY_SELF) + 
	           " max loan: " + AICompany.GetMaxLoanAmount() + 
			   " loan amount: " + AICompany.GetLoanAmount() + 
			   " total_money_reserved: " + total_money_reserved);
			   */
	local aux = AICompany.GetBankBalance(AICompany.COMPANY_SELF) + (AICompany.GetMaxLoanAmount() - AICompany.GetLoanAmount()) - this._total_money_reserved;
	return aux >= 0 ? aux : 0;
}
