/**
 * @author Daniela Plachtova
 * @file rail_builder.nut
 */ 

 
 /**
  * @brief static class RailBuilder manages all rail building functions
  */
class RailBuilder
{
	/**
	 * @brief Checks if we can build this path
	 * @param route_id route id
	 * @param path path to check
	 * @return true if we can build this path else false
	 */
	function CanBuildPath(route_id, path);
	/**
	 * @brief Get aproximate cost of path
	 * @param route_id route id
	 * @param path path to estimate the cost
	 * @return cost of path
	 */
	function GetCostOfPath(route_id, path);
	
	/**
	 * @brief build path
	 * @param route_id route id
	 * @param path path to build
	 * @return true if building a path suceeded else false
	 * @note taken from AdmiralAI, modified
	 */
	function BuildPath(route_id, path);
	
	/**
	 * @brief Checks if new can build industry station with nearby station tracks
	 * @param industry industry id where we want to build station
	 * @param station_tile_list list of suitable spots where we can build station
	 * @param other_industry_tile location of other industry
	 * @param platform_length platform length
	 * @param number_of_tracks number of tracks
	 * @param cargo cargo id of industry
	 * @param accept is accepting cargo or producing
	 * @param radius coverage radius of trains
	 * @true true if we can build indutry station else false
	 */
	function CanBuildIndustryStationWithStationTracks(industry, station_tile_list, other_industry_tile, platform_length, number_of_tracks, cargo, accept, radius);
	
	/**
	 * @brief Build industry station with nearby station tracks
	 * @param industry industry id where we want to build station
	 * @param station_tile_list list of suitable spots where we can build station
	 * @param other_industry_tile location of other industry
	 * @param platform_length platform length
	 * @param number_of_tracks number of tracks
	 * @param cargo cargo id of industry
	 * @param accept is accepting cargo or producing
	 * @param radius coverage radius of trains
	 * @return true if we have build industry station else false
	 */
	function BuildIndustryStationWithStationTracks(industry, station_tile_list, other_industry_tile, platform_length, number_of_tracks, cargo, accept, radius)
	
	/**
	 * @brief Get aproximate cost of industry station with nearby station tracks
	 * @param industry industry id where we want to build station
	 * @param station_tile_list list of suitable spots where we can build station
	 * @param other_industry_tile location of other industry
	 * @param platform_length platform length
	 * @param number_of_tracks number of tracks
	 * @param cargo cargo id of industry
	 * @param accept is accepting cargo or producing
	 * @param radius coverage radius of trains
	 * @return aproximate cost of industry station
	 */
	function GetCostOfIndustryStationWithStationTracks(industry, station_tile_list, other_industry_tile, platform_length, number_of_tracks, cargo, accept, radius)
	
	/**
	 * @brief Checks if new can build town station with nearby station tracks
	 * @param town town id where we want to build station
	 * @param station_tile_list list of suitable spots where we can build station
	 * @param other_industry_tile location of other industry
	 * @param platform_length platform length
	 * @param number_of_tracks number of tracks
	 * @param cargo cargo id of industry
	 * @param radius coverage radius of trains
	 * @true true if we can build town station else false
	 */
	function CanBuildTownStationWithStationTracks(town, station_tile_list, other_town_tile, platform_length, number_of_tracks, cargo, radius)
	
	/**
	 * @brief Build town station with nearby station tracks
	 * @param town town id where we want to build station
	 * @param station_tile_list list of suitable spots where we can build station
	 * @param other_town_tile location of other town
	 * @param platform_length platform length
	 * @param number_of_tracks number of tracks
	 * @param cargo cargo id of industry
	 * @param radius coverage radius of trains
	 * @return true if we have build town station else false
	 */
	function BuildTownStationWithStationTracks(town, station_tile_list, other_town_tile, platform_length, number_of_tracks, cargo, radius)
	/**
	 * @brief Get aproximate cost of town station with nearby station tracks
	 * @param town town id where we want to build station
	 * @param station_tile_list list of suitable spots where we can build station
	 * @param other_town_tile location of other town
	 * @param platform_length platform length
	 * @param number_of_tracks number of tracks
	 * @param cargo cargo id of industry
	 * @param radius coverage radius of trains
	 * @return aproximate cost of town station
	 */
	function GetCostOfTownStationWithStationTracks(town, station_tile_list, other_town_tile, platform_length, number_of_tracks, cargo, radius);
	
	/**
	 * @brief Get suitable spots for building Industry Station
	 * @param industry industry id
	 * @param other_industry other industry we want to build path
	 * @param platform_length platform length
	 * @param number_of_tracks number of tracks
	 * @param cargo cargo id of industry
	 * @param accept is accepting cargo or producing
	 * @param radius coverage radius of trains
	 * @return list of suitable industry station spots
	 */
	function GetIndustryStationSpotList(industry, other_industry, platform_length, number_of_tracks, cargo, accept, radius)
	
	/**
	 * @brief Get suitable spots for building Town Station
	 * @param town town id
	 * @param other_town other town we want to build path
	 * @param platform_length platform length
	 * @param number_of_tracks number of tracks
	 * @param radius coverage radius of trains
	 * @return list of suitable industry station spots
	 */
	function GetTownStationSpotList(town, other_town, platform_length, number_of_tracks, radius)
	
	/**
	 * @brief Index Station tracks (into Rail Tile info list), it is seperate function because we also need to index existing station_exit_dir_list
	 * @param exit_tile_pair station exit tile pair
	 * @param exit_direction direction from which station exits
	 */
	function IndexStationTracks(route_id, exit_tile_pair, exit_direction) 
	/**
	 * @brief Build depot nerby specified path
	 * @param route_id route id
	 * @param path path to buidl depot to
	 * @return depot tile or null ibuilding depot not suceeded
	 * @note taken from AdmiralAI, modified
	 */
	function BuildDepot(route_id, path);
	
	/**
	 * @brief Signal Path
	 * @param path path to signal
	 */
	function SignalPath(path);
	/**
	 * @brief Check if Signal are OK, fix if they are not
	 * @param path path to check
	 * @note mostly fix start and end signals (they are too close together)
	 */
	function CheckSignalPath(path);
	
	
	/**
	 * @brief Build nearby station tracks
	 * @param exit_tile_pair exit tile pair of station
	 * @param exit_direction direction the station is pointing to
	 * @return true if building nearby station tracks suceeded else false
	 */
	function BuildStationTracks(exit_tile_pair, exit_direction);
	
	/**
	 * @brief Build nearby station tracks signals
	 * @param exit_tile_pair exit tile pair of station
	 * @param exit_direction direction the station is pointing to
	 * @return true if building nearby station tracks signals suceeded else false
	 */
	function BuildStationTracksSignals(exit_tile_pair, exit_direction);
		
	/**
	 * @brief Is station buildable on specified direction
	 * @param direction direction supposed station will point to
     * @param tile tile to check
     * @param platform_length platform length
     * @param number_of_tracks numebr of tracks
     * @return true if we can build station on specified tile and direction
     */	 
	function IsExitDirectionOfStationBuildable(direction, tile, platform_length, number_of_tracks);
	
	/**
	 * @brief Get Direction priority of station we want to build
	 * @param station_tile possible station tile
	 * @param other_industry_tile tile of the other industry
	 * @param platform_length platform length
	 * @param number_of_tracks number of tracks
	 * @param cargo cargo id of industry
	 * @param accept is accepting cargo or producing station
	 * @param radius coverage radius of trains
	 * @return array of direction priority of building station
	 */	 
	function GetDirectionPriority(station_tile, other_industry_tile, platform_length, number_of_tracks, cargo, accept, radius)
	
	
	/**
	 * @brief connect depot diagonally
	 * @param route_id route id
	 * @param tile_a tile a
	 * @param tile_b tile b
	 * @param tile_c tile c
	 * @return depot tile or null if building depot not suceeded
	 * @note taken from AdmiralAI, modified
	 */
	function ConnectDepotDiagonal(route_id, tile_a, tile_b, tile_c);
	
	/**
	 * @brief Signal Path Advanced
	 * @param path path to signal
	 * @param skip skip interval
	 * @param end end of path
	 * @param signal_count_limit limit of build signals
	 * @return number of signals we have build
	 * @note taken from AdmiralAI, modified
	 */
	function SignalPathAdvanced(path, skip, end, signal_count_limit);
}

function RailBuilder::CanBuildPath(route_id, path) 
{
	local test = AITestMode();
	local value = RailBuilder.BuildPath(route_id, path);	
	return value;
}

function RailBuilder::GetCostOfPath(route_id, path)
{
	local cost = AIAccounting();
	cost.ResetCosts();
	RailBuilder.CanBuildPath(route_id, path);
	return cost.GetCosts();
}

function RailBuilder::BuildPath(route_id, path)
{
	//get current route id we are building
	if (path == null) return false;
	local prev = null;
	local prevprev = null;
	local orig_path = path;
	local railtrack_prev_before = null;
	local railtrack = null;
	
	while (path != null) {
		//AISign.BuildSign(path.GetTile(), "" + path.GetCost());
		
		//get railtrack bitmask before we build new railtrack
		if(prev != null) railtrack_prev_before = AIRail.GetRailTracks(prev);
		if(railtrack_prev_before == 0) AILog.Warning("Railtrack prev " + Util.WriteTile() + " before is 0 " + AIRail.IsRailTile(prev));
		if (prevprev != null) {
			// if distance betweem two tile is > 1 we can build either tunnel or bridge
			if (AIMap.DistanceManhattan(prev, path.GetTile()) > 1) {
				//check we we can build tunnel (next tile is other tunnel end)
				if (AITunnel.GetOtherTunnelEnd(prev) == path.GetTile()) {
					//check if there already exists a tunnel
					if (AITunnel.IsTunnelTile(prev)) {
						//if rail type is not compatible, convert
						if (AIRail.GetCurrentRailType() != AIRail.GetRailType(prev)) {
							assert(AIRail.TrainHasPowerOnRail(AIRail.GetRailType(prev), AIRail.GetCurrentRailType()));
							if (!AIRail.ConvertRailType(prev, prev, AIRail.GetCurrentRailType())) {
								AILog.Error("4 " + AIError.GetLastErrorString());
								return false;
							} else {
								//if this is not the same rail type we are in test mode
								if (AIRail.GetCurrentRailType() == AIRail.GetRailType(prev)) {
									::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(prev, RailDemolisher.RAILTILE_TYPE_TUNNEL, route_id, AIRail.RAILTRACK_INVALID);
									::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(AITunnel.GetOtherTunnelEnd(prev), RailDemolisher.RAILTILE_TYPE_TUNNEL, route_id, AIRail.RAILTRACK_INVALID);
								}
							}
						}
					//no tunnel exists, try to build one
					} else if (!AITunnel.BuildTunnel(AIVehicle.VT_RAIL, prev)) {
						AILog.Error("1 " + AIError.GetLastErrorString());
						return false;
					} else {
						//if there is not tunnel build we are in test mode
						if(AITunnel.IsTunnelTile(prev)) {
							::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(prev, RailDemolisher.RAILTILE_TYPE_TUNNEL, route_id, AIRail.RAILTRACK_INVALID);
							::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(AITunnel.GetOtherTunnelEnd(prev), RailDemolisher.RAILTILE_TYPE_TUNNEL, route_id, AIRail.RAILTRACK_INVALID);

						}
					}
				//building a bridge
				} else {
					//check if there already exists a bridge
					if (AIBridge.IsBridgeTile(prev)) {
						//if rail type is not compatible, convert
						if (AIRail.GetCurrentRailType() != AIRail.GetRailType(prev)) {
							assert(AIRail.TrainHasPowerOnRail(AIRail.GetRailType(prev), AIRail.GetCurrentRailType()));
							if (!AIRail.ConvertRailType(prev, prev, AIRail.GetCurrentRailType())) {
								AILog.Error("5 " + AIError.GetLastErrorString());
								return false;
							} else {
								//if this is not the same rail type we are in test mode
								if (AIRail.GetCurrentRailType() == AIRail.GetRailType(prev)) {
									::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(prev, RailDemolisher.RAILTILE_TYPE_BRIDGE, route_id, AIRail.RAILTRACK_INVALID);
									::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(AIBridge.GetOtherBridgeEnd(prev), RailDemolisher.RAILTILE_TYPE_BRIDGE, route_id, AIRail.RAILTRACK_INVALID);
								}
							}
						}
					//no bridge exists, try to build one
					} else {
						local bridge_list = AIBridgeList_Length(AIMap.DistanceManhattan(path.GetTile(), prev) + 1);
						bridge_list.Valuate(AIBridge.GetMaxSpeed);
						bridge_list.Sort(AIList.SORT_BY_VALUE, AIList.SORT_DESCENDING);
						if (!AIBridge.BuildBridge(AIVehicle.VT_RAIL, bridge_list.Begin(), prev, path.GetTile())) {
							AILog.Error("2 " + AIError.GetLastErrorString());
							return false;
						} else {
							//if there is not bridge build we are in test mode
							if(AIBridge.IsBridgeTile(prev)) {
								::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(prev, RailDemolisher.RAILTILE_TYPE_BRIDGE, route_id, AIRail.RAILTRACK_INVALID);
								::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(AIBridge.GetOtherBridgeEnd(prev), RailDemolisher.RAILTILE_TYPE_BRIDGE, route_id, AIRail.RAILTRACK_INVALID);

							}
						}
					}
				}
				//move to the next tile
				prevprev = prev;
				prev = path.GetTile();
				path = path.GetParent();
			//distance is  == 1 we are building normal railtrack
			} else {
				//if there is railtrack that is not mine abort building
				if (AIRail.IsRailTile(prev) && !AICompany.IsMine(AITile.GetOwner(prev))) return false;
				//check if there is existing railtrack (tiles are connected)
				if (!AIRail.AreTilesConnected(prevprev, prev, path.GetTile())) {
					//no railtrack exists, try to build one
					if (!AIRail.BuildRail(prevprev, prev, path.GetTile())) {
						local num_tries = 20;
						local ok = false;
						while (AIError.GetLastError() == AIError.ERR_VEHICLE_IN_THE_WAY && num_tries-- > 0) {
							AIController.Sleep(74);
							if (AIRail.BuildRail(prevprev, prev, path.GetTile())) { 
								local railtrack;	
								//we need to find out which railtrack we build (using xor) 							
								if(railtrack_prev_before == AIRail.RAILTRACK_INVALID) {
									railtrack = AIRail.GetRailTracks(prev);
								} else railtrack = railtrack_prev_before ^ AIRail.GetRailTracks(prev);
								if(railtrack == 0) AILog.Warning("Railtrack prev " + Util.WriteTile(prev) + " iss 0 " + AIRail.IsRailTile(prev) + " " + railtrack_prev_before + " " + AIRail.GetRailTracks(prev));
								//if railtrack RAILTRACK_INVALID or 0 we are in test mode (because building track suceeded but still did not build anything)
								if(railtrack != AIRail.RAILTRACK_INVALID && railtrack != 0) {
									::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(prev, RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, railtrack);
								}
								ok = true; 
								break; 
							}
						} 
						//if we did not suceeded in building railtrack mark last three tiles and return
						if (!ok) {
							AILog.Error(prevprev + "   " + prev + "   " + path.GetTile());
							AILog.Error("3 " + AIError.GetLastErrorString());
							if (AIController.GetSetting("debug_signs")) {
								AISign.BuildSign(prevprev, "1");
								AISign.BuildSign(prev, "2");
								AISign.BuildSign(path.GetTile(),  "3");
							}
							return false;
						}
					// we suceeded building railtrack on first time
					} else {
						if(railtrack_prev_before == AIRail.RAILTRACK_INVALID) {
							railtrack = AIRail.GetRailTracks(prev);
						} else railtrack = railtrack_prev_before ^ AIRail.GetRailTracks(prev);
						if(railtrack == 0) AILog.Warning("Railtrack prev " + Util.WriteTile(prev) + " iss 0 " + AIRail.IsRailTile(prev) + " " + railtrack_prev_before + " " + AIRail.GetRailTracks(prev));
						//if railtrack RAILTRACK_INVALID or 0 we are in test mode (because building track suceeded but still did not build anything)
						if(railtrack != AIRail.RAILTRACK_INVALID && railtrack != 0) ::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(prev, RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, railtrack);
								
					}
				//railtrack already exists, check if railtrack type is compatible
				} else if (AIRail.GetRailType(prev) != AIRail.GetCurrentRailType()) {
					assert(AIRail.TrainHasPowerOnRail(AIRail.GetRailType(prev), AIRail.GetCurrentRailType()));
					//railtrack type is not compatible, try to convert
					if (!AIRail.ConvertRailType(prev, prev, AIRail.GetCurrentRailType())) {
						AILog.Error("6 " + AIError.GetLastErrorString());
						if(AIController.GetSetting("debug_signs")) AISign.BuildSign(prev, "!! " + AIRail.GetRailType(prev) + "  " + AIRail.GetCurrentRailType());
						return false;
					} else {
						local rail_tile_info = ::ai_instance._rail_manager._rail_demolisher.GetRailTileInfo(prev);
						local railtracks = ::ai_instance._rail_manager._rail_demolisher.GetAllRailtracksOnTileOfThisRoute(rail_tile_info, 0);
						foreach(railtrack in railtracks) {
							if(AIRail.GetRailType(prev) == AIRail.GetCurrentRailType()) ::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(prev, RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, railtrack);
						}
					}
				}
			}
		}
		if (path != null) {
			//add existing tiles to tile list
			if(prev != null) {
				if(AITunnel.IsTunnelTile(prev) && AICompany.IsMine(AITile.GetOwner(prev))) {
					::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(prev, RailDemolisher.RAILTILE_TYPE_TUNNEL, route_id, AIRail.RAILTRACK_INVALID);
					::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(AITunnel.GetOtherTunnelEnd(prev), RailDemolisher.RAILTILE_TYPE_TUNNEL, route_id, AIRail.RAILTRACK_INVALID);
				} else if(AIBridge.IsBridgeTile(prev) && AICompany.IsMine(AITile.GetOwner(prev))) {
					::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(prev, RailDemolisher.RAILTILE_TYPE_BRIDGE, route_id, AIRail.RAILTRACK_INVALID);
					::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(AIBridge.GetOtherBridgeEnd(prev), RailDemolisher.RAILTILE_TYPE_BRIDGE, route_id, AIRail.RAILTRACK_INVALID);
				} else if(AIRail.IsRailTile(prev) && AICompany.IsMine(AITile.GetOwner(prev))) {
					local rail_tile_info = ::ai_instance._rail_manager._rail_demolisher.GetRailTileInfo(prev);
					local railtracks = ::ai_instance._rail_manager._rail_demolisher.GetAllRailtracksOnTileOfThisRoute(rail_tile_info, 0);
					foreach(railtrack in railtracks) {
						::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(prev, rail_tile_info._type, route_id, railtrack);
					}
				}
			}
			prevprev = prev;
			prev = path.GetTile();
			path = path.GetParent();
		}
	}
	return true;
}
		
		
function RailBuilder::CanBuildIndustryStationWithStationTracks(industry, station_tile_list, other_industry_tile, platform_length, number_of_tracks, cargo, accept, radius)
{
	local test = AITestMode();
	local value =  RailBuilder.BuildIndustryStationWithStationTracks(industry, station_tile_list, other_industry_tile, platform_length, number_of_tracks, cargo, accept, radius) ? true : false;
	return value;
	
}

function RailBuilder::GetCostOfIndustryStationWithStationTracks(industry, station_tile_list, other_industry_tile, platform_length, number_of_tracks, cargo, accept, radius)
{
	local test = AITestMode();
	local cost = AIAccounting();
	cost.ResetCosts();
	RailBuilder.BuildIndustryStationWithStationTracks(industry, station_tile_list, other_industry_tile, platform_length, number_of_tracks, cargo, accept, radius)
	return cost.GetCosts();
}

function RailBuilder::BuildIndustryStationWithStationTracks(industry, station_tile_list, other_industry_tile, platform_length, number_of_tracks, cargo, accept, radius) 
{
	local radius = AIStation.GetCoverageRadius(AIStation.STATION_TRAIN);
	local station_exit_dir_list;
	local exit_tile_pair = {};
	local distance = AITile.GetDistanceManhattanToTile(AIIndustry.GetLocation(industry), other_industry_tile);
	local suceeded = false;
	local station_info = null;
	
	//iterate over spots we want to build station at
	for(local tile = station_tile_list.Begin(); !station_tile_list.IsEnd(); tile = station_tile_list.Next()) {
		station_exit_dir_list = RailBuilder.GetDirectionPriority(tile, other_industry_tile, platform_length, number_of_tracks, cargo, accept, radius);
		station_exit_dir_list.reverse(); //reverse priority so we can push items
		while(station_exit_dir_list.len() != 0) {
			local tmpdir = station_exit_dir_list.pop();
			//AILog.Info("Tmp dir: " + Direction.ToString(tmpdir));
			
			//building NE-SW station 
			if(tmpdir == Direction.NORTHEAST || tmpdir == Direction.SOUTHWEST) {
				if(!AIRail.BuildNewGRFRailStation(tile, AIRail.RAILTRACK_NE_SW, number_of_tracks, 
				    platform_length, AIStation.STATION_NEW, cargo, AIIndustryType.INDUSTRYTYPE_UNKNOWN,
					AIIndustryType.INDUSTRYTYPE_UNKNOWN, distance, !accept)) {
					
					if(AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES) {
						::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_LOCAL_AUTHORITY_REFUSES);
						return null;
					}
					AILog.Info("Build NewGRF station not suceeded");
					
					if(AIRail.BuildRailStation(tile, AIRail.RAILTRACK_NE_SW, number_of_tracks, 
				    platform_length, AIStation.STATION_NEW)) suceeded = true;
					else {	
						AILog.Info("Build station not suceeded ");
						if(AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES) {
							::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_LOCAL_AUTHORITY_REFUSES);
							return null;
						}
					}
				} else suceeded = true;
				//building station suceeded
				if(suceeded) {
					if(tmpdir == Direction.NORTHEAST) {
						exit_tile_pair.right <- tile + AIMap.GetTileIndex(0,number_of_tracks - 1);
						exit_tile_pair.left <- tile;				
					} else {
						exit_tile_pair.right <- tile + AIMap.GetTileIndex(platform_length - 1,0);
						exit_tile_pair.left <- tile + AIMap.GetTileIndex(platform_length - 1,number_of_tracks - 1);
					}
					station_info = IndustryStationInfo(AIStation.GetStationID(tile), platform_length, number_of_tracks, tmpdir, exit_tile_pair, industry, accept);

					if(!RailBuilder.BuildStationTracks(station_info._exit_tile_pair, station_info._exit_direction)) {
						RailDemolisher.DemolishStation(station_info._stationID);
						return null;
					}
					return station_info;
				}
				//Building NW-SE station
			} else if(tmpdir == Direction.NORTHWEST || tmpdir == Direction.SOUTHEAST) {
				if(!AIRail.BuildNewGRFRailStation(tile, AIRail.RAILTRACK_NW_SE, number_of_tracks, 
				    platform_length, AIStation.STATION_NEW, cargo, AIIndustryType.INDUSTRYTYPE_UNKNOWN,
					AIIndustryType.INDUSTRYTYPE_UNKNOWN, distance, !accept)) {
					if(AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES) {
						::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_LOCAL_AUTHORITY_REFUSES);
						return null;
					}
					AILog.Info("Build NewGRF station not suceeded");
					
					if(AIRail.BuildRailStation(tile, AIRail.RAILTRACK_NW_SE, number_of_tracks, 
				    platform_length, AIStation.STATION_NEW)) suceeded = true;
					else {
						AILog.Info("Build station not suceeded ");
						if(AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES) {
							::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_LOCAL_AUTHORITY_REFUSES);
							return null;
						}
					}	
				} else suceeded = true;
				//building station suceeded
				if(suceeded) {
					if(tmpdir == Direction.NORTHWEST) {
						exit_tile_pair.right <- tile;
						exit_tile_pair.left <- tile + AIMap.GetTileIndex(number_of_tracks - 1, 0);				
					} else {
						exit_tile_pair.right <- tile + AIMap.GetTileIndex(number_of_tracks - 1, platform_length - 1);
						exit_tile_pair.left <- tile + AIMap.GetTileIndex(0, platform_length - 1);
					}
					station_info = IndustryStationInfo(AIStation.GetStationID(tile), platform_length, number_of_tracks, tmpdir, exit_tile_pair, industry, accept);
					if(!RailBuilder.BuildStationTracks(station_info._exit_tile_pair, station_info._exit_direction)) {
						RailDemolisher.DemolishStation(station_info._stationID);
						return null;
					}
					return station_info;
				}
			}
		}
	}
	return null;
}

		
function RailBuilder::CanBuildTownStationWithStationTracks(town, station_tile_list, other_town_tile, platform_length, number_of_tracks, cargo, radius)
{
	local test = AITestMode();
	local value =  RailBuilder.BuildTownStationWithStationTracks(town, station_tile_list, other_town_tile, platform_length, number_of_tracks, cargo, radius) ? true : false;
	return value;
	
}

function RailBuilder::GetCostOfTownStationWithStationTracks(town, station_tile_list, other_town_tile, platform_length, number_of_tracks, cargo, radius)
{
	local test = AITestMode();
	local cost = AIAccounting();
	cost.ResetCosts();
	RailBuilder.BuildTownStationWithStationTracks(town, station_tile_list, other_town_tile, platform_length, number_of_tracks, cargo, radius)
	return cost.GetCosts();
}


function RailBuilder::BuildTownStationWithStationTracks(town, station_tile_list, other_town_tile, platform_length, number_of_tracks, cargo, radius) 
{
	local radius = AIStation.GetCoverageRadius(AIStation.STATION_TRAIN);
	local station_exit_dir_list;
	local exit_tile_pair = {};
	local distance = AITile.GetDistanceManhattanToTile(AITown.GetLocation(town), other_town_tile);
	local suceeded = false;
	local station_info = null;
	
	//iterate over spots we want to build station at
	for(local tile = station_tile_list.Begin(); !station_tile_list.IsEnd(); tile = station_tile_list.Next()) {
		station_exit_dir_list = RailBuilder.GetDirectionPriority(tile, other_town_tile, platform_length, number_of_tracks, cargo, true, radius);
		station_exit_dir_list.reverse(); //reverse priority so we can push items
		while(station_exit_dir_list.len() != 0) {
			local tmpdir = station_exit_dir_list.pop();
			//AILog.Info("Tmp dir: " + Direction.ToString(tmpdir));
			
			//Building NE-SW station
			if(tmpdir == Direction.NORTHEAST || tmpdir == Direction.SOUTHWEST) {
				if(!AIRail.BuildNewGRFRailStation(tile, AIRail.RAILTRACK_NE_SW, number_of_tracks, 
				    platform_length, AIStation.STATION_NEW, cargo, AIIndustryType.INDUSTRYTYPE_TOWN,
					AIIndustryType.INDUSTRYTYPE_TOWN, distance, true)) {
					if(AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES) {
						::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_LOCAL_AUTHORITY_REFUSES);
						return null;
					}
					AILog.Info("Build NewGRF station not suceeded");
					if(AIRail.BuildRailStation(tile, AIRail.RAILTRACK_NE_SW, number_of_tracks, 
				    platform_length, AIStation.STATION_NEW)) suceeded = true;
					else {
						AILog.Info("Build station not suceeded ");
						if(AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES) {
							::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_LOCAL_AUTHORITY_REFUSES);
							return null;
						}
					}
				
				} else suceeded = true;
				//building station suceeded
				if(suceeded) {
					if(tmpdir == Direction.NORTHEAST) {
						exit_tile_pair.right <- tile + AIMap.GetTileIndex(0,number_of_tracks - 1);
						exit_tile_pair.left <- tile;				
					} else {
						exit_tile_pair.right <- tile + AIMap.GetTileIndex(platform_length - 1,0);
						exit_tile_pair.left <- tile + AIMap.GetTileIndex(platform_length - 1,number_of_tracks - 1);
					}
					station_info = TownStationInfo(AIStation.GetStationID(tile), platform_length, number_of_tracks, tmpdir, exit_tile_pair, town);

					if(!RailBuilder.BuildStationTracks(station_info._exit_tile_pair, station_info._exit_direction)) {
						RailDemolisher.DemolishStation(station_info._stationID);
						return null;
					}
					return station_info;
				}
				//Building NW-SE station
			} else if(tmpdir == Direction.NORTHWEST || tmpdir == Direction.SOUTHEAST) {
				if(!AIRail.BuildNewGRFRailStation(tile, AIRail.RAILTRACK_NW_SE, number_of_tracks, 
				    platform_length, AIStation.STATION_NEW, cargo, AIIndustryType.INDUSTRYTYPE_TOWN,
					AIIndustryType.INDUSTRYTYPE_TOWN, distance, true)) {
					if(AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES) {
						::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_LOCAL_AUTHORITY_REFUSES);
						return null;
					}
					AILog.Info("Build NewGRF station not suceeded");
					
					if(AIRail.BuildRailStation(tile, AIRail.RAILTRACK_NW_SE, number_of_tracks, 
				    platform_length, AIStation.STATION_NEW)) suceeded = true;
					else {
						AILog.Info("Build station not suceeded ");
						if(AIError.GetLastError() == AIError.ERR_LOCAL_AUTHORITY_REFUSES) {
							::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_LOCAL_AUTHORITY_REFUSES);
							return null;
						}
					}
				
				} else suceeded = true;
				//building station suceeded
				if(suceeded) {
					if(tmpdir == Direction.NORTHWEST) {
						exit_tile_pair.right <- tile;
						exit_tile_pair.left <- tile + AIMap.GetTileIndex(number_of_tracks - 1, 0);				
					} else {
						exit_tile_pair.right <- tile + AIMap.GetTileIndex(number_of_tracks - 1, platform_length - 1);
						exit_tile_pair.left <- tile + AIMap.GetTileIndex(0, platform_length - 1);
					}
					station_info = TownStationInfo(AIStation.GetStationID(tile), platform_length, number_of_tracks, tmpdir, exit_tile_pair, town);
					if(!RailBuilder.BuildStationTracks(station_info._exit_tile_pair, station_info._exit_direction)) {
						RailDemolisher.DemolishStation(station_info._stationID);
						return null;
					}
					return station_info;
				}
			}
		}
	}
	return null;
}

function RailBuilder::IndexStationTracks(route_id, exit_tile_pair, exit_direction) 
{

	if(exit_direction == Direction.NORTHEAST) {
		::ai_instance._rail_manager._rail_demolisher.AddRouteTilesIntoRailTilesList([exit_tile_pair.right - AIMap.GetTileIndex(1,0), exit_tile_pair.right - AIMap.GetTileIndex(2,0), exit_tile_pair.right - AIMap.GetTileIndex(3,0)], 
																					RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NE_SW);									
		::ai_instance._rail_manager._rail_demolisher.AddRouteTilesIntoRailTilesList([exit_tile_pair.left - AIMap.GetTileIndex(1,0), exit_tile_pair.left - AIMap.GetTileIndex(2,0), exit_tile_pair.left - AIMap.GetTileIndex(3,0)], 
																					RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NE_SW);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.right - AIMap.GetTileIndex(2,0), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NW_SW);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.right - AIMap.GetTileIndex(2,0), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NW_NE);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.left - AIMap.GetTileIndex(2,0), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_SW_SE);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.left - AIMap.GetTileIndex(2,0), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NE_SE);												
	} else if(exit_direction == Direction.SOUTHWEST) {
		::ai_instance._rail_manager._rail_demolisher.AddRouteTilesIntoRailTilesList([exit_tile_pair.right + AIMap.GetTileIndex(1,0), exit_tile_pair.right + AIMap.GetTileIndex(2,0), exit_tile_pair.right + AIMap.GetTileIndex(3,0)], 
																					RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NE_SW);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTilesIntoRailTilesList([exit_tile_pair.left + AIMap.GetTileIndex(1,0), exit_tile_pair.left + AIMap.GetTileIndex(2,0), exit_tile_pair.left + AIMap.GetTileIndex(3,0)], 
																					RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NE_SW);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.left + AIMap.GetTileIndex(2,0), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NW_SW);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.left + AIMap.GetTileIndex(2,0), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NW_NE);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.right + AIMap.GetTileIndex(2,0), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_SW_SE);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.right + AIMap.GetTileIndex(2,0), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NE_SE);											
	} else if(exit_direction == Direction.NORTHWEST) {
		::ai_instance._rail_manager._rail_demolisher.AddRouteTilesIntoRailTilesList([exit_tile_pair.right - AIMap.GetTileIndex(0,1), exit_tile_pair.right - AIMap.GetTileIndex(0,2), exit_tile_pair.right - AIMap.GetTileIndex(0,3)], 
																					RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NW_SE);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTilesIntoRailTilesList([exit_tile_pair.left - AIMap.GetTileIndex(0,1), exit_tile_pair.left - AIMap.GetTileIndex(0,2), exit_tile_pair.left - AIMap.GetTileIndex(0,3)], 
																					RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NW_SE);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.left - AIMap.GetTileIndex(0,2), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NE_SE);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.left - AIMap.GetTileIndex(0,2), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NW_NE);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.right - AIMap.GetTileIndex(0,2), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NW_SW);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.right - AIMap.GetTileIndex(0,2), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_SW_SE);											
	} else if(exit_direction == Direction.SOUTHEAST) {
		::ai_instance._rail_manager._rail_demolisher.AddRouteTilesIntoRailTilesList([exit_tile_pair.right + AIMap.GetTileIndex(0,1), exit_tile_pair.right + AIMap.GetTileIndex(0,2), exit_tile_pair.right + AIMap.GetTileIndex(0,3)], 
																					RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NW_SE);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTilesIntoRailTilesList([exit_tile_pair.left + AIMap.GetTileIndex(0,1), exit_tile_pair.left + AIMap.GetTileIndex(0,2), exit_tile_pair.left + AIMap.GetTileIndex(0,3)], 
																					RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NW_SE);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.right + AIMap.GetTileIndex(0,2), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NE_SE);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.right + AIMap.GetTileIndex(0,2), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NW_NE);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.left + AIMap.GetTileIndex(0,2), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_NW_SW);								
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(exit_tile_pair.left + AIMap.GetTileIndex(0,2), RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, AIRail.RAILTRACK_SW_SE);											
	} 
}
function RailBuilder::BuildStationTracks(exit_tile_pair, exit_direction) 
{		
	if(exit_direction == Direction.NORTHEAST) {
		if(!AIRail.BuildRail(exit_tile_pair.right, exit_tile_pair.right - AIMap.GetTileIndex(1,0), exit_tile_pair.right - AIMap.GetTileIndex(4,0))) return false;
		if(!AIRail.BuildRail(exit_tile_pair.left, exit_tile_pair.left - AIMap.GetTileIndex(1,0), exit_tile_pair.left - AIMap.GetTileIndex(4,0))) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.right - AIMap.GetTileIndex(2,0), AIRail.RAILTRACK_NW_SW)) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.right - AIMap.GetTileIndex(2,0), AIRail.RAILTRACK_NW_NE)) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.left - AIMap.GetTileIndex(2,0), AIRail.RAILTRACK_SW_SE)) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.left - AIMap.GetTileIndex(2,0), AIRail.RAILTRACK_NE_SE)) return false;
	} else if(exit_direction == Direction.SOUTHWEST) {
		if(!AIRail.BuildRail(exit_tile_pair.right, exit_tile_pair.right + AIMap.GetTileIndex(1,0), exit_tile_pair.right + AIMap.GetTileIndex(4,0))) return false;
		if(!AIRail.BuildRail(exit_tile_pair.left, exit_tile_pair.left + AIMap.GetTileIndex(1,0), exit_tile_pair.left + AIMap.GetTileIndex(4,0))) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.left + AIMap.GetTileIndex(2,0), AIRail.RAILTRACK_NW_SW)) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.left + AIMap.GetTileIndex(2,0), AIRail.RAILTRACK_NW_NE)) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.right + AIMap.GetTileIndex(2,0), AIRail.RAILTRACK_SW_SE)) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.right + AIMap.GetTileIndex(2,0), AIRail.RAILTRACK_NE_SE)) return false;
	} else if(exit_direction == Direction.NORTHWEST) {
		if(!AIRail.BuildRail(exit_tile_pair.right, exit_tile_pair.right - AIMap.GetTileIndex(0,1), exit_tile_pair.right - AIMap.GetTileIndex(0,4)))return false;
		if(!AIRail.BuildRail(exit_tile_pair.left, exit_tile_pair.left - AIMap.GetTileIndex(0,1), exit_tile_pair.left - AIMap.GetTileIndex(0,4)))return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.left - AIMap.GetTileIndex(0,2), AIRail.RAILTRACK_NE_SE)) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.left - AIMap.GetTileIndex(0,2), AIRail.RAILTRACK_NW_NE)) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.right - AIMap.GetTileIndex(0,2), AIRail.RAILTRACK_NW_SW)) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.right - AIMap.GetTileIndex(0,2), AIRail.RAILTRACK_SW_SE)) return false;
	} else if(exit_direction == Direction.SOUTHEAST) {
		if(!AIRail.BuildRail(exit_tile_pair.right, exit_tile_pair.right + AIMap.GetTileIndex(0,1), exit_tile_pair.right + AIMap.GetTileIndex(0,4))) return false;
		if(!AIRail.BuildRail(exit_tile_pair.left, exit_tile_pair.left + AIMap.GetTileIndex(0,1), exit_tile_pair.left + AIMap.GetTileIndex(0,4))) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.right + AIMap.GetTileIndex(0,2), AIRail.RAILTRACK_NE_SE)) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.right + AIMap.GetTileIndex(0,2), AIRail.RAILTRACK_NW_NE)) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.left + AIMap.GetTileIndex(0,2), AIRail.RAILTRACK_NW_SW)) return false;
		if(!AIRail.BuildRailTrack(exit_tile_pair.left + AIMap.GetTileIndex(0,2), AIRail.RAILTRACK_SW_SE)) return false;
	} 
	if(!RailBuilder.BuildStationTracksSignals(exit_tile_pair, exit_direction)) return false;
	RailBuilder.IndexStationTracks(route_id, exit_tile_pair, exit_direction);
	
	return true;
}

function RailBuilder::BuildStationTracksSignals(exit_tile_pair, exit_direction) {
	
	//offsets
	local direction_offset_1 = Direction.GetDirectionOffset(exit_direction, 1, 1);
	local direction_offset_2 = Direction.GetDirectionOffset(exit_direction, 2, 2);
	local direction_offset_3 = Direction.GetDirectionOffset(exit_direction, 3, 3);
	local direction_offset_4 = Direction.GetDirectionOffset(exit_direction, 4, 4);
	
	if(!AIRail.BuildSignal(exit_tile_pair.right + direction_offset_3, exit_tile_pair.right + direction_offset_4, AIRail.SIGNALTYPE_ENTRY)) return false;
	if(!AIRail.BuildSignal(exit_tile_pair.left + direction_offset_3, exit_tile_pair.left + direction_offset_2, AIRail.SIGNALTYPE_PBS_ONEWAY)) return false;	
	if(!AIRail.BuildSignal(exit_tile_pair.right + direction_offset_1, exit_tile_pair.right, AIRail.SIGNALTYPE_EXIT_TWOWAY)) return false;
	if(!AIRail.BuildSignal(exit_tile_pair.left + direction_offset_1, exit_tile_pair.left, AIRail.SIGNALTYPE_EXIT_TWOWAY)) return false;
	
	return true;
}


function RailBuilder::GetTownStationSpotList(town, other_town, platform_length, number_of_tracks, radius)
{
	local town_tilelist = AITileList();
	local town_tile = AITown.GetLocation(town);

	//add tiles around town
	Util.SafeAddRectangle(town_tilelist, town_tile, radius*3);
	
	//filter tiles that will belong to specified town (not the neighbour town)
	town_tilelist.Valuate(AITile.GetClosestTown);
	town_tilelist.KeepValue(town);
	local other_town_direction = Direction.GetDirectionToTile(town_tile, AITown.GetLocation(other_town));
	local valuator = CanBuildRailStationWithValueValuator(AITown.GetLocation(town), platform_length, number_of_tracks, other_town_direction, 
																  ::ai_instance._general_manager.passenger_cargo_id, true, radius);
	town_tilelist.Valuate(valuator.Valuate, valuator);
	town_tilelist.KeepAboveValue(0);

	town_tilelist.Sort(AIList.SORT_BY_VALUE, false);
	town_tilelist.KeepTop(25);
	
	/*
	AILog.Info("RailBuilder::GetTownStationSpotList");
	local i = 0;
	for(local tile = town_tilelist.Begin(); !town_tilelist.IsEnd(); tile = town_tilelist.Next()) {
		AISign.BuildSign(tile, "T" + ++i);
		AILog.Info("Tile " + i + ". " + Util.WriteTile(tile) + " Value: " + town_tilelist.GetValue(tile));
	}*/
	return town_tilelist;
}
function RailBuilder::GetIndustryStationSpotList(industry, other_industry, platform_length, number_of_tracks, cargo, accept, radius)
{
	local industry_tilelist = null;
	local industry_tile = AIIndustry.GetLocation(industry);
	if(accept) {
		industry_tilelist = AITileList_IndustryAccepting(industry, radius);
	}
	else {
		industry_tilelist = AITileList_IndustryProducing(industry, radius);
	}

	local other_direction = Direction.GetDirectionToTile(AIIndustry.GetLocation(industry), AIIndustry.GetLocation(other_industry));
		local i = 0;

	local valuator = CanBuildRailStationWithValueValuator(AIIndustry.GetLocation(industry), platform_length, number_of_tracks, other_direction, cargo, accept, radius);
	industry_tilelist.Valuate(valuator.Valuate, valuator);
	industry_tilelist.RemoveValue(0);
	industry_tilelist.Sort(AIList.SORT_BY_VALUE, false);
	industry_tilelist.KeepTop(10);
	/*
	AILog.Info("RailBuilder::GetIndustryStationSpotList");
	local i = 0;
	for(local item = industry_tilelist.Begin(); !industry_tilelist.IsEnd(); item = industry_tilelist.Next()) {
		AISign.BuildSign(item, "V" + ++i + " - " + industry_tilelist.GetValue(item));
		AILog.Info("Value " + industry_tilelist.GetValue(item));
	}
	*/
	return industry_tilelist;
}

function RailBuilder::IsExitDirectionOfStationBuildable(direction, tile, platform_length, number_of_tracks)
{
	switch(direction) {
		case Direction.NORTHEAST: return Util.IsTileRectagleFlatAndBuildable(tile - AIMap.GetTileIndex(5, 0), 
							             tile + AIMap.GetTileIndex(platform_length + 1, number_of_tracks - 1));
		case Direction.NORTHWEST: return Util.IsTileRectagleFlatAndBuildable(tile - AIMap.GetTileIndex(0, 5), 
										 tile + AIMap.GetTileIndex(number_of_tracks - 1, platform_length - 1));
		case Direction.SOUTHEAST: return Util.IsTileRectagleFlatAndBuildable(tile, tile + 
		                                 AIMap.GetTileIndex(number_of_tracks - 1, platform_length + 5 - 1));
		case Direction.SOUTHWEST: return Util.IsTileRectagleFlatAndBuildable(tile, tile + 
										 AIMap.GetTileIndex(platform_length + 5 - 1 , number_of_tracks - 1));
		
		default: return null;
	}
}

function RailBuilder::GetDirectionPriority(station_tile, other_industry_tile, platform_length, number_of_tracks, cargo, accept, radius)
{
	local direction = Direction.GetDirectionToTile(station_tile, other_industry_tile);
	//AILog.Info("Other industry/town direction: " + Direction.ToString(direction));
	local direction_priority_list = array(0);
	local ne_sw_acc = 0;
	local nw_se_acc = 0;
	if(accept) {
		ne_sw_acc = AITile.GetCargoAcceptance(station_tile, cargo, platform_length, number_of_tracks, radius);
		nw_se_acc = AITile.GetCargoAcceptance(station_tile, cargo, number_of_tracks, platform_length, radius);
	} 
	//AILog.Info("ne_sw_acc: " + ne_sw_acc + " nw_se_acc: " + nw_se_acc);
	local base_distance = AITile.GetDistanceManhattanToTile(station_tile, other_industry_tile);
	local other_distance = 0; 
		
	//it will put direction into array if the station is buildable based on priority (1.straight direction, 2. which exit direction is closer to other possible station)
	//if accepting it will check better cargo acceptance first
	if(accept) { //ACCEPTING
		if(direction == Direction.NORTH) {
			if(ne_sw_acc > nw_se_acc) {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHEAST);
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHWEST);
				}
			} else {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHWEST);
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHEAST);
				}
			}	
		} else if(direction == Direction.SOUTH) {
			if(ne_sw_acc > nw_se_acc) {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHWEST);
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHEAST);
				}
			} else {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHEAST);
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHWEST);
				}
			}
		} else if(direction == Direction.WEST) {
			if(ne_sw_acc > nw_se_acc) {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHWEST);
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHWEST);
				}
			} else {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHWEST);
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHWEST);
				}
			}
		} else if(direction == Direction.EAST) {
			if(ne_sw_acc > nw_se_acc) {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHEAST);
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHEAST);
				}
			} else {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHEAST);
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHEAST);
				}
			}
		} else if(direction == Direction.NORTHEAST) {
			other_distance = AITile.GetDistanceManhattanToTile(station_tile + AIMap.GetTileIndex(0, platform_length), other_industry_tile);
			if(ne_sw_acc > nw_se_acc) {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHEAST);
				}
				if(base_distance < other_distance) {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHWEST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHEAST);
					}
				} else {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHEAST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHWEST);
					}
				}
			} else {
				if(base_distance < other_distance) {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHWEST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHEAST);
					}
				} else {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHEAST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHWEST);
					}
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHEAST);
				}
			}				
		} else if(direction == Direction.NORTHWEST) {
			other_distance = AITile.GetDistanceManhattanToTile(station_tile + AIMap.GetTileIndex(platform_length, 0), other_industry_tile);
			if(ne_sw_acc > nw_se_acc) {
				if(base_distance < other_distance) {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHEAST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHWEST);
					}
				} else {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHWEST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHEAST);
					}
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHWEST);
				}
			} else {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHWEST);
				}
				if(base_distance < other_distance) {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHEAST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHWEST);
					}
				} else {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHWEST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHEAST);
					}
				}
			}		
		} else if(direction == Direction.SOUTHEAST) {
			other_distance = AITile.GetDistanceManhattanToTile(station_tile + AIMap.GetTileIndex(platform_length, 5), other_industry_tile);
			if(ne_sw_acc > nw_se_acc) {
				if(base_distance < other_distance) {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHEAST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHWEST);
					}
				} else {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHWEST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHEAST);
					}
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHEAST);
				}
			} else {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHEAST);
				}
				if(base_distance < other_distance) {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHEAST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHWEST);
					}
				} else {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHWEST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHEAST);
					}
				}
			} 
		} else if(direction == Direction.SOUTHWEST) {
			other_distance = AITile.GetDistanceManhattanToTile(station_tile + AIMap.GetTileIndex(0, platform_length), other_industry_tile);
			if(ne_sw_acc > nw_se_acc) {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHWEST);
				}
				if(base_distance < other_distance) {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHWEST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHEAST);
					}
				} else {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHEAST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHWEST);
					}
				}
			} else {
				if(base_distance < other_distance) {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHWEST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHEAST);
					}
				} else {
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.SOUTHEAST);
					}
					if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
						direction_priority_list.push(Direction.NORTHWEST);
					}
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHWEST);
				}
			}				
		}
	} else { //PRODUCING
		if(direction == Direction.NORTH) {
			if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHEAST);
			}
			if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHWEST);
			}
			
		} else if(direction == Direction.SOUTH) {
			if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHEAST);
			}
			if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHWEST);
			}
		} else if(direction == Direction.WEST) {
			if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHWEST);
			}
			if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHWEST);
			}
		} else if(direction == Direction.EAST) {
			if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHEAST);
			}
			if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
				direction_priority_list.push(Direction.SOUTHEAST);
			}
		} else if(direction == Direction.NORTHEAST) {
		    other_distance = AITile.GetDistanceManhattanToTile(station_tile + AIMap.GetTileIndex(0, platform_length), other_industry_tile);
			if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHEAST);
				}
			if(base_distance < other_distance) {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHWEST);
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHEAST);
				}
			} else {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHEAST);
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHWEST);
				}
			}	
		} else if(direction == Direction.NORTHWEST) {
		other_distance = AITile.GetDistanceManhattanToTile(station_tile + AIMap.GetTileIndex(platform_length, 0), other_industry_tile);
			if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHWEST);
			}
			if(base_distance < other_distance) {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHEAST);
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHWEST);	
				}
			} else {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHWEST);
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHEAST);	
				}			
			}
		} else if(direction == Direction.SOUTHEAST) {
		    other_distance = AITile.GetDistanceManhattanToTile(station_tile + AIMap.GetTileIndex(0, platform_length), other_industry_tile);
			if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
				direction_priority_list.push(Direction.SOUTHEAST);
			}
			if(base_distance < other_distance) {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHEAST);
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHWEST);
				}
			} else {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHWEST);
				}
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHEAST);
				}			
			}			
		} else if(direction == Direction.SOUTHWEST) {
			other_distance = AITile.GetDistanceManhattanToTile(station_tile + AIMap.GetTileIndex(platform_length, 0), other_industry_tile);
			if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, station_tile, platform_length, number_of_tracks)) {
				direction_priority_list.push(Direction.SOUTHWEST);
			} 
			if(base_distance < other_distance) {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.NORTHWEST);
				} 
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHEAST);
				} 
			} else {
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, station_tile, platform_length, number_of_tracks)) {
					direction_priority_list.push(Direction.SOUTHEAST);
				} 
				if(RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, station_tile, platform_length, number_of_tracks)) {				
					direction_priority_list.push(Direction.NORTHWEST);
				} 
			}
		}
	}
	/*
	AILog.Warning("DIRECTIONS! " + Util.WriteTile(station_tile) + " Other: " + Util.WriteTile(other_industry_tile));
	AISign.BuildSign(station_tile, "DIRECTIONS!");
	foreach(item in direction_priority_list) {
		AILog.Info(Direction.ToString(item));
	}
	*/
	return direction_priority_list;
	
}


function RailBuilder::ConnectDepotDiagonal(route_id, tile_a, tile_b, tile_c)
{

	if (!AITile.IsBuildable(tile_c) && (!AIRail.IsRailTile(tile_c) || !AICompany.IsMine(AITile.GetOwner(tile_c)))) return null;
	local offset1 = (tile_c - tile_a) / 2;
	local offset2 = (tile_c - tile_b) / 2;
	local depot_tile = null;
	local depot_build = false;
	local tiles = [];
	tiles.append([tile_a, tile_a + offset1, tile_c]);
	tiles.append([tile_b, tile_b + offset2, tile_c]);
	if (AIRail.IsRailDepotTile(tile_c + offset1) && AIRail.GetRailDepotFrontTile(tile_c + offset1) == tile_c &&
			AIRail.TrainHasPowerOnRail(AIRail.GetRailType(tile_c + offset1), AIRail.GetCurrentRailType())) {
		/* If we can't build trains for the current rail type in the depot, see if we can
		 * convert it without problems. */
		if (!AIRail.TrainHasPowerOnRail(AIRail.GetCurrentRailType(), AIRail.GetRailType(tile_c + offset1))) {
			if (!AIRail.ConvertRailType(tile_c + offset1, tile_c + offset1, AIRail.GetCurrentRailType())) return null;
		}
		depot_tile = tile_c + offset1;
		depot_build = true;
		tiles.append([tile_a + offset1, tile_c, tile_c + offset1]);
		tiles.append([tile_b + offset2, tile_c, tile_c + offset1]);
	} else if (AIRail.IsRailDepotTile(tile_c + offset2) && AIRail.GetRailDepotFrontTile(tile_c + offset2) == tile_c &&
			AIRail.TrainHasPowerOnRail(AIRail.GetRailType(tile_c + offset2), AIRail.GetCurrentRailType())) {
		/* If we can't build trains for the current rail type in the depot, see if we can
		 * convert it without problems. */
		if (!AIRail.TrainHasPowerOnRail(AIRail.GetCurrentRailType(), AIRail.GetRailType(tile_c + offset2))) {
			if (!AIRail.ConvertRailType(tile_c + offset2, tile_c + offset2, AIRail.GetCurrentRailType())) return null;
		}
		depot_tile = tile_c + offset2;
		depot_build = true;
		tiles.append([tile_a + offset1, tile_c, tile_c + offset2]);
		tiles.append([tile_b + offset2, tile_c, tile_c + offset2]);
	} else if (AITile.IsBuildable(tile_c + offset1)) {
		if (AITile.GetMaxHeight(tile_c) != AITile.GetMaxHeight(tile_a) &&
			!AITile.RaiseTile(tile_c, AITile.GetComplementSlope(AITile.GetSlope(tile_c)))) return null;
		if (AITile.GetMaxHeight(tile_c) != AITile.GetMaxHeight(tile_a) &&
			!AITile.RaiseTile(tile_c, AITile.GetComplementSlope(AITile.GetSlope(tile_c)))) return null;
		depot_tile = tile_c + offset1;
		tiles.append([tile_a + offset1, tile_c, tile_c + offset1]);
		tiles.append([tile_b + offset2, tile_c, tile_c + offset1]);
	} else if (AITile.IsBuildable(tile_c + offset2)) {
		if (AITile.GetMaxHeight(tile_c) != AITile.GetMaxHeight(tile_a) &&
			!AITile.RaiseTile(tile_c, AITile.GetComplementSlope(AITile.GetSlope(tile_c)))) return null;
		if (AITile.GetMaxHeight(tile_c) != AITile.GetMaxHeight(tile_a) &&
			!AITile.RaiseTile(tile_c, AITile.GetComplementSlope(AITile.GetSlope(tile_c)))) return null;
		depot_tile = tile_c + offset2;
		tiles.append([tile_a + offset1, tile_c, tile_c + offset2]);
		tiles.append([tile_b + offset2, tile_c, tile_c + offset2]);
	} else {
		return null;
	}
	{
		local test = AITestMode();
		foreach (t in tiles) {
			if (!AIRail.AreTilesConnected(t[0], t[1], t[2]) && !AIRail.BuildRail(t[0], t[1], t[2])) return null;
		}
		if (!depot_build && !AIRail.BuildRailDepot(depot_tile, tile_c)) return null;
	}
	foreach (t in tiles) {
		local railtrack;
		local railtrack_t1_before = AIRail.GetRailTracks(t[1]);
		
		if (!AIRail.AreTilesConnected(t[0], t[1], t[2]) && !AIRail.BuildRail(t[0], t[1], t[2])) return null;
		
		if(railtrack_t1_before == AIRail.RAILTRACK_INVALID ) railtrack = AIRail.GetRailTracks(t[1]);
		else railtrack = railtrack_t1_before ^ AIRail.GetRailTracks(t[1]);						
		::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(t[1], RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, railtrack);								
	}
	if (!depot_build && !AIRail.BuildRailDepot(depot_tile, tile_c)) return null;
	//index railtracks of existing rail depot	
	if(depot_build) {
		foreach(tile in tiles[0]) {
			local rail_tile_info = ::ai_instance._rail_manager._rail_demolisher.GetRailTileInfo(tile);
			local railtracks = ::ai_instance._rail_manager._rail_demolisher.GetAllRailtracksOnTileOfThisRoute(rail_tile_info, 0);
			foreach(railtrack in railtracks) {
				::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(tile, RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, railtrack);
			}
		}
	}
	return depot_tile;
}

function RailBuilder::BuildDepot(route_id, path)
{	
	local prev = null;
	local pp = null;
	local ppp = null;
	local pppp = null;
	local ppppp = null;
	while (path != null) {
		if (ppppp != null) {
			if (ppppp - pppp == pppp - ppp && pppp - ppp == ppp - pp && ppp - pp == pp - prev) {
				local offsets = [AIMap.GetTileIndex(0, 1), AIMap.GetTileIndex(0, -1),
				                 AIMap.GetTileIndex(1, 0), AIMap.GetTileIndex(-1, 0)];
				foreach (offset in offsets) {
					if (AITile.GetMaxHeight(ppp + offset) != AITile.GetMaxHeight(ppp)) continue;
					
					local depot_build = false;
					if (AIRail.IsRailDepotTile(ppp + offset)) {
						if (AIRail.GetRailDepotFrontTile(ppp + offset) != ppp) continue;
						if (!AIRail.TrainHasPowerOnRail(AIRail.GetRailType(ppp + offset), AIRail.GetCurrentRailType())) continue;
						/* If we can't build trains for the current rail type in the depot, see if we can
						 * convert it without problems. */
						if (!AIRail.TrainHasPowerOnRail(AIRail.GetCurrentRailType(), AIRail.GetRailType(ppp + offset))) {
							if (!AIRail.ConvertRailType(ppp + offset, ppp + offset, AIRail.GetCurrentRailType())) continue;
						}
						depot_build = true;
					} else {
					
						local test = AITestMode();
						if (!AIRail.BuildRailDepot(ppp + offset, ppp)) continue;
					}
					
					local railtrack;
					local railtrack_ppp_before = AIRail.GetRailTracks(ppp);
					
					if (!AIRail.AreTilesConnected(pp, ppp, ppp + offset) && !AIRail.BuildRail(pp, ppp, ppp + offset)) continue;
					
					railtrack = railtrack_ppp_before ^ AIRail.GetRailTracks(ppp);						
					::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(ppp, RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, railtrack);								
					railtrack_ppp_before = AIRail.GetRailTracks(ppp);
					if (!AIRail.AreTilesConnected(pppp, ppp, ppp + offset) && !AIRail.BuildRail(pppp, ppp, ppp + offset)) continue;
					
					railtrack = railtrack_ppp_before ^ AIRail.GetRailTracks(ppp);						
					::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(ppp, RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, railtrack);								
					
					//index railtracks of existing rail depot					
					if (depot_build) {
						local rail_tile_info = ::ai_instance._rail_manager._rail_demolisher.GetRailTileInfo(ppp);
						local railtracks = ::ai_instance._rail_manager._rail_demolisher.GetAllRailtracksOnTileOfThisRoute(rail_tile_info, 0);
						foreach(railtrack in railtracks) {
							::ai_instance._rail_manager._rail_demolisher.AddRouteTileIntoRailTilesList(ppp, RailDemolisher.RAILTILE_TYPE_RAILTRACK, route_id, railtrack);
						}
						return ppp + offset;
					}
					if(AIRail.BuildRailDepot(ppp + offset, ppp)) return ppp + offset;
				}
			} else if (ppppp - ppp == ppp - prev && ppppp - pppp != pp - prev) {
				local offsets = null;
				if (abs(ppppp - ppp) == AIMap.GetTileIndex(1, 1)) {
					if (ppppp - pppp == AIMap.GetTileIndex(1, 0) || prev - pp == AIMap.GetTileIndex(1, 0)) {
						local d = RailBuilder.ConnectDepotDiagonal(route_id, prev, ppppp, max(prev, ppppp) + AIMap.GetTileIndex(-2, 0));
						if (d != null) return d;
					} else {
						local d = RailBuilder.ConnectDepotDiagonal(route_id, prev, ppppp, max(prev, ppppp) + AIMap.GetTileIndex(0, -2));
						if (d != null) return d;
					}
				} else {
					if (ppppp - pppp == AIMap.GetTileIndex(0, -1) || prev - pp == AIMap.GetTileIndex(0, -1)) {
						local d = RailBuilder.ConnectDepotDiagonal(route_id, prev, ppppp, max(prev, ppppp) + AIMap.GetTileIndex(2, 0));
						if (d != null) return d;
					} else {
						local d = RailBuilder.ConnectDepotDiagonal(route_id, prev, ppppp, max(prev, ppppp) + AIMap.GetTileIndex(0, -2));
						if (d != null) return d;
					}
				}
			}
		}
		ppppp = pppp;
		pppp = ppp;
		ppp = pp;
		pp = prev;
		prev = path.GetTile();
		path = path.GetParent();
	}
	AILog.Warning("Depot not build");
	return null;
}

function RailBuilder::SignalPath(path)
{
	RailBuilder.SignalPathAdvanced(path, 0, null, 999999)
}

function RailBuilder::SignalPathAdvanced(path, skip, end, signal_count_limit) 
{
	local i = 0;
	local prev = null;
	local prevprev = null;
	local prevprevprev = null;
	local prevprevprevprev = null;
	local tiles_skipped = 50-(skip)*10;
	local lastbuild_tile = null;
	local lastbuild_front_tile = null;
	while (path != null && path != end ) {
		if(path.GetIsExistingRailTile()) {
			tiles_skipped = 0;
			//AISign.BuildSign(path.GetTile(), "E");
		}
		if (prevprev != null) {
			if (AIMap.DistanceManhattan(prev, path.GetTile()) > 1) {
				tiles_skipped += 10 * AIMap.DistanceManhattan(prev, path.GetTile());
			} else {
				if (path.GetTile() - prev != prev - prevprev) {
					tiles_skipped += 5;
				} else {
					tiles_skipped += 10;
				}
				//AISign.BuildSign(path.GetTile(), "tiles skipped: "+tiles_skipped)
				if (AIRail.GetSignalType(prev, path.GetTile()) != AIRail.SIGNALTYPE_NONE) tiles_skipped = 0;
				AISign.BuildSign(path.GetTile(), tiles_skipped)
				if (tiles_skipped > 55 && path.GetParent() != null && signal_count_limit>0) {
					if (AIRail.BuildSignal(path.GetTile(), prev, AIRail.SIGNALTYPE_PBS_ONEWAY)) {
						i++;
						tiles_skipped = 0;
						lastbuild_tile = prev;
						lastbuild_front_tile = path.GetTile();
						signal_count_limit--;
					}
				}
			}
		}
		prevprevprevprev = prevprevprev;
		prevprevprev = prevprev;
		prevprev = prev;
		prev = path.GetTile();
		path = path.GetParent();
	}
	/* Although this provides better signalling (trains cannot get stuck half in the station),
	* it is also the cause of using the same track of rails both ways, possible causing deadlocks.
	if (tiles_skipped < 50 && lastbuild_tile != null) {
		AIRail.RemoveSignal(lastbuild_tile, lastbuild_front_tile);
	}*/

	if (AIRail.GetSignalType(prevprev, prevprevprev) == AIRail.SIGNALTYPE_NONE) {
		AIRail.BuildSignal(prevprev, prevprevprev, AIRail.SIGNALTYPE_PBS_ONEWAY);
	}
	if (AIRail.GetSignalType(prevprevprev, prevprevprevprev) == AIRail.SIGNALTYPE_NONE && AIRail.GetSignalType(prevprev, prevprevprev) == AIRail.SIGNALTYPE_NONE) {
		AIRail.BuildSignal(prevprevprev, prevprevprevprev, AIRail.SIGNALTYPE_PBS_ONEWAY);
	}
	//AISign.BuildSign(prevprev, "prevprev "+AIRail.GetSignalType(prevprev, prevprevprev))
	//AISign.BuildSign(prevprevprev, "prevprevprev")
	//AISign.BuildSign(prevprevprevprev, "prevprevprevprev")
	
	return i;
}
function RailBuilder::CheckSignalPath(path) 
{
	local back = path;
	local current = path.GetParent();
	local front = current.GetParent();
	local last_signal_current_tile = path.GetTile();
	local last_signal_front_tile = current.GetTile();
	
	while(front != null) {
		//AISign.BuildSign(current.GetTile(), "CURRENT");
		//AISign.BuildSign(front.GetTile(), "FRONT");
		if(AIRail.GetSignalType(front.GetTile(), current.GetTile()) != AIRail.SIGNALTYPE_NONE) {
			
			local dist = AITile.GetDistanceManhattanToTile(current.GetTile(), last_signal_current_tile);
			//if it is too close remove signal
			if(AITile.GetDistanceManhattanToTile(current.GetTile(), last_signal_current_tile) < 6) {
				if(AIRail.GetSignalType(front.GetTile(), current.GetTile()) != AIRail.SIGNALTYPE_ENTRY) {
			   
					local type = AIRail.GetSignalType(front.GetTile(), current.GetTile()) != AIRail.SIGNALTYPE_NONE;
					if(!AIRail.RemoveSignal(front.GetTile(), current.GetTile())) {
						AILog.Warning(AIError.GetLastErrorString());
					} 
				} else {
					if(!AIRail.RemoveSignal(last_signal_front_tile, last_signal_current_tile)) {
						AILog.Warning(AIError.GetLastErrorString());
					}
				}
				
			} else {
				last_signal_current_tile = current.GetTile();
				last_signal_front_tile = front.GetTile();
			}
		}	
		
		back = current;
		current = front;
		front = front.GetParent();

	}
}