/**
 * @author Daniela Plachtova
 * @file rail_route.nut
 */

 /**
  * @brief class Rail Route manages all route vehicle functions 
  */
class RailRoute
{
	static INDUSTRY_PRODUCTION_PERCENTAGE_LOW = 55;				/* Limit of industry production percentage below we add 2 trains */
	static INDUSTRY_PRODUCTION_PERCENTAGE_HIGH = 75;			/* Limit of industry production percentage below we add a train */
	static MINIMAL_CARGO_AMOUNT_WAITING_IN_STATION = 50; 		/* Limit of waiting cargo (other than passengers) in rail station below which we do not need to add a train. */
	static MINIMAL_PASS_AMOUNT_WAITING_IN_STATION = 250;		/* Limit of waiting cargo (passengers) in rail station above which we add a train. */	
	
	_route_id = null;					/* route id */
	_station_producing = null;			/* station_producing station industry producing (if town it is first town in town pair) */
	_station_accepting = null;			/* station_accepting station industry accepting (if town it is second town in town pair) */
	_cargoID = null						/* cargo id */
	_vehicle_list = null;				/* list of all vehicles in route */
	_depot_pro = null;					/* depot on pro acc path */
	_depot_acc = null;					/* depot on acc pro path */
	_railtype = null;					/* type of railtrack */
	_is_town_route = null;				/* is this town route */
	_routeOK = null;					/* is route OK */
	
	/**
	 * @brief constructor for RailRoute
	 * @param route_id assigned route id
	 * @param station_producing station industry producing (if town it is first town in town pair)
	 * @param station_accepting station industry accepting (if town it is second town in town pair)
	 * @param cargoID cargo id
	 * @param depot_pro depot on pro-acc path
	 * @param depot_acc depot on acc-pro path
	 * @param railtype used railtype
	 * @param is_town_route is this town route
	 */
	constructor(route_id, station_producing, station_accepting, cargoID, depot_pro, depot_acc, railtype, is_town_route) 
	{
		_route_id = route_id;
		_station_producing = station_producing;
		_station_accepting = station_accepting;
		_cargoID = cargoID;
		_depot_pro = depot_pro;
		_depot_acc = depot_acc;
		_railtype = railtype;
		_is_town_route = is_town_route;
		_routeOK = true;	
		_vehicle_list = AIList();		
	}
	
	/**
	 * @brief Add number of trains in route
	 * @param number number of trains to add
	 * @return number of trains added to route
	 */
	function AddNumberOfTrains(number);
	
	/**
	 * @brief build train and add it in route
	 * @return true if building train suceeded else false
	 * @note modified, taken from trAIns
	 */
	function AddTrain();
	
	/**
	 * @brief Adjust Number of Trains in Route
	 * @note modified, taken from trAIns 
	 */
	function AdjustNumberOfTrainsInRoute();
	
	/**
	 * @brief auxiliary save function for game
	 * @return RailRoute in table
	 */
	function Save();
	
	/**
	 * @brief auxiliary load function for game
	 * @param data data to load
	 * @return Created RailRoute
	 */
	function Load(data);
	
	/**
	 * @brief ToString
	 * @return string RailRoute info
	 */
	function ToString();
	
	/**
	 * @brief Calculate number of wagons for train
	 * @param locomotive_engine locomotive engine
	 * @param wagon_engine waogn engine
	 * @return number of wagon we can build to specified locomotive or null error occured
	 * @note modified, taken from trAIns
	 */
	function CalculateNumberOfWagons(locomotive_engine , wagon_engine);
	
	/**
	 * @brief Get number of trains in route
	 * @return number of trains in route
	 */
	function GetNumberOfTrains();	
	
	/**
	 * @brief Choose best wagon available 
	 * @param cargo cargo of wagon
	 * @param railtype railtype of engine
	 * @return best wagon available or null if nothing suitable have been found
	 */
	function ChooseWagon(cargo, railtype);
	
	/**
	 * @brief Choose best locomotive available 
	 * @param cargo cargo
	 * @param railtype railtype of tracks
	 * @param reliable true if we want more reliable locomotive 
	 * @return best locomotive available or null if nothing suitable have been found
	 */	
	function ChooseLocomotive(cargo, railtype, reliable);
	
	/**
	 * @brief Prints Vehicle List of Route
	 */
	function PrintVehicleList();
	
}

function RailRoute::PrintVehicleList() 
{
	local vehicleType;

	for(local v = this._vehicle_list.Begin(); !this._vehicle_list.IsEnd(); v = this._vehicle_list.Next()) {
		AILog.Info(v + " " + AIVehicle.GetName(v) + " Valid: " + AIVehicle.IsValidVehicle(v));
	}

}
function RailRoute::ToString()
{
	return "ID: " + this._route_id + " S_PRO: " + AIStation.GetName(this._station_producing._stationID) + " S_ACC: " + AIStation.GetName(this._station_accepting._stationID) +
	       " Cargo: " + AICargo.GetCargoLabel(this._cargoID) + " IsTownRoute: " + this._is_town_route + " VehicleList count: " + this._vehicle_list.Count() + " Railtype: " + this._railtype + " routeOK: " + this._routeOK;
}

function RailRoute::GetNumberOfTrains() 
{
	return this._vehicle_list.Count();
}

function RailRoute::AddNumberOfTrains(number) 
{
	local vl = AIVehicleList();
	vl.Valuate(AIVehicle.GetVehicleType);
	vl.KeepValue(AIVehicle.VT_RAIL);
	//AILog.Info("MAX NUMBER OF ALLOWED TRAINS : " + Vehicle.GetVehicleLimit(AIVehicle.VT_RAIL) + " we have: " + vl.Count() + " trains.");
	local i = 0;
	for(; i < number; i++) {
		this.AddTrain();
		switch(::ai_instance._error_manager.GetLastErrorString()) {
			case ErrorManager.ERROR_NOT_ENOUGH_CASH:
				AILog.Info("We could only add " + i + "/" + number + " trains because we do not have enough money for more");
				return i;
			case ErrorManager.ERROR_TOO_MANY_VEHICLES:
				AILog.Info("We could only add " + i + "/" + number + " trains because we reached maximum number of trains in game");
				return i;
			case ErrorManager.ERROR_UNKNOWN:	
				AILog.Info("We could only add " + i + "/" + number + " trains because of unknown error");
				return i;
			default: break;
		}
	}
	if(::ai_instance._error_manager.GetLastError() == ErrorManager.ERROR_NONE) {
		AILog.Info("We added all " + number + " trains");
		return number;
	}
	return i;

}

function RailRoute::AddTrain() 
{
	local money_reservationID = 0;
	local locomotive_engine = RailRoute.ChooseLocomotive(this._cargoID, this._railtype, true);
	//AILog.Info("Chosen locomotive engine: " + AIEngine.GetName(locomotive_engine));
	local wagon_engine = RailRoute.ChooseWagon(this._cargoID, this._railtype);
	local wagon_engine_mail;
	if(this._is_town_route) wagon_engine_mail = RailRoute.ChooseWagon(::ai_instance._general_manager.mail_cargo_id, this._railtype);
	//AILog.Info("Chosen wagon engine: " + AIEngine.GetName(wagon_engine));
	local cost;
	local locomotive;
	local wagon;
	
	if(locomotive_engine == null || wagon_engine == null || 
	  !AIEngine.IsBuildable(locomotive_engine) || !AIEngine.IsBuildable(wagon_engine)) {
		AILog.Warning("chosen engines are not buildable");
		return false;
	}
	local number_of_wagons = RailRoute.CalculateNumberOfWagons(locomotive_engine, wagon_engine);
	
	if(number_of_wagons == null) {
		
		AILog.Warning("number of wagon is null " + AIError.GetLastErrorString());
		return false;
	}
	if(number_of_wagons == 0) {
		AILog.Warning("number of wagon is zero");
		return false;
	}
	//AILog.Warning("Number of wagons: " + number_of_wagons);
	cost = AIEngine.GetPrice(locomotive_engine) + AIEngine.GetPrice(wagon_engine) * number_of_wagons;
	//AILog.Info("RailRoute::AddTrain cost: " + cost);
	money_reservationID = ::ai_instance._money_manager.ReserveMoney(cost, (1.25*cost).tointeger());
	if(money_reservationID == null) {
		::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
		return false;
	}
	
	//if there is a train in route clone the vehicle
	if(this._vehicle_list.Count() > 0) {
		locomotive = AIVehicle.CloneVehicle(this._depot_pro, this._vehicle_list.Begin(), true);
		if(!AIVehicle.IsValidVehicle(locomotive)){
			AILog.Warning("cloning locomotive is not valid " + AIError.GetLastErrorString());
			
			switch(AIError.GetLastError()){
				case AIError.ERR_NOT_ENOUGH_CASH:
					::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);	
					break;
				case AIVehicle.ERR_VEHICLE_TOO_MANY:
					::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_TOO_MANY_VEHICLES);
					break;
				case AIError.ERR_UNKNOWN:
					::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_UNKNOWN);
					break;
			}
			::ai_instance._money_manager.ReleaseReservation(money_reservationID);
			return false;
		}
	} else { //else build new train
		locomotive = AIVehicle.BuildVehicle(this._depot_pro, locomotive_engine);
		if(!AIVehicle.IsValidVehicle(locomotive)) {
			AILog.Warning("build locomotive is not valid");
			switch(AIError.GetLastError()){
				case AIError.ERR_NOT_ENOUGH_CASH:
					 ::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
					 break;
				case AIVehicle.ERR_VEHICLE_TOO_MANY:
					 ::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_TOO_MANY_VEHICLES);
					 break;
				case AIError.ERR_UNKNOWN:
					 ::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_UNKNOWN);
					 break;
			}
			::ai_instance._money_manager.ReleaseReservation(money_reservationID);
			return false;
		}
		AIVehicle.RefitVehicle(locomotive, this._cargoID);
		for(local i = 0 ; i < number_of_wagons ; i++){
			//if we have passangers cargo, we also add one extra mail wagon
			if(wagon_engine_mail == null || i < number_of_wagons-1) wagon = AIVehicle.BuildVehicle(this._depot_pro , wagon_engine);
			else wagon = AIVehicle.BuildVehicle(this._depot_pro , wagon_engine_mail);
			if(!AIVehicle.IsValidVehicle(wagon)){
				AILog.Warning("wagon is not valid " + i);
				AIVehicle.SellVehicle(locomotive);
				::ai_instance._money_manager.ReleaseReservation(money_reservationID);
				switch(AIError.GetLastError()){
					case AIError.ERR_NOT_ENOUGH_CASH:
						 ::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
						 break;
					case AIVehicle.ERR_VEHICLE_TOO_MANY:
						 ::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_TOO_MANY_VEHICLES);
						 break;
					case AIError.ERR_UNKNOWN:
						 ::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_UNKNOWN);
						 break;
				}
				return false;
			}
			AIVehicle.RefitVehicle(wagon , this._cargoID);
			if(!AIVehicle.MoveWagon(wagon , 0 , locomotive , 0)){
				AILog.Warning("moving not suceeded" + i);
				AIVehicle.SellVehicle(locomotive);
				AIVehicle.SellVehicle(wagon);
				::ai_instance._money_manager.ReleaseReservation(money_reservationID);
				::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_LOCOMOTIVE_CANNOT_PULL_WAGON);
				return false;
			}
		}
		
		//set orders for train
		if(this._is_town_route) AIOrder.AppendOrder(locomotive, AIStation.GetLocation(this._station_producing._stationID), AIOrder.OF_NONE);
		else AIOrder.AppendOrder(locomotive, AIStation.GetLocation(this._station_producing._stationID), AIOrder.OF_FULL_LOAD_ANY);
		
		if(this._depot_acc != null) AIOrder.AppendOrder(locomotive, this._depot_acc, AIOrder.OF_SERVICE_IF_NEEDED);
		
		if(this._is_town_route) AIOrder.AppendOrder(locomotive, AIStation.GetLocation(this._station_accepting._stationID),  AIOrder.OF_NONE );
		else AIOrder.AppendOrder(locomotive, AIStation.GetLocation(this._station_accepting._stationID), AIOrder.OF_UNLOAD | AIOrder.OF_NO_LOAD);
		
		if(this._depot_pro != null)AIOrder.AppendOrder(locomotive, this._depot_pro, AIOrder.OF_SERVICE_IF_NEEDED);
		
	}		
	
	AIVehicle.StartStopVehicle(locomotive);
	//sometimes happened that train had not been added to list (no idea why) so instead every time we add train we refresh the list
	//station producing because there is only one producing station per route
	this._vehicle_list = AIVehicleList_Station(this._station_producing._stationID);
	::ai_instance._money_manager.ReleaseReservation(money_reservationID);
	::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NONE);
	return true;
}

function RailRoute::CalculateNumberOfWagons(locomotive_engine , wagon_engine)
{
	local locomotive_length = null , wagon_length = null;
	local cost;
	local money_reservationID = 0;
	local platform_length = (this._station_producing._platform_length < this._station_accepting._platform_length) ?
							this._station_producing._platform_length : this._station_accepting._platform_length;
	
	cost = AIEngine.GetPrice(locomotive_engine) + AIEngine.GetPrice(wagon_engine);
	//AILog.Info("RailRoute::CalculateNumberOfWagons cost: " + cost);
	money_reservationID = ::ai_instance._money_manager.ReserveMoney(cost , (1.25 * cost).tointeger());

	if(money_reservationID == null){
		::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
		AILog.Warning("RESERVE RailRoute::CalculateNumberOfWagons " + ::ai_instance._error_manager.GetLastErrorString());
		return null;
	}
	//build locomotive
	local locomotive = AIVehicle.BuildVehicle(this._depot_pro , locomotive_engine);
	if(!AIVehicle.IsValidVehicle(locomotive)){
		::ai_instance._money_manager.ReleaseReservation(money_reservationID);
		AILog.Info("CalculateNumberOfWagons locomotive: " + AIError.GetLastErrorString())
		switch(AIError.GetLastError()){
			case AIError.ERR_NOT_ENOUGH_CASH:
				 ::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
				 break;
			case AIVehicle.ERR_VEHICLE_TOO_MANY:
				 ::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_TOO_MANY_VEHICLES);
				 break;
			case AIError.ERR_UNKNOWN:
				 ::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_UNKNOWN);
				 break;
		}
		//AILog.Warning("RailRoute::CalculateNumberOfWagons locomotive: " + AIError.GetLastErrorString());
		return null;
	}
	locomotive_length = AIVehicle.GetLength(locomotive);
	AIVehicle.SellVehicle(locomotive);
	//build wagon we want to use
	local wagon = AIVehicle.BuildVehicle(this._depot_pro, wagon_engine);
	if(!AIVehicle.IsValidVehicle(wagon)){
		AILog.Info("CalculateNumberOfWagons wagons: " + AIError.GetLastErrorString())
		::ai_instance._money_manager.ReleaseReservation(money_reservationID);
		switch(AIError.GetLastError()){
			case AIError.ERR_NOT_ENOUGH_CASH:
				 ::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NOT_ENOUGH_CASH);
				 break;
			case AIVehicle.ERR_VEHICLE_TOO_MANY:
				 ::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_TOO_MANY_VEHICLES);
				 break;
			case AIError.ERR_UNKNOWN:
				 ::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_UNKNOWN);
				 break;
			default:
				 break;
		}
		//AILog.Warning("RailRoute::CalculateNumberOfWagons wagon: " + ::ai_instance._error_manager.GetLastErrorString());

		return null;
	}
	wagon_length = AIVehicle.GetLength(wagon);
	AIVehicle.SellVehicle(wagon);
	::ai_instance._money_manager.ReleaseReservation(money_reservationID);
	::ai_instance._error_manager.SetLastError(ErrorManager.ERROR_NONE);
	//Calculate the number of wagons.
	local number_of_wagons = (platform_length * 16 - locomotive_length) / wagon_length;
	//for towns we rather want more smaller trains (because of AIOrder.OF_NONE)
	if(this._is_town_route) number_of_wagons = (number_of_wagons*2/3).tointeger(); 
	return  number_of_wagons;
}

function RailRoute::ChooseWagon(cargo, railtype)
{
	local enginelist = AIEngineList(AIVehicle.VT_RAIL);
	enginelist.Valuate(AIEngine.IsValidEngine);
	enginelist.KeepValue(1);
	enginelist.Valuate(AIEngine.IsBuildable);
	enginelist.KeepValue(1);
	enginelist.Valuate(AIEngine.IsWagon);
	enginelist.KeepValue(1);
		
	if(railtype != null) {
		enginelist.Valuate(AIEngine.CanRunOnRail, railtype);
		enginelist.KeepValue(1);
	}
		
	enginelist.Valuate(AIEngine.CanRefitCargo, cargo);
	enginelist.KeepValue(1);
	enginelist.Valuate(AIEngine.GetCapacity);
	enginelist.Sort(AIList.SORT_BY_VALUE , false);
	//AILog.Info("Wagon: " + AIEngine.GetName(enginelist.Begin()));
	return enginelist.Count() == 0 ? null : enginelist.Begin();
		
}
	
function RailRoute::ChooseLocomotive(cargo, railtype, reliable) 
{
	local enginelist = AIEngineList(AIVehicle.VT_RAIL);
	enginelist.Valuate(AIEngine.IsValidEngine);
	enginelist.KeepValue(1);
	enginelist.Valuate(AIEngine.IsBuildable);
	enginelist.KeepValue(1);
	enginelist.Valuate(AIEngine.IsWagon);
	enginelist.KeepValue(0);
		
	if(railtype != null) {
		enginelist.Valuate(AIEngine.CanRunOnRail, railtype);
		enginelist.KeepValue(1);
	}
		
	if(cargo != null){
		enginelist.Valuate(AIEngine.CanPullCargo , cargo);
		enginelist.KeepValue(1);
	}
	if(reliable) {
		enginelist.Valuate(AIEngine.GetReliability);
		enginelist.KeepAboveValue(60);
	}
	enginelist.Valuate(AIEngine.GetMaxSpeed);
	enginelist.Sort(AIList.SORT_BY_VALUE, false);
		
	/*
	AILog.Info("Locomotive: " + AIEngine.GetName(enginelist.Begin()) + 
	           " MaxSpeed: " + AIEngine.GetMaxSpeed(enginelist.Begin()) + 
			   " Reliability: " + AIEngine.GetReliability(enginelist.Begin()));
	*/
	return enginelist.Count() == 0 ? null : enginelist.Begin();
		
}

function RailRoute::AdjustNumberOfTrainsInRoute()
{
	local vehicle_list_copy = AIList();
	vehicle_list_copy.AddList(this._vehicle_list);
	
	//filter trains newer than 2 years
	vehicle_list_copy.Valuate(AIVehicle.GetAge);
	vehicle_list_copy.KeepAboveValue(365*3);
	vehicle_list_copy.Valuate(AIVehicle.GetProfitLastYear);
	vehicle_list_copy.KeepBelowValue(GeneralManager.BAD_YEARLY_PROFIT);
	local cargo_waiting = AIStation.GetCargoWaiting(this._station_producing._stationID, this._cargoID);
	
	//for town compute average passangers waiting in both stations
	if(this._is_town_route) cargo_waiting = ((cargo_waiting + AIStation.GetCargoWaiting(this._station_accepting._stationID, this._cargoID))/2).tointeger(); 
	if(vehicle_list_copy.Count() != 0) {
		for(local vehicle = vehicle_list_copy.Begin(); !vehicle_list_copy.IsEnd(); vehicle = vehicle_list_copy.Next()) {
			::ai_instance._general_manager.SendToDepotForSelling(vehicle, 1);
		}		
	} else {
		if(this._is_town_route) {
			if(cargo_waiting < MINIMAL_PASS_AMOUNT_WAITING_IN_STATION) {
				//AILog.Info("No need to add more trains");
			} else {
				this.AddNumberOfTrains(1);
			}
		} else {
		//if there is no cargo stockpiled in station, there is no need to add new train
			if(cargo_waiting < MINIMAL_CARGO_AMOUNT_WAITING_IN_STATION) {
				//AILog.Info("No need to add more trains");
			} else if(AIIndustry.GetLastMonthTransportedPercentage(this._station_producing._industryID, this._cargoID) < INDUSTRY_PRODUCTION_PERCENTAGE_LOW) {
				this.AddNumberOfTrains(2);
			} else if(AIIndustry.GetLastMonthTransportedPercentage(this._station_producing._industryID, this._cargoID) < INDUSTRY_PRODUCTION_PERCENTAGE_HIGH) {
				this.AddNumberOfTrains(1);
			}// else AILog.Info("No need to add more trains");
		}
	}
}


function RailRoute::Save()
{
	local vl = ExtendedList();
	vl.AddList(this._vehicle_list);
	
	//AILog.Info(this.ToString());
	local data = {
		routeid = this._route_id,
		cargoid = this._cargoID,
		vehiclelist = vl.toarray(),
		depotpro = this._depot_pro,
		depotacc = this._depot_acc,
		railtype = this._railtype,
		istownroute = this._is_town_route,
		routeok = this._routeOK
	}
	data.rawset("stationproducing", this._station_producing.Save());
	data.rawset("stationaccepting", this._station_accepting.Save());
	//AILog.Info("SAVEDATA: " + "S_PRO: " + AIStation.GetName(data.stationproducing.stationid) + " S_ACC: " + AIStation.GetName(data.stationaccepting.stationid) +
	 //      " Cargo: " + AICargo.GetCargoLabel(data.cargoid) + " IsTownRoute: " + data.istownroute + " VehicleList count: " + data.vehiclelist.len() + " routeOK: " + data.routeok);
	//AILog.Info("railroute save data len (should be 9): " + data.len());
	return data;
}

function RailRoute::Load(data)
{

	
	local route_id = data["routeid"];
	local depot_pro = data["depotpro"];
	local depot_acc = data["depotacc"];
	local railtype = data["railtype"];
	local is_town_route = data["istownroute"];
	local cargoID = data["cargoid"];
	local station_accepting;
	local station_producing;
	if(is_town_route) {
		station_accepting = TownStationInfo.Load(data.rawget("stationaccepting"));
		station_producing = TownStationInfo.Load(data.rawget("stationproducing"));
	} else {
		station_accepting = IndustryStationInfo.Load(data.rawget("stationaccepting"));
		station_producing = IndustryStationInfo.Load(data.rawget("stationproducing"));
	}
	
	local route = RailRoute(route_id, station_producing, station_accepting, cargoID, depot_pro, depot_acc, railtype, is_town_route);

	route._routeOK = data["routeok"];
	
	local vl = ExtendedList();
	vl.AddFromArray(data.vehiclelist)
	route._vehicle_list.AddList(vl); 
	
	return route;
}