/**
 * @author Daniela Plachtova and others 
 * @file valuators.nut
 * 
 * @note this is collection of all valuators used in this AI
 */
 
 
 /**
  * @brief class CanBuildRailStationWithValueValuator valuates tiles based on 1. if you are able to build stations_around
  *																		      2. tile value
  */																		  
class CanBuildRailStationWithValueValuator
{
	
	_base_tile = null; 				/* base tile of industry/town */
	_platform_length = null;		/* platform length */
	_number_of_tracks = null;		/* number of tracks */
	_direction = null;				/* direction where base industry is pointing to */		
	_cargo = null;					/* cargo id */
	_accept = null;					/* is accepting industry or town */
	_radius = null;					/* coverage radius of train station */
	
	/**
	 * @brief constructor for CanBuildRailStationWithValueValuator
	 * @param base_tile base tile of industry/town
	 * @param platform_length platform length
	 * @param number_of_tracks number of tracks
	 * @param direction direction where base industry is pointing to
	 * @param cargo cargo id
	 * @param accept true if industry accepts or false if it produces (for town it is always true)
	 * @param radius coverage radius of train station
	 */
	constructor(base_tile, platform_length, number_of_tracks, direction, cargo, accept, radius) 
	{
		_base_tile = base_tile;
		_platform_length = platform_length;
		_number_of_tracks = number_of_tracks;	
		_direction = direction;
		_cargo = cargo;
		_accept = accept;
		_radius = radius;
	}
	
	/**
	 * @brief Valuates  - 1. station direction priority (how many directions can we build station on tile)
	 *					  2. if accept - prefer better cargo acceptance tiles)
							 if produces - prefer closer tiles
	 * @param tile tile to valuate 
	 * @param self CanBuildRailStationWithValueValuator
	 * @return value
	 */
	function Valuate(tile, self) 
	{
		this = self;
		
		//AILog.Info(_platform_length + " " + _number_of_tracks + " " + Direction.ToString(this._direction));
		local north_ne_sw = RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHEAST, tile, _platform_length, _number_of_tracks);
		local north_nw_se = RailBuilder.IsExitDirectionOfStationBuildable(Direction.NORTHWEST, tile, _platform_length, _number_of_tracks);
		local south_ne_sw = RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHEAST, tile, _platform_length, _number_of_tracks);
		local south_nw_se = RailBuilder.IsExitDirectionOfStationBuildable(Direction.SOUTHWEST, tile, _platform_length, _number_of_tracks);
		local counter = 0;
		
		//check if we can build station in specific direction
		//straight direction has bigger value
		if(this._direction == Direction.NORTH) {
			if(north_ne_sw) counter++;
			if(north_nw_se) counter++;
		} else if(this._direction == Direction.SOUTH) {
			if(south_ne_sw) counter++;
			if(south_nw_se) counter++;
		} else if(this._direction == Direction.WEST) {
			if(south_ne_sw) counter++;
			if(north_nw_se) counter++;
		} else if(this._direction == Direction.EAST) {
			if(north_ne_sw) counter++;
			if(south_nw_se) counter++;
		} else if(this._direction == Direction.NORTHEAST ) {
			if(north_ne_sw) counter += 3;
			if(north_nw_se) counter++;
			if(south_nw_se) counter++;
		} else if(this._direction == Direction.NORTHWEST) {
			if(north_nw_se) counter += 3;
			if(north_ne_sw) counter++;
			if(south_ne_sw) counter++;
		} else if(this._direction == Direction.SOUTHEAST) {
			if(south_nw_se) counter += 3;
			if(south_ne_sw) counter++;
			if(north_ne_sw) counter++;
		} else if(this._direction == Direction.SOUTHWEST) {
			if(south_ne_sw) counter += 3;
			if(south_nw_se) counter++;
			if(north_nw_se) counter++;
		}
		
		local value = 0;
		local distance = AITile.GetDistanceManhattanToTile(_base_tile, tile);
		//we multiply cargo acceptances in both station directions with tiles closer to industry/town 
		if(this._accept) {
			local ne_sw = AITile.GetCargoAcceptance(tile, _cargo, _platform_length, _number_of_tracks, _radius);
			local nw_se = AITile.GetCargoAcceptance(tile, _cargo, _number_of_tracks, _platform_length, _radius);
			value = ne_sw * nw_se * ( _radius * 4 - distance);
			//AILog.Info("ACC:  ne_sw " + ne_sw + " nw_se " + nw_se);
		} else {
			//we do this so closer tiles has bigger value
			value = _radius * 4 - distance;
		}
		
		//we multiply direction posibilities with value
		return (counter == 0) ? 0 : counter * value;
	}
}

/**
 * @brief Check if tile is flat and buildable
 * @param tile tile to check
 * @return true if tile is flat and buildable else false
 */
function IsTileFlatAndBuildableValuator(tile)
{
	return (AITile.IsBuildable(tile) && AITile.SLOPE_FLAT == AITile.GetSlope(tile));
}

/**
 * @brief Valuates producing industry ((lastMonthProduction * (1 - LastMonthTransporedPercentage/100))/(stations_around + 1) * CargoValue 
 * @param industry to value
 * @param cargo cargo id 
 * @return value
 * @note algorithm was taken from trAIns valuating of producing industries
 */
function ProducingIndustryValuator(industry, cargo)
{
		local cargos = GeneralManager.GetOrdedCargoList();
		local stations_around;
		local v = 1;
		//AILog.Info("Industry: " + AIIndustry.GetName(industry) + " Cargo: " + AICargo.GetCargoLabel(cargo)); 
	    v = AIIndustry.GetLastMonthProduction(industry , cargo).tofloat();
		v *= 1.0 - (AIIndustry.GetLastMonthTransportedPercentage(industry, cargo).tofloat() / 100.0);
		stations_around = AIIndustry.GetAmountOfStationsAround(industry);
		v /= (stations_around.tofloat() + 1.0);
		v *= cargos.GetValue(cargo).tofloat();
		return (v * 10000).tointeger();	
}

/**
 * @brief Gets Distance difference of two industries/towns
 * @param base base tile 
 * @param other other tile
 * @param tolerance tolerance of distance difference
 * @param is_town true if we valuate town else false for industry
 * @return distance difference or -1 for distance differences out of tolerance range
 */
function GetDistanceDiffToleranceValuator(base, other, tolerance, is_town)
{	
	local distance;
	if(is_town) distance = AITile.GetDistanceManhattanToTile(AITown.GetLocation(base), AITown.GetLocation(other));
	else distance = AITile.GetDistanceManhattanToTile(AIIndustry.GetLocation(base), AIIndustry.GetLocation(other));
	local isAccepted = (distance <= ::ai_instance._rail_manager.optimal_railroad_route_length + tolerance && 
						::ai_instance._rail_manager.optimal_railroad_route_length - tolerance <= distance);
	if(!isAccepted) return -1;
	else {
	local value = abs(distance - ::ai_instance._rail_manager.optimal_railroad_route_length);
	//AILog.Info("ACC: " + AIIndustry.GetName(industry) + " PRO: " + AIIndustry.GetName(other_industry) + " Distance: " + distance + " Value: " + value);
	return value;
	} 
	
}

/**
 * @brief Get Cost Factor for engine
 * @param engine engine to valuate
 * @param costfactor_list engine usefullness list
 * @return value of engine
 * @author Wormnest (WormAI)
 */
function GetCostFactorValuator(engine, costfactor_list) 
{
	// For some reason we can't access this.engine_usefullness from inside the Valuate function,
	// thus we add that as a parameter
	//AILog.Info("usefullness list count: " + costfactor_list.Count());
	if (costfactor_list == null) {
		return 0;
	}
	else {
		return costfactor_list.GetValue(engine);
		//return AIEngine.GetCapacity(engine);
	}
}
