class Transport
{
    constructor()
    {
    }
    function MinimumMoneyRequired();         
    function BuildRoute(routePlan);          
    function DepotsIds();
    function EstimateGain(from, to, cargo);
    function EstimateDays(from, to, cargo);  
    function EstimateNVehicles(from, to, cargo);
    function BuildVehicleFor(depot,cargo, from, to);
    function ScheduleRoute(vehicle, stationFrom, stationTo );
    function TransportType();
    function DefaultStationFlags(vehicle);
    function VehiclesLimit();
    function VehiclesCount();
    function MinimumDistance();
    function MaximumDistance();
    /* Try to enlarge the station for the given transport 
     * Returns true if station was expanded, false otherwise
     */
    function ExpandStation(stationId);
    /*
     * Return the number of days that a vehicle is expected to stay in a station for a full
     * unload/load of cargo.
     */
    function DaysAtStation();
    function Id();
    function VehicleType();
    function TransportType();
    function TransportSubType();
    function _tostring();

    /*
     * Return the list of tiles where a station of the given kind
     * can be build for in the given city
     */
    function _FindPossibleTownStationLocations(town, stationType);
    function _FindPossibleIndustryStationLocations(industry, stationType);
    function _BuildStations(routePlan, path);
    _logger = null;
};

function Transport::CanBuildVehicleFor(depot,cargoId, from, to)
{
    local test = AITestMode();
    local engine = GetEngineFor(cargoId, from, to);
    if(!engine)
        return false;
    local result = AIVehicle.BuildVehicle(depot, engine._id)==0;
    test = null;
    return result;

}

function Transport::EstimateGain(from, to, cargo)
{
    // Monthly gain is given by the number of load each moths times the capacity of the vehicle
    // times the income for each quantity, Minus the maintenance cost of the vehicle times
    
    // What we don't take into account yet
    // Structures (roads,stations) maintenance cost. Would make little productions less desiderable
    local fromTile = from.Location();
    local toTile = to.Location();

    local distance = EstimateDistance(from.Location(), to.Location() );
    if (distance > MaximumDistance() || distance < MinimumDistance() )
        return -100000;

    local days = EstimateDays(from.Location(), to.Location(), cargo); 
    if(!days)
    {
        return -100000;
    }
    local engine = GetEngineFor(cargo, null, null);
    
    local gain = AICargo.GetCargoIncome(cargo, distance.tointeger(), days.tointeger());
    local capacity = engine.GetCapacity(cargo);
    local monthlyGain = capacity * gain * 30.0 / days;

    local nVehicles = EstimateNVehicles(from, to, cargo);
    local expenses = engine.GetRunningCost() * 30.0 / 365; // vehicle cost
    return monthlyGain - expenses;


}


function Transport::EstimateDays(tileFrom, tileTo, cargo)
{
    local distance = EstimateDistance(tileFrom, tileTo );
    local distanceInKm = distance * KMISH_PER_TILE;
    local engine = GetEngineFor(cargo, null, null);
    if(!engine)
        return;
    local speed = engine.GetMaxSpeed();
    local travelTimeInDays = distanceInKm  / speed / 24;
    return travelTimeInDays + 2*DaysAtStation(); // add a few days for load/unload
}


function Transport::BuildRoute(routePlan)
{
    _logger.debug("Trying to build route plan", routePlan);
    local from;
    local to;

    if (routePlan.fromTown) 
    {
	from = _FindPossibleTownStationLocations(routePlan.from, routePlan.cargo, true);
    }
    else 
    {
	from = _FindPossibleIndustryStationLocations(routePlan.from, routePlan.cargo, true);
    }
    if (from.len() == 0) 
    {
        _logger.debug("No place for building a station at the origin");
	return null;
    }


    if (routePlan.toTown) 
    {
	to = _FindPossibleTownStationLocations(routePlan.to, routePlan.cargo, false);
    }
    else 
    {
	to = _FindPossibleIndustryStationLocations(routePlan.to, routePlan.cargo, false);
    }
    if (to.len() == 0)
    {
        _logger.debug("No places for building a station at the destination");
	return null;
    }
    
    _logger.info("Looking for a path for", routePlan);
    local path = FindPath(from, to);
    if (!path)
    {
        _logger.trace("Unable to find path");
        return null;
    }
    local stations = _BuildStations(routePlan, path)
    if (stations == null)
    {
	_logger.trace("Cannot build stations for path");
	return null;
    }
    local stationFrom = stations[0];
    local stationTo = stations[1];
    if (!BuildPath(path))
    {
        _logger.trace("Unable to build path");
        stationFrom.DestroyIfTemporary();
        stationTo.DestroyIfTemporary();
        return null;
    }
    local depot = FindOrBuildDepot(stationFrom);
    if (!depot) 
    {
	_logger.debug("Cannot build depot at origin", location(stationFrom.Location()),", Will build one at destination");
	depot = FindOrBuildDepot(stationTo);
    }
    if (depot == null) 
    {
        stationFrom.DestroyIfTemporary();
        stationTo.DestroyIfTemporary();
	return null;
    }
    stationFrom.temporary = false;
    stationTo.temporary = false;
    assert( stationFrom != null);
    assert( stationTo != null  );
    assert( depot !=null );
    assert( routePlan.from != null );
    assert( routePlan.to !=null );
    return Route( stationFrom, stationTo, depot,
                  routePlan.from, routePlan.to, routePlan.fromTown, routePlan.toTown, this);
}


function Transport::EstimateNVehicles(fromIndustry, toIndustry, cargo)
{
    local engine = GetEngineFor(cargo, null, null);
    local from = fromIndustry;
    local to = toIndustry;
    if(!engine)
        return 0;
    local dest = to.Location();
    // double time, we have to return :)
    local days = 2*EstimateDays(from.Location(), dest, cargo);
    local production = from.EstimateProduction(cargo);
    local capacity = engine.GetCapacity(cargo);
    local loadsPerMonth = 0.7* production / capacity; // we hardly go above 70%. btw value is still wrong
    local nVehicles = loadsPerMonth * days / 30; // Yes, because february also has 30 days :P
    
    return nVehicles + 1;
}


function Transport::BuildVehicleFor(depot,cargo, from, to)
{
    check("Invalid cargo passed to BuildVehicleFor", AICargo.IsValidCargo, cargo );
    check("Invalid depot passed to BuildVehicleFor", isValidDepot, depot, TransportType());
    local engine = GetEngineFor(cargo, from, to);
    if (!engine) 
    {
        _logger.warning("Cannot get engine for", cargo);
        return null;
    }
    local result =  AIVehicle.BuildVehicle(depot, engine._id);
    if (! AIVehicle.IsValidVehicle(result))
    {
	if (AIError.GetLastError() != AIError.ERR_NOT_ENOUGH_CASH )
        {
	        _logger.error("Unable to build vehicle " + engine.GetName(),"at depot", ::location(depot), ":", AIError.GetLastErrorString() );
        }
        return null;
    }
    if (! AIVehicle.RefitVehicle(result,cargo))
    {      
        _logger.error("Unable to refit vehicle ", engine.GetName(), " to ",  AICargo.GetName(cargo),  " : ", AIError.GetLastErrorString() );
	AIVehicle.SellVehicle(result);
	return null
    }
    return result;
}

function Transport::VehiclesLimit() 
{
    return 50000;
}

function Transport::ExpandStation(station)
{
    return false;
}


function Transport::_tostring() 
{
    return GetDescription();
}
	    

require("trucks.nut");
require("air.nut");
require("tram.nut");

enum TransportId { road, air, tram };
// This must match the order given by the above enum
ALL_TRANSPORTS <- [ RoadTransport(), AirTransport(), TramTransport() ];
TRANSPORT_ENABLED <- [true, true, false];

function isTransportEnabled(id)
{
    return TRANSPORT_ENABLED[id];
}

function enabledTransports()
{
    return filterKeys(ALL_TRANSPORTS,isTransportEnabled);
}


