/*
 * This file is part of PAXLink, which is an AI for OpenTTD
 * Copyright (C) 2008, 2009  Leif Linse
 *
 * PAXLink is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * PAXLink is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PAXLink; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

class Helper
{
	static function SetSign(tile, message);
	static function BreakPoint(sign_tile);
	static function HasSign(text);
	static function ClearAllSigns();
	static function MyClassValuate(list, valuator, valuator_class, ...);
	static function GetStationsInTown(town_id);
	static function GetBusOnlyStationsInTown(town_id);
	static function ListValueSum(ai_list);
	static function CopyListSwapValuesAndItems(old_list);
	static function GetPAXCargo();
	static function GetMaxHeightOfTile(tile);
	static function Min(x1, x2);
	static function Max(x1, x2);
	static function Clamp(x, min, max);
}

function Helper::SetSign(tile, message)
{
	if(AIController.GetSetting("debug_signs") != 1)
		return;

	local found = false;
	for(local i = 0; i < AISign.GetMaxSignID(); ++i)
	{
		if(AISign.IsValidSign(i) && AISign.GetLocation(i) == tile)
		{
			if(found)
				AISign.RemoveSign(i);
			else
			{
				AISign.SetName(i, message);
				found = true;
			}
		}
	}

	if(!found)
		AISign.BuildSign(tile, message);
}

// Places a sign on tile sign_tile and waits until the sign gets removed
function Helper::BreakPoint(sign_tile)
{
	if(AIController.GetSetting("debug_signs") != 1)
		return;

	if(Helper.HasSign("no_break"))
		return;

	AILog.Warning("Break point reached. -> Remove the \"break\" sign to continue.");
	local sign = AISign.BuildSign(sign_tile, "break");
	while(AISign.IsValidSign(sign)) { AIController.Sleep(1); }
}

function Helper::HasSign(text)
{
	for(local i = 0; i < AISign.GetMaxSignID(); ++i)
	{
		if(AISign.IsValidSign(i) && AISign.GetName(i) == text)
		{
			return true;
		}
	}
	return false;
}
function Helper::ClearAllSigns()
{
	local i = 0;
	for(i = 0; i < AISign.GetMaxSignID(); ++i)
	{
		if(i != null && AISign.IsValidSign(i))
		{
			AISign.RemoveSign(i);
		}
	}
}

function Helper::MyClassValuate(list, valuator, valuator_class, ...)
{
   assert(typeof(list) == "instance");
   assert(typeof(valuator) == "function");
   
   local args = [valuator_class, null];
   
   for(local c = 0; c < vargc; c++) {
      args.append(vargv[c]);
   }

   foreach(item, _ in list) {
      args[1] = item;
      local value = valuator.acall(args);
      if (typeof(value) == "bool") {
         value = value ? 1 : 0;
      } else if (typeof(value) != "integer") {
         throw("Invalid return type from valuator");
      }
      list.SetValue(item, value);
   }
}

function Helper::GetStationsInTown(town_id)
{
	local stations = AIStationList(AIStation.STATION_ANY);
	stations.Valuate(AIStation.GetNearestTown);
	stations.KeepValue(town_id);

	return stations;
}

function Helper::GetBusOnlyStationsInTown(town_id)
{
	local stations = AIStationList(AIStation.STATION_BUS_STOP);
	stations.Valuate(AIStation.GetNearestTown);
	stations.KeepValue(town_id);

	// remove stations that has other things than bus stops also
	stations.Valuate(AIStation.HasStationType, AIStation.STATION_TRAIN);
	stations.KeepValue(0);
	stations.Valuate(AIStation.HasStationType, AIStation.STATION_TRUCK_STOP);
	stations.KeepValue(0);
	stations.Valuate(AIStation.HasStationType, AIStation.STATION_AIRPORT);
	stations.KeepValue(0);
	stations.Valuate(AIStation.HasStationType, AIStation.STATION_DOCK);
	stations.KeepValue(0);

	return stations;
}

function Helper::ListValueSum(ai_list)
{
	local sum = 0;
	local item = ai_list.Begin();
	while(ai_list.HasNext())
	{
		sum += ai_list.GetValue(item);

		item = ai_list.Next();
	}

	return sum;
}

function Helper::CopyListSwapValuesAndItems(old_list)
{
	local new_list = AIList();
	for(local i = old_list.Begin(); old_list.HasNext(); i = old_list.Next())
	{
		local value = old_list.GetValue(i);
		new_list.AddItem(value, i);
	}

	return new_list;
}

function Helper::GetPAXCargo()
{
	local cargo_list = AICargoList();
	cargo_list.Valuate(AICargo.HasCargoClass, AICargo.CC_PASSENGERS);
	cargo_list.KeepValue(1);
	cargo_list.Valuate(AICargo.GetTownEffect);
	cargo_list.KeepValue(AICargo.TE_PASSENGERS);

	if(!AICargo.IsValidCargo(cargo_list.Begin()))
	{
		AILog.Error("PAX Cargo do not exist");
	}

	//AILog.Info("PAX Cargo is: " + AICargo.GetCargoLabel(cargo_list.Begin()));

	return cargo_list.Begin();
}

// Function written by Yexo: (Yexo called it GetRealHeight(tile) however)
function Helper::GetMaxHeightOfTile(tile)
{
	local height = AITile.GetHeight(tile);
	local slope = AITile.GetSlope(tile);
	if (AITile.IsSteepSlope(slope)) {
		switch (slope) {
			case AITile.SLOPE_STEEP_N: return height;
			case AITile.SLOPE_STEEP_E: return height + 1;
			case AITile.SLOPE_STEEP_W: return height + 1;
			case AITile.SLOPE_STEEP_S: return height + 2;
		}
	}
	if (slope & AITile.SLOPE_N) height--;
	if (slope != AITile.SLOPE_FLAT) height++;
	return height;
}

function Helper::Min(x1, x2)
{
	return x1 < x2? x1 : x2;
}

function Helper::Max(x1, x2)
{
	return x1 > x2? x1 : x2;
}

function Helper::Clamp(x, min, max)
{
	x = Max(x, min);
	x = Min(x, max);
	return x;
}
