/*
 * This file is part of PAXLink, which is an AI for OpenTTD
 * Copyright (C) 2008, 2009  Leif Linse
 *
 * PAXLink is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * PAXLink is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PAXLink; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

class Helper
{
	// Check the API for presence of bugs
	static function CheckOpenTTDAPI();

	static function SetSign(tile, message, force_build_sign = false);
	static function BreakPoint(sign_tile);
	static function HasSign(text);
	static function ClearAllSigns();
	static function GetStationsInTown(town_id);
	static function GetPAXCargo();
	static function GetMaxHeightOfTile(tile);

	//static function MyClassValuate(list, valuator, valuator_class, ...);
	static function DummyValuator(a) { return a; }
	static function ListValueSum(ai_list);
	static function CopyListSwapValuesAndItems(old_list);
	static function GetListMinValue(ai_list);
	static function GetListMaxValue(ai_list);
	static function SquirrelListToAIList(squirrel_list);

	static function Min(x1, x2);
	static function Max(x1, x2);
	static function Clamp(x, min, max);

	static function GetTileRelative(relative_to_tile, delta_x, delta_y);
}

function Helper::CheckOpenTTDAPI()
{
	local ap_types = [AIAirport.AT_SMALL, 
			AIAirport.AT_LARGE,
			AIAirport.AT_METROPOLITAN,
			AIAirport.AT_INTERNATIONAL,
			AIAirport.AT_COMMUTER,
			AIAirport.AT_INTERCON,
			AIAirport.AT_HELIPORT,
			AIAirport.AT_HELISTATION,
			AIAirport.AT_HELIDEPOT];

	foreach(ap_type in ap_types)
	{
		if(!AIAirport.IsValidAirportType(ap_type))
		{
			// PAXLink requires to being able to get the width of airports that are not available for construction at the moment.
			// This is needed because it needs to be able to get the geometry of airports that it has constructed but are no longer available for construction.
			if(AIAirport.GetAirportWidth(ap_type) == -1)
				return false;
		}
	}

	return true;
}

function Helper::SetSign(tile, message, force_build_sign = false)
{
	if(!force_build_sign && AIController.GetSetting("debug_signs") != 1)
		return;

	local found = false;
	local sign_list = AISignList();
	for(local i = sign_list.Begin(); sign_list.HasNext(); i = sign_list.Next())
	{
		if(AISign.GetLocation(i) == tile)
		{
			if(found)
				AISign.RemoveSign(i);
			else
			{
				if(message == "")
					AISign.RemoveSign(i);
				else
					AISign.SetName(i, message);
				found = true;
			}
		}
	}

	if(!found)
		AISign.BuildSign(tile, message);
}

// Places a sign on tile sign_tile and waits until the sign gets removed
function Helper::BreakPoint(sign_tile)
{
	if(Helper.HasSign("no_break"))
		return;

	if(!Helper.HasSign("break_on"))
	{
		if(AIController.GetSetting("debug_signs") != 1)
			return;

	}

	AILog.Warning("Break point reached. -> Remove the \"break\" sign to continue.");
	local sign = AISign.BuildSign(sign_tile, "break");
	while(AISign.IsValidSign(sign)) { AIController.Sleep(1); }
}

function Helper::HasSign(text)
{
	local sign_list = AISignList();
	for(local i = sign_list.Begin(); sign_list.HasNext(); i = sign_list.Next())
	{
		if(AISign.GetName(i) == text)
		{
			return true;
		}
	}
	return false;
}
function Helper::ClearAllSigns()
{
	local sign_list = AISignList();
	for(local i = sign_list.Begin(); sign_list.HasNext(); i = sign_list.Next())
	{
		AISign.RemoveSign(i);
	}
}

/*function Helper::MyClassValuate(list, valuator, valuator_class, ...)
{
   assert(typeof(list) == "instance");
   assert(typeof(valuator) == "function");
   
   local args = [valuator_class, null];
   
   for(local c = 0; c < vargc; c++) {
      args.append(vargv[c]);
   }

   foreach(item, _ in list) {
      args[1] = item;
      local value = valuator.acall(args);
      if (typeof(value) == "bool") {
         value = value ? 1 : 0;
      } else if (typeof(value) != "integer") {
         throw("Invalid return type from valuator");
      }
      list.SetValue(item, value);
   }
}*/

function Helper::GetStationsInTown(town_id)
{
	local stations = AIStationList(AIStation.STATION_ANY);
	stations.Valuate(AIStation.GetNearestTown);
	stations.KeepValue(town_id);

	return stations;
}

function Helper::ListValueSum(ai_list)
{
	local sum = 0;
	local item = ai_list.Begin();
	while(ai_list.HasNext())
	{
		sum += ai_list.GetValue(item);

		item = ai_list.Next();
	}

	return sum;
}

function Helper::CopyListSwapValuesAndItems(old_list)
{
	local new_list = AIList();
	for(local i = old_list.Begin(); old_list.HasNext(); i = old_list.Next())
	{
		local value = old_list.GetValue(i);
		new_list.AddItem(value, i);
	}

	return new_list;
}

function Helper::GetListMinValue(ai_list)
{
	ai_list.Sort(AIAbstractList.SORT_BY_VALUE, true); // highest last
	return ai_list.GetValue(ai_list.Begin());
}

function Helper::GetListMaxValue(ai_list)
{
	ai_list.Sort(AIAbstractList.SORT_BY_VALUE, false); // highest first
	return ai_list.GetValue(ai_list.Begin());
}

function Helper::SquirrelListToAIList(squirrel_list)
{
	local ai_list = AIList();
	foreach(item in squirrel_list)
	{
		ai_list.AddItem(item, 0);
	}

	return ai_list;
}

function Helper::GetPAXCargo()
{
	local cargo_list = AICargoList();
	cargo_list.Valuate(AICargo.HasCargoClass, AICargo.CC_PASSENGERS);
	cargo_list.KeepValue(1);
	cargo_list.Valuate(AICargo.GetTownEffect);
	cargo_list.KeepValue(AICargo.TE_PASSENGERS);

	if(!AICargo.IsValidCargo(cargo_list.Begin()))
	{
		AILog.Error("PAX Cargo do not exist");
	}

	//AILog.Info("PAX Cargo is: " + AICargo.GetCargoLabel(cargo_list.Begin()));

	return cargo_list.Begin();
}

// Function written by Yexo: (Yexo called it GetRealHeight(tile) however)
function Helper::GetMaxHeightOfTile(tile)
{
	return AITile.GetMaxHeight(tile);
}

function Helper::Min(x1, x2)
{
	return x1 < x2? x1 : x2;
}

function Helper::Max(x1, x2)
{
	return x1 > x2? x1 : x2;
}

function Helper::Clamp(x, min, max)
{
	x = Max(x, min);
	x = Min(x, max);
	return x;
}

function Helper::GetTileRelative(relative_to_tile, delta_x, delta_y)
{
	local tile_x = AIMap.GetTileX(relative_to_tile);
	local tile_y = AIMap.GetTileY(relative_to_tile);

	local new_x = tile_x + delta_x;
	local new_y = tile_y + delta_y;

	return AIMap.GetTileIndex(new_x, new_y);
}

