/*
 * This file is part of PAXLink, which is an AI for OpenTTD
 * Copyright (C) 2008, 2009  Leif Linse
 *
 * PAXLink is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * PAXLink is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with PAXLink; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

require("globals.nut");

require("builder.nut");
require("roadbuilder.nut");
require("station.nut");
require("vehicle.nut");
require("engine.nut");
require("orders.nut");

require("helper.nut");
require("valuator.nut");

require("townmanager.nut");
require("pairoptimizer.nut");
require("airmanager.nut");

class PAXLink extends AIController {

	loaded_from_save = false;

	builder = null;

	last_serviced_towns_check = null;
	last_depot_check = null;
	last_look_for_new_towns = null;
	last_inter_city_check = null;
	last_town_stats_management = null;
	town_stats_management_counter = null;
	
	each_year_counter = null;

	constructor()
	{
		loaded_from_save = false;

		local bigger_than_max_intervall = 10000;
		main_instance = this; 
		this.builder = Builder();
		this.last_serviced_towns_check = -bigger_than_max_intervall;
		this.last_depot_check =          -bigger_than_max_intervall;
		this.last_look_for_new_towns =   -bigger_than_max_intervall;
		this.last_inter_city_check =     -bigger_than_max_intervall;
		this.last_town_stats_management = -bigger_than_max_intervall;
		this.town_stats_management_counter = 0;
		this.each_year_counter = -1;
	}

	function Start();
	function Stop()
	{
		Helper.ClearAllSigns();
	}

	function Save();
	function Load(version, data);

	function EachYear();
}

function PAXLink::SetCompanyName()
{
	local i = 2;
	if(!AICompany.SetName("PAXLink"))
	while(!AICompany.SetName("PAXLink #" + i))
	{
		i = i + 1;
		if(i > 255)
			break;
	}
	AICompany.SetPresidentName("PAX-King");
}

//// INTRA CITY TRANSPORT

function NumBusOnlyStationsInTownValuator(town_id)
{
	local stations = Station.GetAllFeederStationsOfTown(town_id);
	
	return stations.Count();
}

function PAXLink::ManageIntraCityTransport()
{
	// look through cities that currently are serviced first
	local serviced_towns = AITownList();
	serviced_towns.Valuate(NumBusOnlyStationsInTownValuator);
	serviced_towns.KeepAboveValue(0);

	//AILog.Info("Has stations in " + serviced_towns.Count() + " towns.");

	// Look for vehicles in depot every 50 tick
	if(this.last_depot_check + 50 < this.GetTick())
	{
		this.last_depot_check = this.GetTick();
		Vehicle.ScanForStopedVehiclesInDepots();
	}

	// Only check serviced town at a certain interval
	if(this.last_serviced_towns_check + 500 < this.GetTick())
	{
		this.last_serviced_towns_check = this.GetTick();

		AILog.Info("#### Checking the already serviced towns ####");

		Vehicle.SellOldVehicles();
		Vehicle.SellVehiclesWithNoOrders();

		// Start with town with biggest POP and walk down, for easier debugging
		serviced_towns.Valuate(AITown.GetPopulation);
		serviced_towns.Sort(AIAbstractList.SORT_BY_VALUE, false); // highest value first


		local bus_engine = this.builder.GetEngine(1000, AIVehicle.VT_ROAD);
		local aircraft_engine = this.builder.GetEngine(1000, AIVehicle.VT_AIR);

		local town_id = serviced_towns.Begin();
		for(; serviced_towns.HasNext(); town_id = serviced_towns.Next())
		{
			AILog.Error("Maintaining local transport in town " + AITown.GetName(town_id)); // print as error to make it easier to catch

			local town_mgr = this.towns.GetTownManager(town_id);

			local ic_stations = Station.GetAllInterCityStations();
			local aircrafts = Station.GetAllAirVehiclesOfStationList(ic_stations);
			local buses = Vehicle.GetAllBuses();
			if(aircrafts.Count() < 2 && ic_stations.Count() < 3 && buses.Count() > 5)
			{
				AILog.Info("Has 0 to 1 aircrafts and at least 2 airports and more than 5 buses, so don't buy more buses until we get 2 aircrafts.");
			}
			else
			{
				this.builder.BuyBusesForTown(town_id);
			}

			// Require that we have money for 2 new buses for constructing new road stations and/or airport
			// but do still call ConstructTownStations since it may remove unused stations in the town.
			local money_needed_for_new_construction = AIEngine.GetPrice(bus_engine) * 2;
			this.builder.ConstructTownStations(town_mgr, money_needed_for_new_construction);

			local money = AICompany.GetBankBalance(AICompany.COMPANY_SELF);
			if(!AITown.HasStatue(town_id) && 
					money > AIEngine.GetPrice(aircraft_engine) * 2)
				AITown.PerformTownAction(town_id, AITown.TOWN_ACTION_BUILD_STATUE);
		}
	}

	// check if there are any non-serviced towns
	// ...
	if(this.last_look_for_new_towns + 500 < this.GetTick())
	{
		AILog.Info("#### Decide if PAXLink should expand to another town ####");

		local airports = Station.GetAllInterCityStations();
		local expand = true;
		if(airports.Count() > 1)
		{
			airports.Valuate(AIStation.GetCargoWaiting, Helper.GetPAXCargo());
			local tot_waiting = Helper.ListValueSum(airports);
			local mean_waiting = 0;
			if(airports.Count() > 0)
				mean_waiting = tot_waiting / airports.Count();
			airports.Sort(AIAbstractList.SORT_BY_VALUE, false); // highest first
			local max_waiting = airports.GetValue(airports.Begin());

			if(max_waiting >= 300 + airports.Count() * 20 || mean_waiting >= 250 + airports.Count() * 20)
			{
				AILog.Info("To many passengers is waiting on existing IC airports, so don't expand further until those has been served good enough.");
				AILog.Info("max waiting: " + max_waiting + ", mean waiting: " + mean_waiting);
				expand = false;
			}

			// Also check inner-city stations
			local other_stations = AIStationList(AIStation.STATION_ANY);
			other_stations.RemoveList(airports);

			other_stations.Valuate(AIStation.GetCargoWaiting, Helper.GetPAXCargo());
			local tot_waiting = Helper.ListValueSum(other_stations);
			local mean_waiting = 0;
			if(other_stations.Count() > 0)
				mean_waiting = tot_waiting / other_stations.Count();
			other_stations.Sort(AIAbstractList.SORT_BY_VALUE, false); // highest first
			local max_waiting = other_stations.GetValue(other_stations.Begin());
			
			if(max_waiting >= 300 + airports.Count() * 50 || mean_waiting >= 150 + airports.Count() * 10)
			{
				AILog.Info("To many passengers is waiting to get on the bus, possible to IC stations, so wait with further expansions");
				AILog.Info("max waiting: " + max_waiting + ", mean waiting: " + mean_waiting);
				expand = false;
			}
		}

		if(expand)
		{

			// Then look through cities that doesn't yet has any service
			// - But only if there is enough money or we don't have any serviced town yet and must try anyways.
			local bus_engine = this.builder.GetEngine(1000, AIVehicle.VT_ROAD);
			local money_needed = AIEngine.GetPrice(bus_engine) * 5;
			if(serviced_towns.Count() == 0 || AICompany.GetBankBalance(AICompany.COMPANY_SELF) > money_needed);
			{
				AILog.Info("#### Looking for new towns to service ####");

				AILog.Info("Make a list of towns that are interesting to connect");

				// Start by going through all towns and add them to town_list if they are unconnected. 
				local town_list = AIList();
				foreach(town_mgr in towns.list)
				{
					if(town_mgr.station_list.len() == 0)
					{
						local population = AITown.GetPopulation(town_mgr.town_id);
						if(serviced_towns.Count() > 0 && population < 450)
							continue;

						// For each town compute a score based on their population and the expandability of transport services in that town
						town_mgr.GetExpandability();
						local score = population * town_mgr.expandability;
						town_list.AddItem(town_mgr.town_id, score);
					}
				}
				// Sort the town list with highest score first
				town_list.Sort(AIAbstractList.SORT_BY_VALUE, false); // highest value first


				if(serviced_towns.Count() != 0)
					town_list.KeepAboveValue(450);
			
				for(local town_id = town_list.Begin(); town_list.HasNext(); town_id = town_list.Next())
				{
					local town_mgr = towns.GetTownManager(town_id);

					this.Sleep(1);
					if(!town_mgr.CanAirportBeAdded())
					{
						AILog.Info("Skip town " + AITown.GetName(town_mgr.town_id) + " since it is not possible to build an airport there.");
						continue;
					}

					if(this.builder.ConstructTownStations(town_mgr, money_needed))
					{
						this.last_look_for_new_towns = this.GetTick() - 400; // only wait 100 ticks (instead of 500) for next look for new town if found a town
						return;
					}

					AILog.Info("take next town");
				}
			}
		}

		this.last_look_for_new_towns = this.GetTick();
	}
}

function PrintItemsValuator(item, prefix, postfix)
{
	AILog.Info(prefix + item + postfix);
}

// Between cities
function PAXLink::ManageInterCityTransport()
{
	if(this.last_inter_city_check + 50 < this.GetTick())
	{
		this.last_inter_city_check = this.GetTick();

		local bus_engine = this.builder.GetEngine(1000, AIVehicle.VT_ROAD);

		local serviced_towns = AITownList();
		serviced_towns.Valuate(NumBusOnlyStationsInTownValuator);
		serviced_towns.KeepAboveValue(0);

		if(serviced_towns.Count() >= 2)
		{
			AILog.Info("#### Managing inter-city transport ####");

			local towns_with_inter_city = AITownList();
			towns_with_inter_city.Valuate(AITown.GetPopulation);
			towns_with_inter_city.Sort(AIAbstractList.SORT_BY_VALUE, false); // highest pop first

			//this.builder.FindInterCityStation(towns_with_inter_city.Begin());
			//Helper.MyClassValuate(towns_with_inter_city, this.builder.FindInterCityStation, this.builder);
			Station.FindInterCityStation_EachInList(towns_with_inter_city);

			towns_with_inter_city.RemoveValue(-1);

			local towns_to_connect = AITownList();
			towns_to_connect.Valuate(NumBusOnlyStationsInTownValuator);
			towns_to_connect.KeepAboveValue(1);
			towns_to_connect.Valuate(AITown.GetPopulation);
			towns_to_connect.KeepAboveValue(700);

			if(towns_with_inter_city.Count() > 1)
			{
				local stations_need_connection = AIList();

				// First go through existing inter-city stations and check if any of them has no service
				for(local town_id = towns_with_inter_city.Begin(); towns_with_inter_city.HasNext(); town_id = towns_with_inter_city.Next())
				{
					local inter_city_station_id = towns_with_inter_city.GetValue(town_id);
					local vehicles = AIVehicleList_Station(inter_city_station_id);
					vehicles.Valuate(AIVehicle.GetVehicleType);
					vehicles.KeepValue(AIVehicle.VT_AIR);

					if(vehicles.IsEmpty())
					{
						AILog.Info("Add station " + inter_city_station_id + " to list of stations that need a connection");
						stations_need_connection.AddItem(inter_city_station_id, 0);
					}
					else
					{
						local vehicle_id = vehicles.Begin();
						local connection_stations = Order.GetStationListFromOrders(vehicles.Begin());
						AILog.Info("Manage vehicles of existing connection");
						this.builder.ManageInterCityVehiclesOfInterCityConnection(connection_stations);

						towns_with_inter_city.RemoveList(connection_stations); // TODO: Check if this is safe
					}
				}

				// as towns_with_inter_city was sorted with highest pop first
				// then stations_need_connection should be sorted this way too
				while(stations_need_connection.Count() > 1)
				{
					local station_1 = stations_need_connection.Begin();
					stations_need_connection.RemoveTop(1);
					
					stations_need_connection.Valuate(AIMap.DistanceSquare, station_1);
					stations_need_connection.Sort(AIAbstractList.SORT_BY_VALUE, false); // as far a way as possible

					local station_2 = stations_need_connection.Begin();
					stations_need_connection.RemoveTop(1);

					local new_connection = AIList();
					new_connection.AddItem(station_1, 0);
					new_connection.AddItem(station_2, 0);
					AILog.Info("Manage vehicles of new connection");
					this.builder.ManageInterCityVehiclesOfInterCityConnection(new_connection);
					AILog.Info("Manage vehicles of new connection done");

					// sort with highest pop first for next round
					if(stations_need_connection.Count() > 1)
					{
						stations_need_connection.Valuate(AITown.GetPopulation);
						stations_need_connection.Sort(AIAbstractList.SORT_BY_VALUE, false); // highest pop first
					}
				}
			}
			else
			{
				AILog.Info("Not enough towns has inter-city stations");
			}
		}
	}
}

function PAXLink::Start()
{
	this.Sleep(1);
	if(!this.loaded_from_save)
	{
		this.SetCompanyName();
	}

	AILog.Info("Checking the API for some known old bugs that affects PAXLink ...");
	if(!Helper.CheckOpenTTDAPI())
	{
		AILog.Info(" ");
		AILog.Error("PAXLink is not compatible with your OpenTTD");
		AILog.Error("version!");
		AILog.Info(" ");
		AILog.Info("PAXLink has tested the API for functionality");
		AILog.Info("it needs, and it has found out that your");
		AILog.Info("OpenTTD version does not support some of the");
		AILog.Info("functionalities that PAXLink requires.");
		AILog.Info(" ");
		AILog.Info("Please upgrade both PAXLink and OpenTTD to");
		AILog.Info("the last available versions and see if that");
		AILog.Info("solves the problem.");
		AILog.Info(" ");

		if(AIController.GetSetting("quit_on_api_check_failure") == 1)
		{
			AILog.Info("Quiting...");
			return false;
		}
	}

	AIRoad.SetCurrentRoadType(AIRoad.ROADTYPE_ROAD);

	towns = TownList();

	while (true) {
		if(this.last_town_stats_management + 40 < this.GetTick())
		{
			this.last_town_stats_management = this.GetTick();
			this.town_stats_management_counter++;
			towns.UpdateTownStatistics(this.town_stats_management_counter);
		}

		// Only keep a loan if we are not bilionaries
		if(AICompany.GetBankBalance(AICompany.COMPANY_SELF) < AICompany.GetMaxLoanAmount() * 2)
		{
			AICompany.SetLoanAmount(AICompany.GetMaxLoanAmount());
		}
		else
		{
			AICompany.SetLoanAmount(0);
		}

		this.ManageIntraCityTransport();
		this.ManageInterCityTransport();

		// Check if it is a new year
		local current_year = AIDate.GetYear(AIDate.GetCurrentDate());
		if(current_year > this.each_year_counter)
		{
			this.each_year_counter = current_year;
			this.EachYear();
		}

		if(AIController.GetSetting("clear_signs") == 1)
		{
			Helper.ClearAllSigns();
		}

		this.Sleep(1);
	}
}

function PAXLink::Save()
{
	local table = {};
	return table;
}

function PAXLink::Load(version, data)
{
	this.loaded_from_save = true;
	AILog.Info("Previously saved with AI version " + version);
}

function PAXLink::EachYear()
{

	foreach(town_mgr in this.towns.list)
	{
		// Make sure that all bus stops and depots are connected within each town
		Builder.ConnectAllBusStationsAndDepotsByRoad(town_mgr);
	}

	this.Sleep(1);
	if(AIController.GetSetting("use_ap_pair_optimizer") == 1)
	{
		AirManager.OptimizeAirportPairs();
	}
}

