
class Vehicle
{
	static function GetAllBuses();

	static function SendVehicleToDepotForSelling(vehicle_id);
	static function SellVehiclesWithNoOrders();
	static function SellOldVehicles();
	static function ScanForStopedVehiclesInDepots();

	static function GetAirVehiclePlaneType(vehicle_id);
};

function Vehicle::GetAllBuses()
{
	local buses = AIVehicleList(); 
	buses.Valuate(AIVehicle.GetVehicleType);
	buses.KeepValue(AIVehicle.VT_ROAD);
	return buses;
}

function Vehicle::SendVehicleToDepotForSelling(vehicle_id)
{
	local current_order_dest = Order.GetCurrentOrderDestination(vehicle_id);
	if(AIOrder.GetOrderCount(vehicle_id) == 0 && (AIRoad.IsRoadDepotTile(current_order_dest) || AIAirport.IsHangarTile(current_order_dest)))
	{
		// Don't try to sell vehicles that heads to a depot without any orders 
		return false;
	}

	main_instance.Sleep(5);
	local start_tick = main_instance.GetTick();
	while(!AIVehicle.SendVehicleToDepot(vehicle_id))
	{
		if(start_tick + 100 < main_instance.GetTick())
		{
			AILog.Info("Failed to send vehicle " + AIVehicle.GetName(vehicle_id) + " to depot, for more than 100 ticks, aborting selling of this vehicle.");
			return false;
		}

		AILog.Info("Failed to send vehicle " + AIVehicle.GetName(vehicle_id) + " to depot, retrying soon .. | start_tick = " + start_tick + ", curr tick = " + main_instance.GetTick());
		main_instance.Sleep(5);
	}

	AIOrder.UnshareOrders(vehicle_id);
	Order.ClearOrdersOfSharedGroup(vehicle_id);

	return true;
}

function Vehicle::SellVehiclesWithNoOrders()
{
	AILog.Info("Looking for lost vehicles to sell");
	local vehicle_list = AIVehicleList();

	AILog.Info("has " + vehicle_list.Count() + " vehicles");

	vehicle_list.Valuate(AIOrder.GetOrderCount);
	vehicle_list.KeepValue(0);

	AILog.Info("has " + vehicle_list.Count() + " vehicles without orders");

	for(local vehicle_id = vehicle_list.Begin(); vehicle_list.HasNext(); vehicle_id = vehicle_list.Next()) 
	{
		AILog.Info("Sending vehicle without orders to depot for selling");
		Vehicle.SendVehicleToDepotForSelling(vehicle_id);
	}
}

function Vehicle::SellOldVehicles()
{
	AILog.Info("Looking for old vehicles to sell");
	local vehicle_list = AIVehicleList();

	AILog.Info("has " + vehicle_list.Count() + " vehicles");

	vehicle_list.Valuate(AIVehicle.GetAgeLeft);
	vehicle_list.KeepBelowValue(365*2);

	AILog.Info("has " + vehicle_list.Count() + " old vehicles");

	for(local vehicle_id = vehicle_list.Begin(); vehicle_list.HasNext(); vehicle_id = vehicle_list.Next()) 
	{
		AILog.Info("Sending old vehicle to depot for selling");
		Vehicle.SendVehicleToDepotForSelling(vehicle_id);
	}
}

function Vehicle::ScanForStopedVehiclesInDepots()
{
	local vehicle_list = AIVehicleList();
	vehicle_list.Valuate(AIVehicle.IsStoppedInDepot);
	vehicle_list.KeepValue(1);

	for(local vehicle_id = vehicle_list.Begin(); vehicle_list.HasNext(); vehicle_id = vehicle_list.Next()) 
	{
		// vehicles without orders in depot is a signal that they should be sold
		if(AIOrder.GetOrderCount(vehicle_id) == 0)
			AIVehicle.SellVehicle(vehicle_id);
	}
}

function Vehicle::GetAirVehiclePlaneType(vehicle_id)
{
	local engine_id = AIVehicle.GetEngineType(vehicle_id);
	if(AIEngine.IsValidEngine(engine_id))
	{
		return AIEngine.GetPlaneType(engine_id);
	}
	else
	{
		// invalid engine
		AILog.Error("Trying to determine the plane type of an aircraft failed because the engine is no longer valid!");
		return AIAirport.PT_INVALID;
	}
}
