// Alias of AIMap.GetTileIndex(), this looks cleaner
function dXY(x, y) {
	return AIMap.GetTileIndex(x,y);
}

function getPerp(delta) {
	if (delta == 1)
		return dXY(0,1)
	if (delta == -1)
		return dXY(0,-1)
	if (delta > 0) {
		return 1
	} else {
		return -1
	}
}

function add(tile, delta) {
	return tile + AIMap.GetTileIndex(delta[0], delta[1]);
}

function LowerTileWith(tile, slope) {
	foreach (corner in [AITile.SLOPE_W, AITile.SLOPE_N, AITile.SLOPE_S, AITile.SLOPE_E]) {
		if (slope & corner)
			AITile.LowerTile(tile, corner)
	}
}
function RaiseTileWith(tile, slope) {
	foreach (corner in [AITile.SLOPE_W, AITile.SLOPE_N, AITile.SLOPE_S, AITile.SLOPE_E]) {
		if (slope & corner)
			AITile.RaiseTile(tile, corner)
	}
}

/**
 * Add a rectangular area to an AITileList containing tiles that are within /radius/
 * tiles from the center tile, taking the edges of the map into account.
 */  
function SafeAddRectangle(list, tile, radius) {
	local x1 = max(1, AIMap.GetTileX(tile) - radius);
	local y1 = max(1, AIMap.GetTileY(tile) - radius);
	
	local x2 = min(AIMap.GetMapSizeX() - 2, AIMap.GetTileX(tile) + radius);
	local y2 = min(AIMap.GetMapSizeY() - 2, AIMap.GetTileY(tile) + radius);
	
	list.AddRectangle(AIMap.GetTileIndex(x1, y1),AIMap.GetTileIndex(x2, y2)); 
}

//Find the cargo ID for a given class of Cargo
function GetCargoID(cargoClass) {
	local list = AICargoList();
	local candidate = -1;
	for (local i = list.Begin(); list.HasNext(); i = list.Next()) {
		if (AICargo.HasCargoClass(i, cargoClass))
		candidate = i;
	}
	
	if(candidate != -1)
		return candidate;
	
	throw "missing required cargo class";
}
function Range(from, to) {
	local range = [];
	for (local i=from; i<to; i++) {
		range.append(i);
	}
	
	return range;
}

function BuildTunnel(vehicleType, from, to, direction, length, testmode = false) {
	LogTile("Building tunnel from", from, "tunnel")
	LogTile("Building tunnel to", to, "tunnel")
	local delta = (to-from)/length
	if (AITile.IsBuildable(from+delta) && AITile.IsBuildable(to-delta)) {
		if (testmode) {
			return true										
		} else {
			// Let's try to build a tunnel
			if (direction == Direction.SW) {AITile.LowerTile(from, AITile.SLOPE_N); AITile.LowerTile(from, AITile.SLOPE_E); AITile.LowerTile(to, AITile.SLOPE_S); AITile.LowerTile(to, AITile.SLOPE_W);}
			if (direction == Direction.SE) {AITile.LowerTile(from, AITile.SLOPE_N); AITile.LowerTile(from, AITile.SLOPE_W); AITile.LowerTile(to, AITile.SLOPE_S); AITile.LowerTile(to, AITile.SLOPE_E);}
			if (direction == Direction.NE) {AITile.LowerTile(from, AITile.SLOPE_S); AITile.LowerTile(from, AITile.SLOPE_W); AITile.LowerTile(to, AITile.SLOPE_N); AITile.LowerTile(to, AITile.SLOPE_E);}
			if (direction == Direction.NW) {AITile.LowerTile(from, AITile.SLOPE_S); AITile.LowerTile(from, AITile.SLOPE_E); AITile.LowerTile(to, AITile.SLOPE_N); AITile.LowerTile(to, AITile.SLOPE_W);}
			if (AITunnel.GetOtherTunnelEnd(from+delta) == to-delta) {
				if (AITunnel.BuildTunnel(vehicleType, from+delta) || IsAlreadyBuilt()) {
					if (vehicleType == AIVehicle.VT_ROAD) {
						if ((AIRoad.BuildRoad(from, from+delta) || IsAlreadyBuilt()) && (AIRoad.BuildRoad(to-delta, to) || IsAlreadyBuilt()))
							return true
					} else if (vehicleType == AIVehicle.VT_RAIL) {
						if ((AIRail.BuildRail(from-delta, from, from+delta) || IsAlreadyBuilt()) && (AIRail.BuildRail(to-delta, to, to+delta) || IsAlreadyBuilt()))
							return true
					}
				}
			}
		}
	}
	return false
}


function VehicleInTheWay() {
	local e = AIError.GetLastError()
	return (e == AIError.ERR_VEHICLE_IN_THE_WAY)
}
function NotEnoughCash() {
	local e = AIError.GetLastError()
	return (e == AIError.ERR_NOT_ENOUGH_CASH) // || e == AIError.ERR_UNKNOWN)
}
function IsAlreadyBuilt() {
	local e = AIError.GetLastError()
	return (e == AIError.ERR_ALREADY_BUILT) // || e == AIError.ERR_UNKNOWN)
}
function VehicleInTheWay() {
	local e = AIError.GetLastError()
	return (e == AIError.ERR_VEHICLE_IN_THE_WAY) // || e == AIError.ERR_UNKNOWN)
}



//Function to get an indication whether two tiles are connected by land
function IsConnectedByLand(from, to) {
	if (AITile.GetMaxHeight(from) == 0 || AITile.GetMaxHeight(to) == 0) return false //one of both ends is loacted in the sea
	
	local pathfinder = Land();		
	pathfinder.cost.max_cost = 5*pathfinder.cost.water; //5 tiles of water is too much
	pathfinder.InitializePath(from, to);
	
	local path = false;
	LogTiles("Starting land connection check at " + RailwAI.GetTick() + " from ", from, " to ", to)
	path = pathfinder.FindPath(20000);
	if (!path) {
		LogError("Not connected by land. Determined at " + RailwAI.GetTick());
		return false
	}
	LogWarning("Tiles can be connected by land. Determined at " + RailwAI.GetTick())
	return true
}


function IsLandStrokeX(xmin, xmax, y) {
	local water = 0
	for (local x = xmin; x < xmax; x++) {
		if (AITile.IsWaterTile(AIMap.GetTileIndex(x, y)))
			water++
		else
			water /= 2
		if (water > 10)
			return false
	}
	return true
}
function IsLandStrokeY(x, ymin, ymax) {
	local water = 0
	for (local y = ymin; y < ymax; y++) {
		if (AITile.IsWaterTile(AIMap.GetTileIndex(x, y)))
			water++
		else
			water /= 2
		if (water > 10)
			return false
	}
	return true
}

function HasAnyRoad(tile) {
	return AIRoad.HasRoadType(tile, AIRoad.ROADTYPE_ROAD) || AIRoad.HasRoadType(tile, AIRoad.ROADTYPE_TRAM)
}

function CanBuildRoadOn(tile) {
	return AITile.IsBuildable(tile) || AIRoad.HasRoadType(tile, AIRoad.ROADTYPE_ROAD) || AIRoad.HasRoadType(tile, AIRoad.ROADTYPE_TRAM)
}

function ConnectRailTile(tile) {
	local myCompany = AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)
	local railType = AIRail.GetRailType(tile)
	if (railType == AIRail.RAILTYPE_INVALID) railType = AIRail.GetCurrentRailType()
	//AIRail.SetCurrentRailType(railType)
	foreach (delta in [dXY(0,1), dXY(0,-1), dXY(1,0)]) {
		foreach (delta2 in [dXY(0,-1), dXY(1,0), dXY(-1,0)]) {
			if (delta != delta2) {
				local connect = false
				local t1 = tile + delta
				local t2 = tile + delta2
				if (AIRail.GetRailType(t1) == railType && AITile.GetOwner(t1) == myCompany && AIRail.GetRailType(t2) == railType && AITile.GetOwner(t2) == myCompany) {
					connect = true
					if (delta2 != -delta) {//if corner
						local f1 = GetNextRailTile(t1, tile)
						local f2 = GetNextRailTile(t2, tile)
						if (f1 == t1 + delta2 || f2 == t2 + delta)
							connect = false
					}
					//LogTile("Building rail from", t1)
					//LogTile("Building rail to", t2)
				}
				if (connect) {
					AIRail.BuildRail(t1, tile, t2)
				} else {
					AIRail.RemoveRail(t1, tile, t2)
					//LogTile("Removing rail from", t1)
					//LogTile("Removing rail to", t2)
				}
			}
		}
	}
}

function ConnectRailTileTowards(tile, towardsTile) {
	local myCompany = AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)
	local railType = AIRail.GetRailType(tile)
	//LogTile("Connecting rail tile", tile)
	if (railType == AIRail.RAILTYPE_INVALID) railType = AIRail.GetCurrentRailType()
	AIRail.SetCurrentRailType(railType)
	local delta = towardsTile - tile
	foreach (delta2 in [dXY(0,1), dXY(0,-1), dXY(1,0), dXY(-1,0)]) {
		if (delta != delta2) {
			local connect = false
			local t1 = towardsTile
			local t2 = tile + delta2
			if (AIRail.GetRailType(t1) == railType && AIRail.GetRailType(t2) == railType && AITile.GetOwner(t2) == myCompany) {
				if (!AIRail.BuildRail(t1, tile, t2)) {
					if (VehicleInTheWay()) {
						RailwAI.Sleep(50)
						AIRail.BuildRail(t1, tile, t2)
					}
				}
			} else {
				AIRail.RemoveRail(t1, tile, t2)
			}
		}
	}
}

function HasRailSignal(tile) {
	foreach (delta in [dXY(0,1), dXY(0,-1), dXY(1,0), dXY(-1,0)]) {
		if (AIRail.GetSignalType(tile, tile+delta) != AIRail.SIGNALTYPE_NONE) {
			return true
		}
	}
	return false
}

function RemoveAllSignals(tile) {
	//LogTile("Removing all signals", tile)
	foreach (delta in [dXY(0,1), dXY(0,-1), dXY(1,0), dXY(-1,0)]) {
		if (AIRail.GetSignalType(tile, tile+delta) != AIRail.SIGNALTYPE_NONE) {
			if (!AIRail.RemoveSignal(tile, tile+delta)) {
				//we probably don't have enough money anymore
				IncreaseLoan()
				if (!AIRail.RemoveSignal(tile, tile+delta)) {
					LogError("Something went wrong when I tried to remove a railroad signal!")
					return false
				}
			}
		}
	}
	return true
}

function MigrateToPathSignalsAround(tile, radius = 4) {
	local myCompany = AICompany.ResolveCompanyID(AICompany.COMPANY_SELF)
	for (local x = -radius; x <= radius; x++) {
		for (local y = -radius; y <= radius; y++) {
			local checkTile = tile + dXY(x, y)
			if (AITile.GetOwner(checkTile) == myCompany && AIRail.IsRailTile(checkTile)) {
				foreach (delta in [dXY(0,1), dXY(0,-1), dXY(1,0), dXY(-1,0)]) {
					local signalType = AIRail.GetSignalType(checkTile, checkTile+delta)
					if (signalType == AIRail.SIGNALTYPE_NORMAL) {
						AIRail.RemoveSignal(checkTile, checkTile+delta)
						AIRail.BuildSignal(checkTile, checkTile+delta, AIRail.SIGNALTYPE_PBS)
					}
				}
			}
		}
	}
}

function RemoveStraightRail(from, tile, to) {
	local delta = tile - from
	local t = tile
	for (local i = 0; i < 256; i++) {
		if (!AIRail.RemoveRail(t-delta, t, t+delta))
			return false
		if (t+delta == to)
			return true
		t += delta
	}
	throw ("shouldn't remove this long rails")
}

function RebuildStraightRail(from, tile, to) {
	local delta = tile - from
	local t = tile
	local ret = true
	for (local i = 0; i < 256; i++) {
		if (!AIRail.BuildRail(t-delta, t, t+delta))
			ret = false
		if (t+delta == to)
			return ret
		t += delta
	}
	throw ("shouldn't rebuild this long rails")
}

//check if a tile is part of a station owned by you or another company
function IsStationTile(tile) {
	return IsAnyValidStation(AIStation.GetStationID(tile))
	//return (AIRoad.IsRoadStationTile(tile) || AIRail.IsRailStationTile(tile) || AIMarine.IsDockTile(tile) || AIAirport.IsAirportTile(tile))
}

//check if a station ID seems valid for a station owned by your or another company
function IsAnyValidStation(stationID) {
	return (stationID < 65535)
}

//check if a tile is a rail bridge. Unluckily, this is not as straightforward as IsRailTile && IsBridgeTile,
//because IsRailTile returns false for bridges
function IsRailBridge(tile) {
	return (AIBridge.IsBridgeTile(tile) && IsARailTile(tile))
}
//check if a tile is a rail tunnel.
function IsRailTunnel(tile) {
	return (AITunnel.IsTunnelTile(tile) && IsARailTile(tile))
}
//check if a tile is a rail tunnel or rail bridge.
function IsRailTunnelBridge(tile) {
	return (IsARailTile(tile) && (AITunnel.IsTunnelTile(tile) || AIBridge.IsBridgeTile(tile)))
}

//returns if a tile is any rail tile (a normal rail tile, a rail bridge tile or a rail tunnel tile)
function IsARailTile(tile) {
	return (AIRail.GetRailType(tile) != AIRail.RAILTYPE_INVALID)
}
//returns if a tile is any water tile (sea, canal or buoy)
function IsAWaterTile(tile) {
	return (AITile.IsWaterTile(tile) || AIMarine.IsBuoyTile(tile))
}

function AreRailTilesConnected(from, tile, to) {
	if (AITunnel.IsTunnelTile(tile))
		return (AITunnel.GetOtherTunnelEnd(tile) == from || AITunnel.GetOtherTunnelEnd(tile) == to)
	if (AIBridge.IsBridgeTile(tile))
		return (AIBridge.GetOtherBridgeEnd(tile) == from || AIBridge.GetOtherBridgeEnd(tile) == to)
	return AIRail.AreTilesConnected(from,tile,to)
}

function GetRailSignalDirection(current, prev) {
	local c = current
	local p = prev
	for (local i = 0; i < 16; i++) {
		if (AIRail.GetSignalType(c, p) != AIRail.SIGNALTYPE_NONE)
			return 1
		if (AIRail.GetSignalType(p, c) != AIRail.SIGNALTYPE_NONE)
			return -1
		local f = GetNextRailTile(c, p)
		if (!f) return 0
		p = c
		c = f
	}
	return 0
}

//Get the next rail tile. If a piece of rail is a dead end, it will return the tile that could be the next one
function GetNextRailTile(current, prev) {
	local next = GetNextRailTiles(current, prev)
	if (next == false) return false
	if (next.len() > 1) return false
	return next[0]
}

//get next rail tiles
function GetNextRailTiles(current, prev) {
	if (AIBridge.IsBridgeTile(current)) {
		local otherTile = AIBridge.GetOtherBridgeEnd(current)
		local length = AIMap.DistanceManhattan(current, otherTile)
		local delta = (current-otherTile)/length
		if (otherTile == prev) {
			return [current + delta]
		} else {
			if (current + delta == prev)
				return [otherTile]
			else
				return false
		}
	}
	
	if (AITunnel.IsTunnelTile(current)) {
		local otherTile = AITunnel.GetOtherTunnelEnd(current)
		local length = AIMap.DistanceManhattan(current, otherTile)
		local delta = (current-otherTile)/length
		if (otherTile == prev) {
			return [current + delta]
		} else {
			if (current + delta == prev)
				return [otherTile]
			else
				return false
		}	
	}
	local length = AIMap.DistanceManhattan(current, prev)
	local pre = (prev-current)/length
	local nextTiles = []
	local next = false
	foreach (delta in [dXY(0,1), dXY(0,-1), dXY(1,0), dXY(-1,0)]) {
		if (delta != pre && AIRail.AreTilesConnected(current + pre, current, current + delta)) {
			next = current + delta
			nextTiles.append(next)
		}
	}
	if (next == false) return false
	return nextTiles
}